"use strict";
/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path = __importStar(require("path"));
const application_1 = require("@elyra/application");
const apputils_1 = require("@jupyterlab/apputils");
const React = __importStar(require("react"));
const utils_1 = __importDefault(require("./utils"));
exports.RUNTIMES_NAMESPACE = 'runtimes';
exports.KFP_SCHEMA = 'kfp';
exports.RUNTIME_IMAGES_NAMESPACE = 'runtime-images';
exports.RUNTIME_IMAGE_SCHEMA = 'runtime-image';
class PipelineService {
    /**
     * Returns a list of external runtime configurations available as
     * `runtimes metadata`. This is used to submit the pipeline to be
     * executed on these runtimes.
     */
    static getRuntimes(showError = true) {
        return __awaiter(this, void 0, void 0, function* () {
            const runtimes = yield application_1.FrontendServices.getMetadata('runtimes');
            if (showError && Object.keys(runtimes).length === 0) {
                return application_1.FrontendServices.noMetadataError('runtimes');
            }
            return runtimes;
        });
    }
    /**
     * Return a list of configured docker images that are used as runtimes environments
     * to run the pipeline nodes.
     */
    static getRuntimeImages() {
        return __awaiter(this, void 0, void 0, function* () {
            let runtimeImages = yield application_1.FrontendServices.getMetadata('runtime-images');
            runtimeImages = runtimeImages.sort((a, b) => 0 - (a.name > b.name ? -1 : 1));
            if (Object.keys(runtimeImages).length === 0) {
                return application_1.FrontendServices.noMetadataError('runtime-images');
            }
            const images = {};
            for (const image in runtimeImages) {
                const imageName = runtimeImages[image]['metadata']['image_name'];
                images[imageName] = runtimeImages[image]['display_name'];
            }
            return images;
        });
    }
    static getDisplayName(name, metadataArr) {
        return metadataArr.find(r => r['name'] === name)['display_name'];
    }
    /**
     * The runtime name is currently based on the schema name (one schema per runtime)
     * @param name
     * @param metadataArr
     */
    static getRuntimeName(name, metadataArr) {
        return metadataArr.find(r => r['name'] === name)['schema_name'];
    }
    /**
     * Submit the pipeline to be executed on an external runtime (e.g. Kbeflow Pipelines)
     *
     * @param pipeline
     * @param runtimeName
     */
    static submitPipeline(pipeline, runtimeName) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield application_1.RequestHandler.makePostRequest('elyra/pipeline/schedule', JSON.stringify(pipeline), true);
            let dialogTitle;
            let dialogBody;
            if (response['run_url']) {
                // pipeline executed remotely in a runtime of choice
                dialogTitle = 'Job submission to ' + runtimeName + ' succeeded';
                dialogBody = (React.createElement("p", null,
                    "Check the status of your job at",
                    ' ',
                    React.createElement("a", { href: response['run_url'], target: "_blank", rel: "noopener noreferrer" }, "Run Details."),
                    React.createElement("br", null),
                    "The results and outputs are in the ",
                    response['object_storage_path'],
                    ' ',
                    "working directory in",
                    ' ',
                    React.createElement("a", { href: response['object_storage_url'], target: "_blank", rel: "noopener noreferrer" }, "object storage"),
                    "."));
            }
            else {
                // pipeline executed in-place locally
                dialogTitle = 'Job execution succeeded';
                dialogBody = (React.createElement("p", null, "Your job has been executed in-place in your local environment."));
            }
            return apputils_1.showDialog({
                title: dialogTitle,
                body: dialogBody,
                buttons: [apputils_1.Dialog.okButton()]
            });
        });
    }
    /**
     * Export a pipeline to different formats (e.g. DSL, YAML, etc). These formats
     * are understood by a given runtime.
     *
     * @param pipeline
     * @param pipeline_export_format
     * @param pipeline_export_path
     * @param overwrite
     */
    static exportPipeline(pipeline, pipeline_export_format, pipeline_export_path, overwrite) {
        return __awaiter(this, void 0, void 0, function* () {
            console.log('Exporting pipeline to [' + pipeline_export_format + '] format');
            console.log('Overwriting existing file: ' + overwrite);
            const body = {
                pipeline: pipeline,
                export_format: pipeline_export_format,
                export_path: pipeline_export_path,
                overwrite: overwrite
            };
            const response = yield application_1.RequestHandler.makePostRequest('elyra/pipeline/export', JSON.stringify(body), true);
            return apputils_1.showDialog({
                title: 'Pipeline export succeeded',
                body: React.createElement("p", null,
                    "Exported file: ",
                    response['export_path'],
                    " "),
                buttons: [apputils_1.Dialog.okButton()]
            });
        });
    }
    /**
     * Verify if the given pipeline is "current" by looking on it's version, and perform
     * any conversion if needed.
     *
     * @param pipelineDefinition
     */
    static convertPipeline(pipelineDefinition, pipelinePath) {
        let pipelineJSON = JSON.parse(JSON.stringify(pipelineDefinition));
        const currentVersion = utils_1.default.getPipelineVersion(pipelineJSON);
        if (currentVersion < 1) {
            // original pipeline definition without a version
            console.info('Migrating pipeline to version 1.');
            pipelineJSON = this.convertPipelineV0toV1(pipelineJSON);
        }
        if (currentVersion < 2) {
            // adding relative path on the pipeline filenames
            console.info('Migrating pipeline to version 2.');
            pipelineJSON = this.convertPipelineV1toV2(pipelineJSON, pipelinePath);
        }
        if (currentVersion < 3) {
            // Adding python script support
            console.info('Migrating pipeline to version 3 (current version).');
            pipelineJSON = this.convertPipelineV2toV3(pipelineJSON, pipelinePath);
        }
        return pipelineJSON;
    }
    static convertPipelineV0toV1(pipelineJSON) {
        utils_1.default.renamePipelineAppdataField(pipelineJSON.pipelines[0], 'title', 'name');
        utils_1.default.deletePipelineAppdataField(pipelineJSON.pipelines[0], 'export');
        utils_1.default.deletePipelineAppdataField(pipelineJSON.pipelines[0], 'export_format');
        utils_1.default.deletePipelineAppdataField(pipelineJSON.pipelines[0], 'export_path');
        // look into nodes
        for (const nodeKey in pipelineJSON.pipelines[0]['nodes']) {
            const node = pipelineJSON.pipelines[0]['nodes'][nodeKey];
            utils_1.default.renamePipelineAppdataField(node, 'artifact', 'filename');
            utils_1.default.renamePipelineAppdataField(node, 'image', 'runtime_image');
            utils_1.default.renamePipelineAppdataField(node, 'vars', 'env_vars');
            utils_1.default.renamePipelineAppdataField(node, 'file_dependencies', 'dependencies');
            utils_1.default.renamePipelineAppdataField(node, 'recursive_dependencies', 'include_subdirectories');
        }
        pipelineJSON.pipelines[0]['app_data']['version'] = 1;
        return pipelineJSON;
    }
    static convertPipelineV1toV2(pipelineJSON, pipelinePath) {
        pipelineJSON.pipelines[0] = this.setNodePathsRelativeToPipeline(pipelineJSON.pipelines[0], pipelinePath);
        pipelineJSON.pipelines[0]['app_data']['version'] = 2;
        return pipelineJSON;
    }
    static convertPipelineV2toV3(pipelineJSON, pipelinePath) {
        // No-Op this is to disable old versions of Elyra
        // to see a pipeline with Python Script nodes
        pipelineJSON.pipelines[0]['app_data']['version'] = 3;
        return pipelineJSON;
    }
    static getPipelineRelativeNodePath(pipelinePath, nodePath) {
        const relativePath = path.relative(path.dirname(pipelinePath), nodePath);
        return relativePath;
    }
    static getWorkspaceRelativeNodePath(pipelinePath, nodePath) {
        // since resolve returns an "absolute" path we need to strip off the leading '/'
        const workspacePath = path
            .resolve(path.dirname(pipelinePath), nodePath)
            .substring(1);
        return workspacePath;
    }
    static setNodePathsRelativeToPipeline(pipeline, pipelinePath) {
        for (const node of pipeline.nodes) {
            node.app_data.filename = this.getPipelineRelativeNodePath(pipelinePath, node.app_data.filename);
        }
        return pipeline;
    }
    static setNodePathsRelativeToWorkspace(pipeline, pipelinePath) {
        for (const node of pipeline.nodes) {
            node.app_data.filename = this.getWorkspaceRelativeNodePath(pipelinePath, node.app_data.filename);
        }
        return pipeline;
    }
}
exports.PipelineService = PipelineService;
//# sourceMappingURL=PipelineService.js.map