/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import '../style/index.css';
import { MetadataDisplay, MetadataWidget, METADATA_ITEM } from '@elyra/metadata-common';
import { ExpandableComponent, trashIcon, importIcon } from '@elyra/ui-components';
import { Clipboard, Dialog, showDialog } from '@jupyterlab/apputils';
import { CodeCell, MarkdownCell } from '@jupyterlab/cells';
import { CodeEditor } from '@jupyterlab/codeeditor';
import { PathExt } from '@jupyterlab/coreutils';
import { DocumentWidget } from '@jupyterlab/docregistry';
import { FileEditor } from '@jupyterlab/fileeditor';
import { NotebookPanel } from '@jupyterlab/notebook';
import { copyIcon, editIcon } from '@jupyterlab/ui-components';
import { find } from '@lumino/algorithm';
import React from 'react';
import { CodeSnippetService, CODE_SNIPPET_NAMESPACE, CODE_SNIPPET_SCHEMA } from './CodeSnippetService';
const METADATA_EDITOR_ID = 'elyra-metadata-editor';
/**
 * A React Component for code-snippets display list.
 */
class CodeSnippetDisplay extends MetadataDisplay {
    constructor() {
        super(...arguments);
        this.editors = {};
        // Handle code snippet insert into an editor
        this.insertCodeSnippet = async (snippet) => {
            var _a, _b, _c;
            const widget = this.props.getCurrentWidget();
            const snippetStr = snippet.metadata.code.join('\n');
            if (widget instanceof DocumentWidget &&
                widget.content instanceof FileEditor) {
                const documentWidget = widget;
                const fileEditor = documentWidget.content.editor;
                const markdownRegex = /^\.(md|mkdn?|mdown|markdown)$/;
                if (PathExt.extname(widget.context.path).match(markdownRegex) !== null &&
                    snippet.metadata.language.toLowerCase() !== 'markdown') {
                    // Wrap snippet into a code block when inserting it into a markdown file
                    fileEditor.replaceSelection('```' + snippet.metadata.language + '\n' + snippetStr + '\n```');
                }
                else if (widget.constructor.name == 'PythonFileEditor') {
                    this.verifyLanguageAndInsert(snippet, 'python', fileEditor);
                }
                else {
                    fileEditor.replaceSelection(snippetStr);
                }
            }
            else if (widget instanceof NotebookPanel) {
                const notebookWidget = widget;
                const notebookCell = notebookWidget.content.activeCell;
                const notebookCellIndex = notebookWidget.content
                    .activeCellIndex;
                const notebookCellEditor = notebookCell.editor;
                if (notebookCell instanceof CodeCell) {
                    const kernelInfo = await ((_b = (_a = notebookWidget.sessionContext.session) === null || _a === void 0 ? void 0 : _a.kernel) === null || _b === void 0 ? void 0 : _b.info);
                    const kernelLanguage = ((_c = kernelInfo) === null || _c === void 0 ? void 0 : _c.language_info.name) || '';
                    this.verifyLanguageAndInsert(snippet, kernelLanguage, notebookCellEditor);
                }
                else if (notebookCell instanceof MarkdownCell &&
                    snippet.metadata.language.toLowerCase() !== 'markdown') {
                    // Wrap snippet into a code block when inserting it into a markdown cell
                    notebookCellEditor.replaceSelection('```' + snippet.metadata.language + '\n' + snippetStr + '\n```');
                }
                else {
                    notebookCellEditor.replaceSelection(snippetStr);
                }
                const cell = notebookWidget.model.contentFactory.createCodeCell({});
                notebookWidget.model.cells.insert(notebookCellIndex + 1, cell);
            }
            else {
                this.showErrDialog('Code snippet insert failed: Unsupported widget');
            }
        };
        // Handle language compatibility between code snippet and editor
        this.verifyLanguageAndInsert = async (snippet, editorLanguage, editor) => {
            const snippetStr = snippet.metadata.code.join('\n');
            if (editorLanguage &&
                snippet.metadata.language.toLowerCase() !== editorLanguage.toLowerCase()) {
                const result = await this.showWarnDialog(editorLanguage, snippet.display_name);
                if (result.button.accept) {
                    editor.replaceSelection(snippetStr);
                }
            }
            else {
                // Language match or editorLanguage is unavailable
                editor.replaceSelection(snippetStr);
            }
        };
        // Display warning dialog when inserting a code snippet incompatible with editor's language
        this.showWarnDialog = async (editorLanguage, snippetName) => {
            return showDialog({
                title: 'Warning',
                body: `Code snippet "${snippetName}" is incompatible with ${editorLanguage}. Continue?`,
                buttons: [Dialog.cancelButton(), Dialog.okButton()]
            });
        };
        // Display error dialog when inserting a code snippet into unsupported widget (i.e. not an editor)
        this.showErrDialog = (errMsg) => {
            return showDialog({
                title: 'Error',
                body: errMsg,
                buttons: [Dialog.okButton()]
            });
        };
        this.actionButtons = (metadata) => {
            return [
                {
                    title: 'Copy',
                    icon: copyIcon,
                    feedback: 'Copied!',
                    onClick: () => {
                        Clipboard.copyToSystem(metadata.metadata.code.join('\n'));
                    }
                },
                {
                    title: 'Insert',
                    icon: importIcon,
                    onClick: () => {
                        this.insertCodeSnippet(metadata);
                    }
                },
                {
                    title: 'Edit',
                    icon: editIcon,
                    onClick: () => {
                        this.props.openMetadataEditor({
                            onSave: this.props.updateMetadata,
                            namespace: CODE_SNIPPET_NAMESPACE,
                            schema: CODE_SNIPPET_SCHEMA,
                            name: metadata.name
                        });
                    }
                },
                {
                    title: 'Delete',
                    icon: trashIcon,
                    onClick: () => {
                        CodeSnippetService.deleteCodeSnippet(metadata).then((deleted) => {
                            if (deleted) {
                                this.props.updateMetadata();
                                delete this.editors[metadata.name];
                                const editorWidget = find(this.props.shell.widgets('main'), (value, index) => {
                                    return (value.id ==
                                        `${METADATA_EDITOR_ID}:${CODE_SNIPPET_NAMESPACE}:${CODE_SNIPPET_SCHEMA}:${metadata.name}`);
                                });
                                if (editorWidget) {
                                    editorWidget.dispose();
                                }
                            }
                        });
                    }
                }
            ];
        };
        // Render display of a code snippet
        this.renderMetadata = (metadata) => {
            return (React.createElement("div", { key: metadata.name, className: METADATA_ITEM },
                React.createElement(ExpandableComponent, { displayName: this.getDisplayName(metadata), tooltip: metadata.metadata.description, actionButtons: this.actionButtons(metadata), onExpand: () => {
                        this.editors[metadata.name].refresh();
                    } },
                    React.createElement("div", { id: metadata.name }))));
        };
    }
    getDisplayName(metadata) {
        return `[${metadata.metadata.language}] ${metadata.display_name}`;
    }
    sortMetadata() {
        this.props.metadata.sort((a, b) => this.getDisplayName(a).localeCompare(this.getDisplayName(b)));
    }
    componentDidUpdate() {
        const editorFactory = this.props.editorServices.factoryService
            .newInlineEditor;
        const getMimeTypeByLanguage = this.props.editorServices.mimeTypeService
            .getMimeTypeByLanguage;
        this.props.metadata.map((codeSnippet) => {
            if (codeSnippet.name in this.editors) {
                // Make sure code is up to date
                this.editors[codeSnippet.name].model.value.text = codeSnippet.metadata.code.join('\n');
            }
            else {
                // Add new snippets
                this.editors[codeSnippet.name] = editorFactory({
                    config: { readOnly: true },
                    host: document.getElementById(codeSnippet.name),
                    model: new CodeEditor.Model({
                        value: codeSnippet.metadata.code.join('\n'),
                        mimeType: getMimeTypeByLanguage({
                            name: codeSnippet.metadata.language,
                            codemirror_mode: codeSnippet.metadata.language
                        })
                    })
                });
            }
        });
    }
}
/**
 * A widget for Code Snippets.
 */
export class CodeSnippetWidget extends MetadataWidget {
    constructor(props) {
        super(props);
    }
    // Request code snippets from server
    async fetchMetadata() {
        return await CodeSnippetService.findAll();
    }
    renderDisplay(metadata) {
        return (React.createElement(CodeSnippetDisplay, { metadata: metadata, openMetadataEditor: this.openMetadataEditor, updateMetadata: this.updateMetadata, namespace: CODE_SNIPPET_NAMESPACE, schema: CODE_SNIPPET_SCHEMA, getCurrentWidget: this.props.getCurrentWidget, editorServices: this.props.editorServices, shell: this.props.app.shell, sortMetadata: true }));
    }
}
//# sourceMappingURL=CodeSnippetWidget.js.map