"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Arn = exports.ArnFormat = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_fn_1 = require("./cfn-fn");
const token_1 = require("./token");
const util_1 = require("./util");
/**
 * An enum representing the various ARN formats that different services use.
 *
 * @stability stable
 */
var ArnFormat;
(function (ArnFormat) {
    ArnFormat["NO_RESOURCE_NAME"] = "arn:aws:service:region:account:resource";
    ArnFormat["COLON_RESOURCE_NAME"] = "arn:aws:service:region:account:resource:resourceName";
    ArnFormat["SLASH_RESOURCE_NAME"] = "arn:aws:service:region:account:resource/resourceName";
    ArnFormat["SLASH_RESOURCE_SLASH_RESOURCE_NAME"] = "arn:aws:service:region:account:/resource/resourceName";
})(ArnFormat = exports.ArnFormat || (exports.ArnFormat = {}));
/**
 * @stability stable
 */
class Arn {
    constructor() { }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *    arn:{partition}:{service}:{region}:{account}:{resource}{sep}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     *
     * @stability stable
     */
    static format(components, stack) {
        var _b, _c, _d, _e;
        jsiiDeprecationWarnings._aws_cdk_core_ArnComponents(components);
        jsiiDeprecationWarnings._aws_cdk_core_Stack(stack);
        const partition = (_b = components.partition) !== null && _b !== void 0 ? _b : stack === null || stack === void 0 ? void 0 : stack.partition;
        const region = (_c = components.region) !== null && _c !== void 0 ? _c : stack === null || stack === void 0 ? void 0 : stack.region;
        const account = (_d = components.account) !== null && _d !== void 0 ? _d : stack === null || stack === void 0 ? void 0 : stack.account;
        // Catch both 'null' and 'undefined'
        if (partition == null || region == null || account == null) {
            throw new Error(`Arn.format: partition (${partition}), region (${region}), and account (${account}) must all be passed if stack is not passed.`);
        }
        const sep = (_e = components.sep) !== null && _e !== void 0 ? _e : (components.arnFormat === ArnFormat.COLON_RESOURCE_NAME ? ':' : '/');
        const values = [
            'arn', ':', partition, ':', components.service, ':', region, ':', account, ':',
            ...(components.arnFormat === ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME ? ['/'] : []),
            components.resource,
        ];
        if (sep !== '/' && sep !== ':' && sep !== '') {
            throw new Error('resourcePathSep may only be ":", "/" or an empty string');
        }
        if (components.resourceName != null) {
            values.push(sep);
            values.push(components.resourceName);
        }
        return values.join('');
    }
    /**
     * (deprecated) Given an ARN, parses it and returns components.
     *
     * IF THE ARN IS A CONCRETE STRING...
     *
     * ...it will be parsed and validated. The separator (`sep`) will be set to '/'
     * if the 6th component includes a '/', in which case, `resource` will be set
     * to the value before the '/' and `resourceName` will be the rest. In case
     * there is no '/', `resource` will be set to the 6th components and
     * `resourceName` will be set to the rest of the string.
     *
     * IF THE ARN IS A TOKEN...
     *
     * ...it cannot be validated, since we don't have the actual value yet at the
     * time of this function call. You will have to supply `sepIfToken` and
     * whether or not ARNs of the expected format usually have resource names
     * in order to parse it properly. The resulting `ArnComponents` object will
     * contain tokens for the subexpressions of the ARN, not string literals.
     *
     * If the resource name could possibly contain the separator char, the actual
     * resource name cannot be properly parsed. This only occurs if the separator
     * char is '/', and happens for example for S3 object ARNs, IAM Role ARNs,
     * IAM OIDC Provider ARNs, etc. To properly extract the resource name from a
     * Tokenized ARN, you must know the resource type and call
     * `Arn.extractResourceName`.
     *
     * @param arn The ARN to parse.
     * @param sepIfToken The separator used to separate resource from resourceName.
     * @param hasName Whether there is a name component in the ARN at all.
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     * @deprecated use split instead
     */
    static parse(arn, sepIfToken = '/', hasName = true) {
        jsiiDeprecationWarnings.print("@aws-cdk/core.Arn#parse", "use split instead");
        let arnFormat;
        if (!hasName) {
            arnFormat = ArnFormat.NO_RESOURCE_NAME;
        }
        else {
            arnFormat = sepIfToken === '/' ? ArnFormat.SLASH_RESOURCE_NAME : ArnFormat.COLON_RESOURCE_NAME;
        }
        return this.split(arn, arnFormat);
    }
    /**
     * Splits the provided ARN into its components.
     *
     * Works both if 'arn' is a string like 'arn:aws:s3:::bucket',
     * and a Token representing a dynamic CloudFormation expression
     * (in which case the returned components will also be dynamic CloudFormation expressions,
     * encoded as Tokens).
     *
     * @param arn the ARN to split into its components.
     * @param arnFormat the expected format of 'arn' - depends on what format the service 'arn' represents uses.
     * @stability stable
     */
    static split(arn, arnFormat) {
        jsiiDeprecationWarnings._aws_cdk_core_ArnFormat(arnFormat);
        const components = parseArnShape(arn);
        if (components === 'token') {
            return parseTokenArn(arn, arnFormat);
        }
        const [, partition, service, region, account, resourceTypeOrName, ...rest] = components;
        let resource;
        let resourceName;
        let sep;
        let resourcePartStartIndex = 0;
        let detectedArnFormat;
        let slashIndex = resourceTypeOrName.indexOf('/');
        if (slashIndex === 0) {
            // new-style ARNs are of the form 'arn:aws:s4:us-west-1:12345:/resource-type/resource-name'
            slashIndex = resourceTypeOrName.indexOf('/', 1);
            resourcePartStartIndex = 1;
            detectedArnFormat = ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME;
        }
        if (slashIndex !== -1) {
            // the slash is only a separator if ArnFormat is not NO_RESOURCE_NAME
            if (arnFormat === ArnFormat.NO_RESOURCE_NAME) {
                sep = undefined;
                slashIndex = -1;
                detectedArnFormat = ArnFormat.NO_RESOURCE_NAME;
            }
            else {
                sep = '/';
                detectedArnFormat = resourcePartStartIndex === 0
                    ? ArnFormat.SLASH_RESOURCE_NAME
                    // need to repeat this here, as otherwise the compiler thinks 'detectedArnFormat' is not initialized in all paths
                    : ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME;
            }
        }
        else if (rest.length > 0) {
            sep = ':';
            slashIndex = -1;
            detectedArnFormat = ArnFormat.COLON_RESOURCE_NAME;
        }
        else {
            sep = undefined;
            detectedArnFormat = ArnFormat.NO_RESOURCE_NAME;
        }
        if (slashIndex !== -1) {
            resource = resourceTypeOrName.substring(resourcePartStartIndex, slashIndex);
            resourceName = resourceTypeOrName.substring(slashIndex + 1);
        }
        else {
            resource = resourceTypeOrName;
        }
        if (rest.length > 0) {
            if (!resourceName) {
                resourceName = '';
            }
            else {
                resourceName += ':';
            }
            resourceName += rest.join(':');
        }
        // "|| undefined" will cause empty strings to be treated as "undefined".
        // Optional ARN attributes (e.g. region, account) should return as empty string
        // if they are provided as such.
        return util_1.filterUndefined({
            service: service || undefined,
            resource: resource || undefined,
            partition: partition || undefined,
            region,
            account,
            resourceName,
            sep,
            arnFormat: detectedArnFormat,
        });
    }
    /**
     * Extract the full resource name from an ARN.
     *
     * Necessary for resource names (paths) that may contain the separator, like
     * `arn:aws:iam::111111111111:role/path/to/role/name`.
     *
     * Only works if we statically know the expected `resourceType` beforehand, since we're going
     * to use that to split the string on ':<resourceType>/' (and take the right-hand side).
     *
     * We can't extract the 'resourceType' from the ARN at hand, because CloudFormation Expressions
     * only allow literals in the 'separator' argument to `{ Fn::Split }`, and so it can't be
     * `{ Fn::Select: [5, { Fn::Split: [':', ARN] }}`.
     *
     * Only necessary for ARN formats for which the type-name separator is `/`.
     *
     * @stability stable
     */
    static extractResourceName(arn, resourceType) {
        const components = parseArnShape(arn);
        if (components === 'token') {
            return cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(`:${resourceType}/`, arn));
        }
        // Apparently we could just parse this right away. Validate that we got the right
        // resource type (to notify authors of incorrect assumptions right away).
        const parsed = Arn.split(arn, ArnFormat.SLASH_RESOURCE_NAME);
        if (!token_1.Token.isUnresolved(parsed.resource) && parsed.resource !== resourceType) {
            throw new Error(`Expected resource type '${resourceType}' in ARN, got '${parsed.resource}' in '${arn}'`);
        }
        if (!parsed.resourceName) {
            throw new Error(`Expected resource name in ARN, didn't find one: '${arn}'`);
        }
        return parsed.resourceName;
    }
}
exports.Arn = Arn;
_a = JSII_RTTI_SYMBOL_1;
Arn[_a] = { fqn: "@aws-cdk/core.Arn", version: "1.139.0" };
/**
 * Given a Token evaluating to ARN, parses it and returns components.
 *
 * The ARN cannot be validated, since we don't have the actual value yet
 * at the time of this function call. You will have to know the separator
 * and the type of ARN.
 *
 * The resulting `ArnComponents` object will contain tokens for the
 * subexpressions of the ARN, not string literals.
 *
 * WARNING: this function cannot properly parse the complete final
 * 'resourceName' part if it contains colons,
 * like 'arn:aws:cognito-sync:region:account:identitypool/us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla'.
 *
 * @param arnToken The input token that contains an ARN
 * @param arnFormat the expected format of 'arn' - depends on what format the service the ARN represents uses
 */
function parseTokenArn(arnToken, arnFormat) {
    // ARN looks like:
    // arn:partition:service:region:account:resource
    // arn:partition:service:region:account:resource:resourceName
    // arn:partition:service:region:account:resource/resourceName
    // arn:partition:service:region:account:/resource/resourceName
    const components = cfn_fn_1.Fn.split(':', arnToken);
    const partition = cfn_fn_1.Fn.select(1, components).toString();
    const service = cfn_fn_1.Fn.select(2, components).toString();
    const region = cfn_fn_1.Fn.select(3, components).toString();
    const account = cfn_fn_1.Fn.select(4, components).toString();
    let resource;
    let resourceName;
    let sep;
    if (arnFormat === ArnFormat.NO_RESOURCE_NAME || arnFormat === ArnFormat.COLON_RESOURCE_NAME) {
        // we know that the 'resource' part will always be the 6th segment in this case
        resource = cfn_fn_1.Fn.select(5, components);
        if (arnFormat === ArnFormat.COLON_RESOURCE_NAME) {
            resourceName = cfn_fn_1.Fn.select(6, components);
            sep = ':';
        }
        else {
            resourceName = undefined;
            sep = undefined;
        }
    }
    else {
        // we know that the 'resource' and 'resourceName' parts are separated by slash here,
        // so we split the 6th segment from the colon-separated ones with a slash
        const lastComponents = cfn_fn_1.Fn.split('/', cfn_fn_1.Fn.select(5, components));
        if (arnFormat === ArnFormat.SLASH_RESOURCE_NAME) {
            resource = cfn_fn_1.Fn.select(0, lastComponents);
            resourceName = cfn_fn_1.Fn.select(1, lastComponents);
        }
        else {
            // arnFormat is ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
            // which means there's an extra slash there at the beginning that we need to skip
            resource = cfn_fn_1.Fn.select(1, lastComponents);
            resourceName = cfn_fn_1.Fn.select(2, lastComponents);
        }
        sep = '/';
    }
    return { partition, service, region, account, resource, resourceName, sep, arnFormat };
}
/**
 * Validate that a string is either unparseable or looks mostly like an ARN
 */
function parseArnShape(arn) {
    // assume anything that starts with 'arn:' is an ARN,
    // so we can report better errors
    const looksLikeArn = arn.startsWith('arn:');
    if (!looksLikeArn) {
        if (token_1.Token.isUnresolved(arn)) {
            return 'token';
        }
        else {
            throw new Error(`ARNs must start with "arn:" and have at least 6 components: ${arn}`);
        }
    }
    // If the ARN merely contains Tokens, but otherwise *looks* mostly like an ARN,
    // it's a string of the form 'arn:${partition}:service:${region}:${account}:resource/xyz'.
    // Parse fields out to the best of our ability.
    // Tokens won't contain ":", so this won't break them.
    const components = arn.split(':');
    // const [/* arn */, partition, service, /* region */ , /* account */ , resource] = components;
    const partition = components.length > 1 ? components[1] : undefined;
    if (!partition) {
        throw new Error('The `partition` component (2nd component) of an ARN is required: ' + arn);
    }
    const service = components.length > 2 ? components[2] : undefined;
    if (!service) {
        throw new Error('The `service` component (3rd component) of an ARN is required: ' + arn);
    }
    const resource = components.length > 5 ? components[5] : undefined;
    if (!resource) {
        throw new Error('The `resource` component (6th component) of an ARN is required: ' + arn);
    }
    // Region can be missing in global ARNs (such as used by IAM)
    // Account can be missing in some ARN types (such as used for S3 buckets)
    return components;
}
//# sourceMappingURL=data:application/json;base64,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