"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: "us-east-1" },
});
stack.templateOptions.description =
    "Integration Test for public HTTP API with a existing function and ALB";
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, "lambda-sg", {
    vpc: myVpc,
    allowAllOutbound: false,
});
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: "index.handler",
        vpc: myVpc,
        securityGroups: [testSg],
    },
});
const loadBalancer = defaults.ObtainAlb(stack, "existing-alb", {
    vpc: myVpc,
    publicApi: false,
    loadBalancerProps: {
        internetFacing: false,
        vpc: myVpc,
    },
    logAccessLogs: true,
    loggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
    }
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: "HTTP",
    },
    publicApi: true,
};
const albToLambda = new lib_1.AlbToLambda(stack, "test-one", props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    {
        id: "W56",
        reason: "All integration tests must be HTTP because of certificate limitations.",
    },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0]
    .node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
    { id: "W2", reason: "Rule does not apply for ELB." },
    { id: "W9", reason: "Rule does not apply for ELB." },
]);
defaults.addCfnSuppressRules(testSg, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
]);
defaults.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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