from __future__ import annotations

"""Helpers to send arbitrary Python objects to Ray as JSON.

Usage
-----

    from python_ray.auto_json import enable_auto_object_json
    enable_auto_object_json()

After calling this once at startup, ``ray(obj)`` will try to convert
non-primitive objects (dataclasses, Pydantic models, objects with ``dict()``
or ``__dict__``) into JSON-friendly dicts and send them as structured
JSON payloads instead of just ``repr(obj)``.
"""

from typing import Any, Optional
import dataclasses
import json

from .payloads import PayloadFactory, DecodedJsonPayload


def _object_to_json_payload(value: Any) -> Optional[DecodedJsonPayload]:
    """Best-effort conversion of arbitrary objects to a JSON payload.

    - Primitives are ignored (Ray's default handling is used).
    - Dataclasses use ``dataclasses.asdict``.
    - Pydantic v2-style models use ``model_dump()``.
    - Objects with ``dict()`` use that method.
    - Objects with ``__dict__`` use a filtered ``__dict__`` (no private attrs).
    """

    if isinstance(value, (str, int, float, bool)) or value is None:
        # Let Ray handle primitives as usual
        return None

    data: Any = None

    # 1) dataclasses
    if dataclasses.is_dataclass(value):
        data = dataclasses.asdict(value)

    # 2) Pydantic v2-style models
    elif hasattr(value, "model_dump") and callable(value.model_dump):  # type: ignore[attr-defined]
        data = value.model_dump()  # type: ignore[call-arg]

    # 3) Pydantic v1 / other models with .dict()
    elif hasattr(value, "dict") and callable(value.dict):  # type: ignore[attr-defined]
        data = value.dict()  # type: ignore[call-arg]

    # 4) Fallback to __dict__ (filtering private attrs)
    elif hasattr(value, "__dict__"):
        data = {
            k: v
            for k, v in value.__dict__.items()  # type: ignore[attr-defined]
            if not k.startswith("_")
        }

    if data is None:
        return None

    try:
        # Ensure the data is JSON-serializable; if not, fall back to default
        json.dumps(data)
    except TypeError:
        return None

    return DecodedJsonPayload(json.dumps(data))


def enable_auto_object_json() -> None:
    """Enable automatic JSON serialization for non-primitive objects.

    Call this once during application startup. After that, calls like
    ``ray(user)`` will try to send a structured JSON representation
    instead of a plain ``repr(user)``.
    """

    PayloadFactory.register_payload_finder(_object_to_json_payload)
