from __future__ import annotations

from dataclasses import dataclass, field
from time import time
from typing import List, Optional


@dataclass
class RateLimiter:
    max_calls: Optional[int]
    max_per_second: Optional[int]
    _timestamps: List[float] = field(default_factory=list)
    _notified: bool = False

    @classmethod
    def disabled(cls) -> "RateLimiter":
        return cls(None, None)

    def hit(self) -> "RateLimiter":
        now = time()
        self._timestamps.append(now)
        # prune timestamps older than 1 second for per-second window
        cutoff = now - 1.0
        self._timestamps = [t for t in self._timestamps if t >= cutoff]
        return self

    def max(self, max_calls: Optional[int]) -> "RateLimiter":
        self.max_calls = max_calls
        return self

    def per_second(self, calls_per_second: Optional[int]) -> "RateLimiter":
        self.max_per_second = calls_per_second
        return self

    def _total_count(self) -> int:
        return len(self._timestamps)

    def _last_second_count(self) -> int:
        if not self._timestamps:
            return 0
        now = time()
        cutoff = now - 1.0
        return sum(1 for t in self._timestamps if t >= cutoff)

    def is_max_reached(self) -> bool:
        if self.max_calls is None:
            return False
        reached = self._total_count() >= self.max_calls
        if not reached:
            self._notified = False
        return reached

    def is_max_per_second_reached(self) -> bool:
        if self.max_per_second is None:
            return False
        reached = self._last_second_count() >= self.max_per_second
        if not reached:
            self._notified = False
        return reached

    def clear(self) -> "RateLimiter":
        self.max_calls = None
        self.max_per_second = None
        self._timestamps.clear()
        return self

    def is_notified(self) -> bool:
        return self._notified

    def notify(self) -> None:
        self._notified = True
