import torch


def crop_steerable_pyramid_filters(filters, size):
    """
    Given original 9x9 NYU filters, this crops them to the desired size.
    The size must be an odd number >= 3
    Note this only crops the h0, l0 and band filters (not the l downsampling filter)

    Parameters
    ----------
    filters     : dict of torch.tensor
                    Filters to crop (should in format used by get_steerable_pyramid_filters.)
    size        : int
                    Size to crop to. For example, an input of 3 will crop the filters to a size of 3x3.

    Returns
    =======
    filters     : dict of torch.tensor
                    The cropped filters.
    """
    assert(size >= 3)
    assert(size % 2 == 1)
    r = (size-1) // 2

    def crop_filter(filter, r, normalise=True):
        r2 = (filter.size(-1)-1)//2
        filter = filter[:, :, r2-r:r2+r+1, r2-r:r2+r+1]
        if normalise:
            filter -= torch.sum(filter)
        return filter

    filters["h0"] = crop_filter(filters["h0"], r, normalise=False)
    sum_l = torch.sum(filters["l"])
    filters["l"] = crop_filter(filters["l"], 6, normalise=False)
    filters["l"] *= sum_l / torch.sum(filters["l"])
    sum_l0 = torch.sum(filters["l0"])
    filters["l0"] = crop_filter(filters["l0"], 2, normalise=False)
    filters["l0"] *= sum_l0 / torch.sum(filters["l0"])
    for b in range(len(filters["b"])):
        filters["b"][b] = crop_filter(filters["b"][b], r, normalise=True)
    return filters


def get_steerable_pyramid_filters(size, n_orientations, filter_type):
    """
    This returns filters for a real-valued steerable pyramid.

    Parameters
    ----------

    size            : int
                        Width of the filters (e.g. 3 will return 3x3 filters)
    n_orientations  : int
                        Number of oriented band filters
    filter_type     :  str
                        This can be used to select between the original NYU filters and cropped or trained alternatives.
                        full: Original NYU filters from https://github.com/LabForComputationalVision/pyrtools/blob/master/pyrtools/pyramids/filters.py
                        cropped: Some filters are cut back in size by extracting the centre and scaling as appropriate.
                        trained: Same as reduced, but the oriented kernels are replaced by learned 5x5 kernels.

    Returns
    =======

    filters         : dict of torch.tensor
                        The steerable pyramid filters. Returned as a dict with the following keys:
                        "l" The lowpass downsampling filter
                        "l0" The lowpass residual filter
                        "h0" The highpass residual filter
                        "b" The band filters (a list of torch.tensor filters, one for each orientation).
    """

    if filter_type != "full" and filter_type != "cropped" and filter_type != "trained":
        raise Exception(
            "Unknown filter type %s! Only filter types are full, cropped or trained." % filter_type)

    filters = {}
    if n_orientations == 1:
        filters["l"] = torch.tensor([
            [-2.257000e-04, -8.064400e-04, -5.686000e-05, 8.741400e-04, -1.862800e-04, -1.031640e-03, -
                1.871920e-03, -1.031640e-03, -1.862800e-04, 8.741400e-04, -5.686000e-05, -8.064400e-04, -2.257000e-04],
            [-8.064400e-04, 1.417620e-03, -1.903800e-04, -2.449060e-03, -4.596420e-03, -7.006740e-03, -
                6.948900e-03, -7.006740e-03, -4.596420e-03, -2.449060e-03, -1.903800e-04, 1.417620e-03, -8.064400e-04],
            [-5.686000e-05, -1.903800e-04, -3.059760e-03, -6.401000e-03, -6.720800e-03, -5.236180e-03, -
                3.781600e-03, -5.236180e-03, -6.720800e-03, -6.401000e-03, -3.059760e-03, -1.903800e-04, -5.686000e-05],
            [8.741400e-04, -2.449060e-03, -6.401000e-03, -5.260020e-03, 3.938620e-03, 1.722078e-02, 2.449600e-02,
                1.722078e-02, 3.938620e-03, -5.260020e-03, -6.401000e-03, -2.449060e-03, 8.741400e-04],
            [-1.862800e-04, -4.596420e-03, -6.720800e-03, 3.938620e-03, 3.220744e-02, 6.306262e-02, 7.624674e-02,
                6.306262e-02, 3.220744e-02, 3.938620e-03, -6.720800e-03, -4.596420e-03, -1.862800e-04],
            [-1.031640e-03, -7.006740e-03, -5.236180e-03, 1.722078e-02, 6.306262e-02, 1.116388e-01, 1.348999e-01,
                1.116388e-01, 6.306262e-02, 1.722078e-02, -5.236180e-03, -7.006740e-03, -1.031640e-03],
            [-1.871920e-03, -6.948900e-03, -3.781600e-03, 2.449600e-02, 7.624674e-02, 1.348999e-01, 1.576508e-01,
                1.348999e-01, 7.624674e-02, 2.449600e-02, -3.781600e-03, -6.948900e-03, -1.871920e-03],
            [-1.031640e-03, -7.006740e-03, -5.236180e-03, 1.722078e-02, 6.306262e-02, 1.116388e-01, 1.348999e-01,
                1.116388e-01, 6.306262e-02, 1.722078e-02, -5.236180e-03, -7.006740e-03, -1.031640e-03],
            [-1.862800e-04, -4.596420e-03, -6.720800e-03, 3.938620e-03, 3.220744e-02, 6.306262e-02, 7.624674e-02,
                6.306262e-02, 3.220744e-02, 3.938620e-03, -6.720800e-03, -4.596420e-03, -1.862800e-04],
            [8.741400e-04, -2.449060e-03, -6.401000e-03, -5.260020e-03, 3.938620e-03, 1.722078e-02, 2.449600e-02,
                1.722078e-02, 3.938620e-03, -5.260020e-03, -6.401000e-03, -2.449060e-03, 8.741400e-04],
            [-5.686000e-05, -1.903800e-04, -3.059760e-03, -6.401000e-03, -6.720800e-03, -5.236180e-03, -
                3.781600e-03, -5.236180e-03, -6.720800e-03, -6.401000e-03, -3.059760e-03, -1.903800e-04, -5.686000e-05],
            [-8.064400e-04, 1.417620e-03, -1.903800e-04, -2.449060e-03, -4.596420e-03, -7.006740e-03, -
                6.948900e-03, -7.006740e-03, -4.596420e-03, -2.449060e-03, -1.903800e-04, 1.417620e-03, -8.064400e-04],
            [-2.257000e-04, -8.064400e-04, -5.686000e-05, 8.741400e-04, -1.862800e-04, -1.031640e-03, -1.871920e-03, -1.031640e-03, -1.862800e-04, 8.741400e-04, -5.686000e-05, -8.064400e-04, -2.257000e-04]]
        ).reshape(1, 1, 13, 13)
        filters["l0"] = torch.tensor([
            [-4.514000e-04, -1.137100e-04, -3.725800e-04, -
                3.743860e-03, -3.725800e-04, -1.137100e-04, -4.514000e-04],
            [-1.137100e-04, -6.119520e-03, -1.344160e-02, -
                7.563200e-03, -1.344160e-02, -6.119520e-03, -1.137100e-04],
            [-3.725800e-04, -1.344160e-02, 6.441488e-02, 1.524935e-01,
                6.441488e-02, -1.344160e-02, -3.725800e-04],
            [-3.743860e-03, -7.563200e-03, 1.524935e-01, 3.153017e-01,
                1.524935e-01, -7.563200e-03, -3.743860e-03],
            [-3.725800e-04, -1.344160e-02, 6.441488e-02, 1.524935e-01,
                6.441488e-02, -1.344160e-02, -3.725800e-04],
            [-1.137100e-04, -6.119520e-03, -1.344160e-02, -
                7.563200e-03, -1.344160e-02, -6.119520e-03, -1.137100e-04],
            [-4.514000e-04, -1.137100e-04, -3.725800e-04, -3.743860e-03, -3.725800e-04, -1.137100e-04, -4.514000e-04]]
        ).reshape(1, 1, 7, 7)
        filters["h0"] = torch.tensor([
            [5.997200e-04, -6.068000e-05, -3.324900e-04, -3.325600e-04, -
                2.406600e-04, -3.325600e-04, -3.324900e-04, -6.068000e-05, 5.997200e-04],
            [-6.068000e-05, 1.263100e-04, 4.927100e-04, 1.459700e-04, -
                3.732100e-04, 1.459700e-04, 4.927100e-04, 1.263100e-04, -6.068000e-05],
            [-3.324900e-04, 4.927100e-04, -1.616650e-03, -1.437358e-02, -
                2.420138e-02, -1.437358e-02, -1.616650e-03, 4.927100e-04, -3.324900e-04],
            [-3.325600e-04, 1.459700e-04, -1.437358e-02, -6.300923e-02, -
                9.623594e-02, -6.300923e-02, -1.437358e-02, 1.459700e-04, -3.325600e-04],
            [-2.406600e-04, -3.732100e-04, -2.420138e-02, -9.623594e-02,
                8.554893e-01, -9.623594e-02, -2.420138e-02, -3.732100e-04, -2.406600e-04],
            [-3.325600e-04, 1.459700e-04, -1.437358e-02, -6.300923e-02, -
                9.623594e-02, -6.300923e-02, -1.437358e-02, 1.459700e-04, -3.325600e-04],
            [-3.324900e-04, 4.927100e-04, -1.616650e-03, -1.437358e-02, -
                2.420138e-02, -1.437358e-02, -1.616650e-03, 4.927100e-04, -3.324900e-04],
            [-6.068000e-05, 1.263100e-04, 4.927100e-04, 1.459700e-04, -
                3.732100e-04, 1.459700e-04, 4.927100e-04, 1.263100e-04, -6.068000e-05],
            [5.997200e-04, -6.068000e-05, -3.324900e-04, -3.325600e-04, -2.406600e-04, -3.325600e-04, -3.324900e-04, -6.068000e-05, 5.997200e-04]]
        ).reshape(1, 1, 9, 9)
        filters["b"] = []
        filters["b"].append(torch.tensor([
            -9.066000e-05, -1.738640e-03, -4.942500e-03, -7.889390e-03, -
            1.009473e-02, -7.889390e-03, -4.942500e-03, -1.738640e-03, -9.066000e-05,
            -1.738640e-03, -4.625150e-03, -7.272540e-03, -7.623410e-03, -
            9.091950e-03, -7.623410e-03, -7.272540e-03, -4.625150e-03, -1.738640e-03,
            -4.942500e-03, -7.272540e-03, -2.129540e-02, -2.435662e-02, -
            3.487008e-02, -2.435662e-02, -2.129540e-02, -7.272540e-03, -4.942500e-03,
            -7.889390e-03, -7.623410e-03, -2.435662e-02, -1.730466e-02, -
            3.158605e-02, -1.730466e-02, -2.435662e-02, -7.623410e-03, -7.889390e-03,
            -1.009473e-02, -9.091950e-03, -3.487008e-02, -3.158605e-02, 9.464195e-01, -
            3.158605e-02, -3.487008e-02, -9.091950e-03, -1.009473e-02,
            -7.889390e-03, -7.623410e-03, -2.435662e-02, -1.730466e-02, -
            3.158605e-02, -1.730466e-02, -2.435662e-02, -7.623410e-03, -7.889390e-03,
            -4.942500e-03, -7.272540e-03, -2.129540e-02, -2.435662e-02, -
            3.487008e-02, -2.435662e-02, -2.129540e-02, -7.272540e-03, -4.942500e-03,
            -1.738640e-03, -4.625150e-03, -7.272540e-03, -7.623410e-03, -
            9.091950e-03, -7.623410e-03, -7.272540e-03, -4.625150e-03, -1.738640e-03,
            -9.066000e-05, -1.738640e-03, -4.942500e-03, -7.889390e-03, -1.009473e-02, -7.889390e-03, -4.942500e-03, -1.738640e-03, -9.066000e-05]
        ).reshape(1, 1, 9, 9).permute(0, 1, 3, 2))

    elif n_orientations == 2:
        filters["l"] = torch.tensor(
            [[-4.350000e-05, 1.207800e-04, -6.771400e-04, -1.243400e-04, -8.006400e-04, -1.597040e-03, -2.516800e-04, -4.202000e-04, 1.262000e-03, -4.202000e-04, -2.516800e-04, -1.597040e-03, -8.006400e-04, -1.243400e-04, -6.771400e-04, 1.207800e-04, -4.350000e-05],
             [1.207800e-04, 4.460600e-04, -5.814600e-04, 5.621600e-04, -1.368800e-04, 2.325540e-03, 2.889860e-03, 4.287280e-03, 5.589400e-03,
                 4.287280e-03, 2.889860e-03, 2.325540e-03, -1.368800e-04, 5.621600e-04, -5.814600e-04, 4.460600e-04, 1.207800e-04],
             [-6.771400e-04, -5.814600e-04, 1.460780e-03, 2.160540e-03, 3.761360e-03, 3.080980e-03, 4.112200e-03, 2.221220e-03, 5.538200e-04,
                 2.221220e-03, 4.112200e-03, 3.080980e-03, 3.761360e-03, 2.160540e-03, 1.460780e-03, -5.814600e-04, -6.771400e-04],
             [-1.243400e-04, 5.621600e-04, 2.160540e-03, 3.175780e-03, 3.184680e-03, -1.777480e-03, -7.431700e-03, -9.056920e-03, -
                 9.637220e-03, -9.056920e-03, -7.431700e-03, -1.777480e-03, 3.184680e-03, 3.175780e-03, 2.160540e-03, 5.621600e-04, -1.243400e-04],
             [-8.006400e-04, -1.368800e-04, 3.761360e-03, 3.184680e-03, -3.530640e-03, -1.260420e-02, -1.884744e-02, -1.750818e-02, -
                 1.648568e-02, -1.750818e-02, -1.884744e-02, -1.260420e-02, -3.530640e-03, 3.184680e-03, 3.761360e-03, -1.368800e-04, -8.006400e-04],
             [-1.597040e-03, 2.325540e-03, 3.080980e-03, -1.777480e-03, -1.260420e-02, -2.022938e-02, -1.109170e-02, 3.955660e-03, 1.438512e-02,
                 3.955660e-03, -1.109170e-02, -2.022938e-02, -1.260420e-02, -1.777480e-03, 3.080980e-03, 2.325540e-03, -1.597040e-03],
             [-2.516800e-04, 2.889860e-03, 4.112200e-03, -7.431700e-03, -1.884744e-02, -1.109170e-02, 2.190660e-02, 6.806584e-02, 9.058014e-02,
                 6.806584e-02, 2.190660e-02, -1.109170e-02, -1.884744e-02, -7.431700e-03, 4.112200e-03, 2.889860e-03, -2.516800e-04],
             [-4.202000e-04, 4.287280e-03, 2.221220e-03, -9.056920e-03, -1.750818e-02, 3.955660e-03, 6.806584e-02, 1.445500e-01, 1.773651e-01,
                 1.445500e-01, 6.806584e-02, 3.955660e-03, -1.750818e-02, -9.056920e-03, 2.221220e-03, 4.287280e-03, -4.202000e-04],
             [1.262000e-03, 5.589400e-03, 5.538200e-04, -9.637220e-03, -1.648568e-02, 1.438512e-02, 9.058014e-02, 1.773651e-01, 2.120374e-01,
                 1.773651e-01, 9.058014e-02, 1.438512e-02, -1.648568e-02, -9.637220e-03, 5.538200e-04, 5.589400e-03, 1.262000e-03],
             [-4.202000e-04, 4.287280e-03, 2.221220e-03, -9.056920e-03, -1.750818e-02, 3.955660e-03, 6.806584e-02, 1.445500e-01, 1.773651e-01,
                 1.445500e-01, 6.806584e-02, 3.955660e-03, -1.750818e-02, -9.056920e-03, 2.221220e-03, 4.287280e-03, -4.202000e-04],
             [-2.516800e-04, 2.889860e-03, 4.112200e-03, -7.431700e-03, -1.884744e-02, -1.109170e-02, 2.190660e-02, 6.806584e-02, 9.058014e-02,
                 6.806584e-02, 2.190660e-02, -1.109170e-02, -1.884744e-02, -7.431700e-03, 4.112200e-03, 2.889860e-03, -2.516800e-04],
             [-1.597040e-03, 2.325540e-03, 3.080980e-03, -1.777480e-03, -1.260420e-02, -2.022938e-02, -1.109170e-02, 3.955660e-03, 1.438512e-02,
                 3.955660e-03, -1.109170e-02, -2.022938e-02, -1.260420e-02, -1.777480e-03, 3.080980e-03, 2.325540e-03, -1.597040e-03],
             [-8.006400e-04, -1.368800e-04, 3.761360e-03, 3.184680e-03, -3.530640e-03, -1.260420e-02, -1.884744e-02, -1.750818e-02, -
                 1.648568e-02, -1.750818e-02, -1.884744e-02, -1.260420e-02, -3.530640e-03, 3.184680e-03, 3.761360e-03, -1.368800e-04, -8.006400e-04],
             [-1.243400e-04, 5.621600e-04, 2.160540e-03, 3.175780e-03, 3.184680e-03, -1.777480e-03, -7.431700e-03, -9.056920e-03, -
                 9.637220e-03, -9.056920e-03, -7.431700e-03, -1.777480e-03, 3.184680e-03, 3.175780e-03, 2.160540e-03, 5.621600e-04, -1.243400e-04],
             [-6.771400e-04, -5.814600e-04, 1.460780e-03, 2.160540e-03, 3.761360e-03, 3.080980e-03, 4.112200e-03, 2.221220e-03, 5.538200e-04,
                 2.221220e-03, 4.112200e-03, 3.080980e-03, 3.761360e-03, 2.160540e-03, 1.460780e-03, -5.814600e-04, -6.771400e-04],
             [1.207800e-04, 4.460600e-04, -5.814600e-04, 5.621600e-04, -1.368800e-04, 2.325540e-03, 2.889860e-03, 4.287280e-03, 5.589400e-03,
                 4.287280e-03, 2.889860e-03, 2.325540e-03, -1.368800e-04, 5.621600e-04, -5.814600e-04, 4.460600e-04, 1.207800e-04],
             [-4.350000e-05, 1.207800e-04, -6.771400e-04, -1.243400e-04, -8.006400e-04, -1.597040e-03, -2.516800e-04, -4.202000e-04, 1.262000e-03, -4.202000e-04, -2.516800e-04, -1.597040e-03, -8.006400e-04, -1.243400e-04, -6.771400e-04, 1.207800e-04, -4.350000e-05]]
        ).reshape(1, 1, 17, 17)
        filters["l0"] = torch.tensor(
            [[-8.701000e-05, -1.354280e-03, -1.601260e-03, -5.033700e-04, 2.524010e-03, -5.033700e-04, -1.601260e-03, -1.354280e-03, -8.701000e-05],
             [-1.354280e-03, 2.921580e-03, 7.522720e-03, 8.224420e-03, 1.107620e-03,
                 8.224420e-03, 7.522720e-03, 2.921580e-03, -1.354280e-03],
             [-1.601260e-03, 7.522720e-03, -7.061290e-03, -3.769487e-02, -
                 3.297137e-02, -3.769487e-02, -7.061290e-03, 7.522720e-03, -1.601260e-03],
             [-5.033700e-04, 8.224420e-03, -3.769487e-02, 4.381320e-02, 1.811603e-01,
                 4.381320e-02, -3.769487e-02, 8.224420e-03, -5.033700e-04],
             [2.524010e-03, 1.107620e-03, -3.297137e-02, 1.811603e-01, 4.376250e-01,
                 1.811603e-01, -3.297137e-02, 1.107620e-03, 2.524010e-03],
             [-5.033700e-04, 8.224420e-03, -3.769487e-02, 4.381320e-02, 1.811603e-01,
                 4.381320e-02, -3.769487e-02, 8.224420e-03, -5.033700e-04],
             [-1.601260e-03, 7.522720e-03, -7.061290e-03, -3.769487e-02, -
                 3.297137e-02, -3.769487e-02, -7.061290e-03, 7.522720e-03, -1.601260e-03],
             [-1.354280e-03, 2.921580e-03, 7.522720e-03, 8.224420e-03, 1.107620e-03,
                 8.224420e-03, 7.522720e-03, 2.921580e-03, -1.354280e-03],
             [-8.701000e-05, -1.354280e-03, -1.601260e-03, -5.033700e-04, 2.524010e-03, -5.033700e-04, -1.601260e-03, -1.354280e-03, -8.701000e-05]]
        ).reshape(1, 1, 9, 9)
        filters["h0"] = torch.tensor(
            [[-9.570000e-04, -2.424100e-04, -1.424720e-03, -8.742600e-04, -1.166810e-03, -8.742600e-04, -1.424720e-03, -2.424100e-04, -9.570000e-04],
             [-2.424100e-04, -4.317530e-03, 8.998600e-04, 9.156420e-03, 1.098012e-02,
                 9.156420e-03, 8.998600e-04, -4.317530e-03, -2.424100e-04],
             [-1.424720e-03, 8.998600e-04, 1.706347e-02, 1.094866e-02, -
                 5.897780e-03, 1.094866e-02, 1.706347e-02, 8.998600e-04, -1.424720e-03],
             [-8.742600e-04, 9.156420e-03, 1.094866e-02, -7.841370e-02, -
                 1.562827e-01, -7.841370e-02, 1.094866e-02, 9.156420e-03, -8.742600e-04],
             [-1.166810e-03, 1.098012e-02, -5.897780e-03, -1.562827e-01,
                 7.282593e-01, -1.562827e-01, -5.897780e-03, 1.098012e-02, -1.166810e-03],
             [-8.742600e-04, 9.156420e-03, 1.094866e-02, -7.841370e-02, -
                 1.562827e-01, -7.841370e-02, 1.094866e-02, 9.156420e-03, -8.742600e-04],
             [-1.424720e-03, 8.998600e-04, 1.706347e-02, 1.094866e-02, -
                 5.897780e-03, 1.094866e-02, 1.706347e-02, 8.998600e-04, -1.424720e-03],
             [-2.424100e-04, -4.317530e-03, 8.998600e-04, 9.156420e-03, 1.098012e-02,
                 9.156420e-03, 8.998600e-04, -4.317530e-03, -2.424100e-04],
             [-9.570000e-04, -2.424100e-04, -1.424720e-03, -8.742600e-04, -1.166810e-03, -8.742600e-04, -1.424720e-03, -2.424100e-04, -9.570000e-04]]
        ).reshape(1, 1, 9, 9)
        filters["b"] = []
        filters["b"].append(torch.tensor(
            [6.125880e-03, -8.052600e-03, -2.103714e-02, -1.536890e-02, -1.851466e-02, -1.536890e-02, -2.103714e-02, -8.052600e-03, 6.125880e-03,
             -1.287416e-02, -9.611520e-03, 1.023569e-02, 6.009450e-03, 1.872620e-03, 6.009450e-03, 1.023569e-02, -
             9.611520e-03, -1.287416e-02,
             -5.641530e-03, 4.168400e-03, -2.382180e-02, -5.375324e-02, -
             2.076086e-02, -5.375324e-02, -2.382180e-02, 4.168400e-03, -5.641530e-03,
             -8.957260e-03, -1.751170e-03, -1.836909e-02, 1.265655e-01, 2.996168e-01, 1.265655e-01, -
             1.836909e-02, -1.751170e-03, -8.957260e-03,
             0.000000e+00, 0.000000e+00, 0.000000e+00, 0.000000e+00, 0.000000e+00, 0.000000e+00, 0.000000e+00, 0.000000e+00, 0.000000e+00,
             8.957260e-03, 1.751170e-03, 1.836909e-02, -1.265655e-01, -
             2.996168e-01, -1.265655e-01, 1.836909e-02, 1.751170e-03, 8.957260e-03,
             5.641530e-03, -4.168400e-03, 2.382180e-02, 5.375324e-02, 2.076086e-02, 5.375324e-02, 2.382180e-02, -
             4.168400e-03, 5.641530e-03,
             1.287416e-02, 9.611520e-03, -1.023569e-02, -6.009450e-03, -
             1.872620e-03, -6.009450e-03, -1.023569e-02, 9.611520e-03, 1.287416e-02,
             -6.125880e-03, 8.052600e-03, 2.103714e-02, 1.536890e-02, 1.851466e-02, 1.536890e-02, 2.103714e-02, 8.052600e-03, -6.125880e-03]).reshape(1, 1, 9, 9).permute(0, 1, 3, 2))
        filters["b"].append(torch.tensor(
            [-6.125880e-03, 1.287416e-02, 5.641530e-03, 8.957260e-03, 0.000000e+00, -8.957260e-03, -5.641530e-03, -1.287416e-02, 6.125880e-03,
             8.052600e-03, 9.611520e-03, -4.168400e-03, 1.751170e-03, 0.000000e+00, -
             1.751170e-03, 4.168400e-03, -9.611520e-03, -8.052600e-03,
             2.103714e-02, -1.023569e-02, 2.382180e-02, 1.836909e-02, 0.000000e+00, -
             1.836909e-02, -2.382180e-02, 1.023569e-02, -2.103714e-02,
             1.536890e-02, -6.009450e-03, 5.375324e-02, -
             1.265655e-01, 0.000000e+00, 1.265655e-01, -
             5.375324e-02, 6.009450e-03, -1.536890e-02,
             1.851466e-02, -1.872620e-03, 2.076086e-02, -
             2.996168e-01, 0.000000e+00, 2.996168e-01, -
             2.076086e-02, 1.872620e-03, -1.851466e-02,
             1.536890e-02, -6.009450e-03, 5.375324e-02, -
             1.265655e-01, 0.000000e+00, 1.265655e-01, -
             5.375324e-02, 6.009450e-03, -1.536890e-02,
             2.103714e-02, -1.023569e-02, 2.382180e-02, 1.836909e-02, 0.000000e+00, -
             1.836909e-02, -2.382180e-02, 1.023569e-02, -2.103714e-02,
             8.052600e-03, 9.611520e-03, -4.168400e-03, 1.751170e-03, 0.000000e+00, -
             1.751170e-03, 4.168400e-03, -9.611520e-03, -8.052600e-03,
             -6.125880e-03, 1.287416e-02, 5.641530e-03, 8.957260e-03, 0.000000e+00, -8.957260e-03, -5.641530e-03, -1.287416e-02, 6.125880e-03]).reshape(1, 1, 9, 9).permute(0, 1, 3, 2))

    elif n_orientations == 4:
        filters["l"] = torch.tensor([
            [-4.3500000174E-5, 1.2078000145E-4, -6.7714002216E-4, -1.2434000382E-4, -8.0063997302E-4, -1.5970399836E-3, -2.5168000138E-4, -4.2019999819E-4,
                1.2619999470E-3, -4.2019999819E-4, -2.5168000138E-4, -1.5970399836E-3, -8.0063997302E-4, -1.2434000382E-4, -6.7714002216E-4, 1.2078000145E-4, -4.3500000174E-5],
            [1.2078000145E-4, 4.4606000301E-4, -5.8146001538E-4, 5.6215998484E-4, -1.3688000035E-4, 2.3255399428E-3, 2.8898599558E-3, 4.2872801423E-3, 5.5893999524E-3,
                4.2872801423E-3, 2.8898599558E-3, 2.3255399428E-3, -1.3688000035E-4, 5.6215998484E-4, -5.8146001538E-4, 4.4606000301E-4, 1.2078000145E-4],
            [-6.7714002216E-4, -5.8146001538E-4, 1.4607800404E-3, 2.1605400834E-3, 3.7613599561E-3, 3.0809799209E-3, 4.1121998802E-3, 2.2212199401E-3, 5.5381999118E-4,
                2.2212199401E-3, 4.1121998802E-3, 3.0809799209E-3, 3.7613599561E-3, 2.1605400834E-3, 1.4607800404E-3, -5.8146001538E-4, -6.7714002216E-4],
            [-1.2434000382E-4, 5.6215998484E-4, 2.1605400834E-3, 3.1757799443E-3, 3.1846798956E-3, -1.7774800071E-3, -7.4316998944E-3, -9.0569201857E-3, -
                9.6372198313E-3, -9.0569201857E-3, -7.4316998944E-3, -1.7774800071E-3, 3.1846798956E-3, 3.1757799443E-3, 2.1605400834E-3, 5.6215998484E-4, -1.2434000382E-4],
            [-8.0063997302E-4, -1.3688000035E-4, 3.7613599561E-3, 3.1846798956E-3, -3.5306399222E-3, -1.2604200281E-2, -1.8847439438E-2, -1.7508180812E-2, -
                1.6485679895E-2, -1.7508180812E-2, -1.8847439438E-2, -1.2604200281E-2, -3.5306399222E-3, 3.1846798956E-3, 3.7613599561E-3, -1.3688000035E-4, -8.0063997302E-4],
            [-1.5970399836E-3, 2.3255399428E-3, 3.0809799209E-3, -1.7774800071E-3, -1.2604200281E-2, -2.0229380578E-2, -1.1091699824E-2, 3.9556599222E-3, 1.4385120012E-2,
                3.9556599222E-3, -1.1091699824E-2, -2.0229380578E-2, -1.2604200281E-2, -1.7774800071E-3, 3.0809799209E-3, 2.3255399428E-3, -1.5970399836E-3],
            [-2.5168000138E-4, 2.8898599558E-3, 4.1121998802E-3, -7.4316998944E-3, -1.8847439438E-2, -1.1091699824E-2, 2.1906599402E-2, 6.8065837026E-2, 9.0580143034E-2,
                6.8065837026E-2, 2.1906599402E-2, -1.1091699824E-2, -1.8847439438E-2, -7.4316998944E-3, 4.1121998802E-3, 2.8898599558E-3, -2.5168000138E-4],
            [-4.2019999819E-4, 4.2872801423E-3, 2.2212199401E-3, -9.0569201857E-3, -1.7508180812E-2, 3.9556599222E-3, 6.8065837026E-2, 0.1445499808, 0.1773651242,
                0.1445499808, 6.8065837026E-2, 3.9556599222E-3, -1.7508180812E-2, -9.0569201857E-3, 2.2212199401E-3, 4.2872801423E-3, -4.2019999819E-4],
            [1.2619999470E-3, 5.5893999524E-3, 5.5381999118E-4, -9.6372198313E-3, -1.6485679895E-2, 1.4385120012E-2, 9.0580143034E-2, 0.1773651242, 0.2120374441,
                0.1773651242, 9.0580143034E-2, 1.4385120012E-2, -1.6485679895E-2, -9.6372198313E-3, 5.5381999118E-4, 5.5893999524E-3, 1.2619999470E-3],
            [-4.2019999819E-4, 4.2872801423E-3, 2.2212199401E-3, -9.0569201857E-3, -1.7508180812E-2, 3.9556599222E-3, 6.8065837026E-2, 0.1445499808, 0.1773651242,
                0.1445499808, 6.8065837026E-2, 3.9556599222E-3, -1.7508180812E-2, -9.0569201857E-3, 2.2212199401E-3, 4.2872801423E-3, -4.2019999819E-4],
            [-2.5168000138E-4, 2.8898599558E-3, 4.1121998802E-3, -7.4316998944E-3, -1.8847439438E-2, -1.1091699824E-2, 2.1906599402E-2, 6.8065837026E-2, 9.0580143034E-2,
                6.8065837026E-2, 2.1906599402E-2, -1.1091699824E-2, -1.8847439438E-2, -7.4316998944E-3, 4.1121998802E-3, 2.8898599558E-3, -2.5168000138E-4],
            [-1.5970399836E-3, 2.3255399428E-3, 3.0809799209E-3, -1.7774800071E-3, -1.2604200281E-2, -2.0229380578E-2, -1.1091699824E-2, 3.9556599222E-3, 1.4385120012E-2,
                3.9556599222E-3, -1.1091699824E-2, -2.0229380578E-2, -1.2604200281E-2, -1.7774800071E-3, 3.0809799209E-3, 2.3255399428E-3, -1.5970399836E-3],
            [-8.0063997302E-4, -1.3688000035E-4, 3.7613599561E-3, 3.1846798956E-3, -3.5306399222E-3, -1.2604200281E-2, -1.8847439438E-2, -1.7508180812E-2, -
                1.6485679895E-2, -1.7508180812E-2, -1.8847439438E-2, -1.2604200281E-2, -3.5306399222E-3, 3.1846798956E-3, 3.7613599561E-3, -1.3688000035E-4, -8.0063997302E-4],
            [-1.2434000382E-4, 5.6215998484E-4, 2.1605400834E-3, 3.1757799443E-3, 3.1846798956E-3, -1.7774800071E-3, -7.4316998944E-3, -9.0569201857E-3, -
                9.6372198313E-3, -9.0569201857E-3, -7.4316998944E-3, -1.7774800071E-3, 3.1846798956E-3, 3.1757799443E-3, 2.1605400834E-3, 5.6215998484E-4, -1.2434000382E-4],
            [-6.7714002216E-4, -5.8146001538E-4, 1.4607800404E-3, 2.1605400834E-3, 3.7613599561E-3, 3.0809799209E-3, 4.1121998802E-3, 2.2212199401E-3, 5.5381999118E-4,
                2.2212199401E-3, 4.1121998802E-3, 3.0809799209E-3, 3.7613599561E-3, 2.1605400834E-3, 1.4607800404E-3, -5.8146001538E-4, -6.7714002216E-4],
            [1.2078000145E-4, 4.4606000301E-4, -5.8146001538E-4, 5.6215998484E-4, -1.3688000035E-4, 2.3255399428E-3, 2.8898599558E-3, 4.2872801423E-3, 5.5893999524E-3,
                4.2872801423E-3, 2.8898599558E-3, 2.3255399428E-3, -1.3688000035E-4, 5.6215998484E-4, -5.8146001538E-4, 4.4606000301E-4, 1.2078000145E-4],
            [-4.3500000174E-5, 1.2078000145E-4, -6.7714002216E-4, -1.2434000382E-4, -8.0063997302E-4, -1.5970399836E-3, -2.5168000138E-4, -4.2019999819E-4, 1.2619999470E-3, -4.2019999819E-4, -2.5168000138E-4, -1.5970399836E-3, -8.0063997302E-4, -1.2434000382E-4, -6.7714002216E-4, 1.2078000145E-4, -4.3500000174E-5]]
        ).reshape(1, 1, 17, 17)
        filters["l0"] = torch.tensor([
            [-8.7009997515E-5, -1.3542800443E-3, -1.6012600390E-3, -5.0337001448E-4,
                2.5240099058E-3, -5.0337001448E-4, -1.6012600390E-3, -1.3542800443E-3, -8.7009997515E-5],
            [-1.3542800443E-3, 2.9215801042E-3, 7.5227199122E-3, 8.2244202495E-3, 1.1076199589E-3,
                8.2244202495E-3, 7.5227199122E-3, 2.9215801042E-3, -1.3542800443E-3],
            [-1.6012600390E-3, 7.5227199122E-3, -7.0612900890E-3, -3.7694871426E-2, -
                3.2971370965E-2, -3.7694871426E-2, -7.0612900890E-3, 7.5227199122E-3, -1.6012600390E-3],
            [-5.0337001448E-4, 8.2244202495E-3, -3.7694871426E-2, 4.3813198805E-2, 0.1811603010,
                4.3813198805E-2, -3.7694871426E-2, 8.2244202495E-3, -5.0337001448E-4],
            [2.5240099058E-3, 1.1076199589E-3, -3.2971370965E-2, 0.1811603010, 0.4376249909,
                0.1811603010, -3.2971370965E-2, 1.1076199589E-3, 2.5240099058E-3],
            [-5.0337001448E-4, 8.2244202495E-3, -3.7694871426E-2, 4.3813198805E-2, 0.1811603010,
                4.3813198805E-2, -3.7694871426E-2, 8.2244202495E-3, -5.0337001448E-4],
            [-1.6012600390E-3, 7.5227199122E-3, -7.0612900890E-3, -3.7694871426E-2, -
                3.2971370965E-2, -3.7694871426E-2, -7.0612900890E-3, 7.5227199122E-3, -1.6012600390E-3],
            [-1.3542800443E-3, 2.9215801042E-3, 7.5227199122E-3, 8.2244202495E-3, 1.1076199589E-3,
                8.2244202495E-3, 7.5227199122E-3, 2.9215801042E-3, -1.3542800443E-3],
            [-8.7009997515E-5, -1.3542800443E-3, -1.6012600390E-3, -5.0337001448E-4, 2.5240099058E-3, -5.0337001448E-4, -1.6012600390E-3, -1.3542800443E-3, -8.7009997515E-5]]
        ).reshape(1, 1, 9, 9)
        filters["h0"] = torch.tensor([
            [-4.0483998600E-4, -6.2596000498E-4, -3.7829999201E-5, 8.8387000142E-4, 1.5450799838E-3, 1.9235999789E-3, 2.0687500946E-3, 2.0898699295E-3,
                2.0687500946E-3, 1.9235999789E-3, 1.5450799838E-3, 8.8387000142E-4, -3.7829999201E-5, -6.2596000498E-4, -4.0483998600E-4],
            [-6.2596000498E-4, -3.2734998967E-4, 7.7435001731E-4, 1.5874400269E-3, 2.1750701126E-3, 2.5626500137E-3, 2.2892199922E-3, 1.9755100366E-3,
                2.2892199922E-3, 2.5626500137E-3, 2.1750701126E-3, 1.5874400269E-3, 7.7435001731E-4, -3.2734998967E-4, -6.2596000498E-4],
            [-3.7829999201E-5, 7.7435001731E-4, 1.1793200392E-3, 1.4050999889E-3, 2.2253401112E-3, 2.1145299543E-3, 3.3578000148E-4, -
                8.3368999185E-4, 3.3578000148E-4, 2.1145299543E-3, 2.2253401112E-3, 1.4050999889E-3, 1.1793200392E-3, 7.7435001731E-4, -3.7829999201E-5],
            [8.8387000142E-4, 1.5874400269E-3, 1.4050999889E-3, 1.2960999738E-3, -4.9274001503E-4, -3.1295299996E-3, -4.5751798898E-3, -
                5.1014497876E-3, -4.5751798898E-3, -3.1295299996E-3, -4.9274001503E-4, 1.2960999738E-3, 1.4050999889E-3, 1.5874400269E-3, 8.8387000142E-4],
            [1.5450799838E-3, 2.1750701126E-3, 2.2253401112E-3, -4.9274001503E-4, -6.3222697936E-3, -2.7556000277E-3, 5.3632198833E-3, 7.3032598011E-3,
                5.3632198833E-3, -2.7556000277E-3, -6.3222697936E-3, -4.9274001503E-4, 2.2253401112E-3, 2.1750701126E-3, 1.5450799838E-3],
            [1.9235999789E-3, 2.5626500137E-3, 2.1145299543E-3, -3.1295299996E-3, -2.7556000277E-3, 1.3962360099E-2, 7.8046298586E-3, -
                9.3812197447E-3, 7.8046298586E-3, 1.3962360099E-2, -2.7556000277E-3, -3.1295299996E-3, 2.1145299543E-3, 2.5626500137E-3, 1.9235999789E-3],
            [2.0687500946E-3, 2.2892199922E-3, 3.3578000148E-4, -4.5751798898E-3, 5.3632198833E-3, 7.8046298586E-3, -7.9501636326E-2, -
                0.1554141641, -7.9501636326E-2, 7.8046298586E-3, 5.3632198833E-3, -4.5751798898E-3, 3.3578000148E-4, 2.2892199922E-3, 2.0687500946E-3],
            [2.0898699295E-3, 1.9755100366E-3, -8.3368999185E-4, -5.1014497876E-3, 7.3032598011E-3, -9.3812197447E-3, -0.1554141641,
                0.7303866148, -0.1554141641, -9.3812197447E-3, 7.3032598011E-3, -5.1014497876E-3, -8.3368999185E-4, 1.9755100366E-3, 2.0898699295E-3],
            [2.0687500946E-3, 2.2892199922E-3, 3.3578000148E-4, -4.5751798898E-3, 5.3632198833E-3, 7.8046298586E-3, -7.9501636326E-2, -
                0.1554141641, -7.9501636326E-2, 7.8046298586E-3, 5.3632198833E-3, -4.5751798898E-3, 3.3578000148E-4, 2.2892199922E-3, 2.0687500946E-3],
            [1.9235999789E-3, 2.5626500137E-3, 2.1145299543E-3, -3.1295299996E-3, -2.7556000277E-3, 1.3962360099E-2, 7.8046298586E-3, -
                9.3812197447E-3, 7.8046298586E-3, 1.3962360099E-2, -2.7556000277E-3, -3.1295299996E-3, 2.1145299543E-3, 2.5626500137E-3, 1.9235999789E-3],
            [1.5450799838E-3, 2.1750701126E-3, 2.2253401112E-3, -4.9274001503E-4, -6.3222697936E-3, -2.7556000277E-3, 5.3632198833E-3, 7.3032598011E-3,
                5.3632198833E-3, -2.7556000277E-3, -6.3222697936E-3, -4.9274001503E-4, 2.2253401112E-3, 2.1750701126E-3, 1.5450799838E-3],
            [8.8387000142E-4, 1.5874400269E-3, 1.4050999889E-3, 1.2960999738E-3, -4.9274001503E-4, -3.1295299996E-3, -4.5751798898E-3, -
                5.1014497876E-3, -4.5751798898E-3, -3.1295299996E-3, -4.9274001503E-4, 1.2960999738E-3, 1.4050999889E-3, 1.5874400269E-3, 8.8387000142E-4],
            [-3.7829999201E-5, 7.7435001731E-4, 1.1793200392E-3, 1.4050999889E-3, 2.2253401112E-3, 2.1145299543E-3, 3.3578000148E-4, -
                8.3368999185E-4, 3.3578000148E-4, 2.1145299543E-3, 2.2253401112E-3, 1.4050999889E-3, 1.1793200392E-3, 7.7435001731E-4, -3.7829999201E-5],
            [-6.2596000498E-4, -3.2734998967E-4, 7.7435001731E-4, 1.5874400269E-3, 2.1750701126E-3, 2.5626500137E-3, 2.2892199922E-3, 1.9755100366E-3,
                2.2892199922E-3, 2.5626500137E-3, 2.1750701126E-3, 1.5874400269E-3, 7.7435001731E-4, -3.2734998967E-4, -6.2596000498E-4],
            [-4.0483998600E-4, -6.2596000498E-4, -3.7829999201E-5, 8.8387000142E-4, 1.5450799838E-3, 1.9235999789E-3, 2.0687500946E-3, 2.0898699295E-3, 2.0687500946E-3, 1.9235999789E-3, 1.5450799838E-3, 8.8387000142E-4, -3.7829999201E-5, -6.2596000498E-4, -4.0483998600E-4]]
        ).reshape(1, 1, 15, 15)
        filters["b"] = []
        filters["b"].append(torch.tensor(
            [-8.1125000725E-4, 4.4451598078E-3, 1.2316980399E-2, 1.3955879956E-2,  1.4179450460E-2, 1.3955879956E-2, 1.2316980399E-2, 4.4451598078E-3, -8.1125000725E-4,
             3.9103501476E-3, 4.4565401040E-3, -5.8724298142E-3, -2.8760801069E-3, 8.5267601535E-3, -
             2.8760801069E-3, -5.8724298142E-3, 4.4565401040E-3, 3.9103501476E-3,
             1.3462699717E-3, -3.7740699481E-3, 8.2581602037E-3, 3.9442278445E-2, 5.3605638444E-2, 3.9442278445E-2, 8.2581602037E-3, -
             3.7740699481E-3, 1.3462699717E-3,
             7.4700999539E-4, -3.6522001028E-4, -2.2522680461E-2, -0.1105690673, -
             0.1768419296, -0.1105690673, -2.2522680461E-2, -3.6522001028E-4, 7.4700999539E-4,
             0.0000000000, 0.0000000000, 0.0000000000, 0.0000000000, 0.0000000000, 0.0000000000, 0.0000000000, 0.0000000000, 0.0000000000,
             -7.4700999539E-4, 3.6522001028E-4, 2.2522680461E-2, 0.1105690673, 0.1768419296, 0.1105690673, 2.2522680461E-2, 3.6522001028E-4, -7.4700999539E-4,
             -1.3462699717E-3, 3.7740699481E-3, -8.2581602037E-3, -3.9442278445E-2, -
             5.3605638444E-2, -3.9442278445E-2, -
             8.2581602037E-3, 3.7740699481E-3, -1.3462699717E-3,
             -3.9103501476E-3, -4.4565401040E-3, 5.8724298142E-3, 2.8760801069E-3, -
             8.5267601535E-3, 2.8760801069E-3, 5.8724298142E-3, -
             4.4565401040E-3, -3.9103501476E-3,
             8.1125000725E-4, -4.4451598078E-3, -1.2316980399E-2, -1.3955879956E-2, -1.4179450460E-2, -1.3955879956E-2, -1.2316980399E-2, -4.4451598078E-3, 8.1125000725E-4]
        ).reshape(1, 1, 9, 9).permute(0, 1, 3, 2))
        filters["b"].append(torch.tensor(
            [0.0000000000, -8.2846998703E-4, -5.7109999034E-5, 4.0110000555E-5, 4.6670897864E-3, 8.0871898681E-3, 1.4807609841E-2, 8.6204400286E-3, -3.1221499667E-3,
             8.2846998703E-4, 0.0000000000, -9.7479997203E-4, -6.9718998857E-3, -
             2.0865600090E-3, 2.3298799060E-3, -
             4.4814897701E-3, 1.4917500317E-2, 8.6204400286E-3,
             5.7109999034E-5, 9.7479997203E-4, 0.0000000000, -1.2145539746E-2, -
             2.4427289143E-2, 5.0797060132E-2, 3.2785870135E-2, -
             4.4814897701E-3, 1.4807609841E-2,
             -4.0110000555E-5, 6.9718998857E-3, 1.2145539746E-2, 0.0000000000, -
             0.1510555595, -8.2495503128E-2, 5.0797060132E-2, 2.3298799060E-3, 8.0871898681E-3,
             -4.6670897864E-3, 2.0865600090E-3, 2.4427289143E-2, 0.1510555595, 0.0000000000, -
             0.1510555595, -2.4427289143E-2, -2.0865600090E-3, 4.6670897864E-3,
             -8.0871898681E-3, -2.3298799060E-3, -5.0797060132E-2, 8.2495503128E-2, 0.1510555595, 0.0000000000, -
             1.2145539746E-2, -6.9718998857E-3, 4.0110000555E-5,
             -1.4807609841E-2, 4.4814897701E-3, -3.2785870135E-2, -
             5.0797060132E-2, 2.4427289143E-2, 1.2145539746E-2, 0.0000000000, -
             9.7479997203E-4, -5.7109999034E-5,
             -8.6204400286E-3, -1.4917500317E-2, 4.4814897701E-3, -
             2.3298799060E-3, 2.0865600090E-3, 6.9718998857E-3, 9.7479997203E-4, 0.0000000000, -8.2846998703E-4,
             3.1221499667E-3, -8.6204400286E-3, -1.4807609841E-2, -8.0871898681E-3, -4.6670897864E-3, -4.0110000555E-5, 5.7109999034E-5, 8.2846998703E-4, 0.0000000000]
        ).reshape(1, 1, 9, 9).permute(0, 1, 3, 2))
        filters["b"].append(torch.tensor(
            [8.1125000725E-4, -3.9103501476E-3, -1.3462699717E-3, -7.4700999539E-4, 0.0000000000, 7.4700999539E-4, 1.3462699717E-3, 3.9103501476E-3, -8.1125000725E-4,
             -4.4451598078E-3, -4.4565401040E-3, 3.7740699481E-3, 3.6522001028E-4, 0.0000000000, -
             3.6522001028E-4, -3.7740699481E-3, 4.4565401040E-3, 4.4451598078E-3,
             -1.2316980399E-2, 5.8724298142E-3, -8.2581602037E-3, 2.2522680461E-2, 0.0000000000, -
             2.2522680461E-2, 8.2581602037E-3, -5.8724298142E-3, 1.2316980399E-2,
             -1.3955879956E-2, 2.8760801069E-3, -3.9442278445E-2, 0.1105690673, 0.0000000000, -
             0.1105690673, 3.9442278445E-2, -2.8760801069E-3, 1.3955879956E-2,
             -1.4179450460E-2, -8.5267601535E-3, -5.3605638444E-2, 0.1768419296, 0.0000000000, -
             0.1768419296, 5.3605638444E-2, 8.5267601535E-3, 1.4179450460E-2,
             -1.3955879956E-2, 2.8760801069E-3, -3.9442278445E-2, 0.1105690673, 0.0000000000, -
             0.1105690673, 3.9442278445E-2, -2.8760801069E-3, 1.3955879956E-2,
             -1.2316980399E-2, 5.8724298142E-3, -8.2581602037E-3, 2.2522680461E-2, 0.0000000000, -
             2.2522680461E-2, 8.2581602037E-3, -5.8724298142E-3, 1.2316980399E-2,
             -4.4451598078E-3, -4.4565401040E-3, 3.7740699481E-3, 3.6522001028E-4, 0.0000000000, -
             3.6522001028E-4, -3.7740699481E-3, 4.4565401040E-3, 4.4451598078E-3,
             8.1125000725E-4, -3.9103501476E-3, -1.3462699717E-3, -7.4700999539E-4, 0.0000000000, 7.4700999539E-4, 1.3462699717E-3, 3.9103501476E-3, -8.1125000725E-4]
        ).reshape(1, 1, 9, 9).permute(0, 1, 3, 2))
        filters["b"].append(torch.tensor(
            [3.1221499667E-3, -8.6204400286E-3, -1.4807609841E-2, -8.0871898681E-3, -4.6670897864E-3, -4.0110000555E-5, 5.7109999034E-5, 8.2846998703E-4, 0.0000000000,
             -8.6204400286E-3, -1.4917500317E-2, 4.4814897701E-3, -
             2.3298799060E-3, 2.0865600090E-3, 6.9718998857E-3, 9.7479997203E-4, -
             0.0000000000, -8.2846998703E-4,
             -1.4807609841E-2, 4.4814897701E-3, -3.2785870135E-2, -
             5.0797060132E-2, 2.4427289143E-2, 1.2145539746E-2, 0.0000000000, -
             9.7479997203E-4, -5.7109999034E-5,
             -8.0871898681E-3, -2.3298799060E-3, -5.0797060132E-2, 8.2495503128E-2, 0.1510555595, -
             0.0000000000, -1.2145539746E-2, -6.9718998857E-3, 4.0110000555E-5,
             -4.6670897864E-3, 2.0865600090E-3, 2.4427289143E-2, 0.1510555595, 0.0000000000, -
             0.1510555595, -2.4427289143E-2, -2.0865600090E-3, 4.6670897864E-3,
             -4.0110000555E-5, 6.9718998857E-3, 1.2145539746E-2, 0.0000000000, -
             0.1510555595, -8.2495503128E-2, 5.0797060132E-2, 2.3298799060E-3, 8.0871898681E-3,
             5.7109999034E-5, 9.7479997203E-4, -0.0000000000, -1.2145539746E-2, -
             2.4427289143E-2, 5.0797060132E-2, 3.2785870135E-2, -
             4.4814897701E-3, 1.4807609841E-2,
             8.2846998703E-4, -0.0000000000, -9.7479997203E-4, -6.9718998857E-3, -
             2.0865600090E-3, 2.3298799060E-3, -
             4.4814897701E-3, 1.4917500317E-2, 8.6204400286E-3,
             0.0000000000, -8.2846998703E-4, -5.7109999034E-5, 4.0110000555E-5, 4.6670897864E-3, 8.0871898681E-3, 1.4807609841E-2, 8.6204400286E-3, -3.1221499667E-3]
        ).reshape(1, 1, 9, 9).permute(0, 1, 3, 2))

    elif n_orientations == 6:
        filters["l"] = 2 * torch.tensor([
            [0.00085404, -0.00244917, -0.00387812, -0.00944432, -
                0.00962054, -0.00944432, -0.00387812, -0.00244917, 0.00085404],
            [-0.00244917, -0.00523281, -0.00661117, 0.00410600, 0.01002988,
                0.00410600, -0.00661117, -0.00523281, -0.00244917],
            [-0.00387812, -0.00661117, 0.01396746, 0.03277038, 0.03981393,
                0.03277038, 0.01396746, -0.00661117, -0.00387812],
            [-0.00944432, 0.00410600, 0.03277038, 0.06426333, 0.08169618,
                0.06426333, 0.03277038, 0.00410600, -0.00944432],
            [-0.00962054, 0.01002988, 0.03981393, 0.08169618, 0.10096540,
                0.08169618, 0.03981393, 0.01002988, -0.00962054],
            [-0.00944432, 0.00410600, 0.03277038, 0.06426333, 0.08169618,
                0.06426333, 0.03277038, 0.00410600, -0.00944432],
            [-0.00387812, -0.00661117, 0.01396746, 0.03277038, 0.03981393,
                0.03277038, 0.01396746, -0.00661117, -0.00387812],
            [-0.00244917, -0.00523281, -0.00661117, 0.00410600, 0.01002988,
                0.00410600, -0.00661117, -0.00523281, -0.00244917],
            [0.00085404, -0.00244917, -0.00387812, -0.00944432, -0.00962054, -0.00944432, -0.00387812, -0.00244917, 0.00085404]]
        ).reshape(1, 1, 9, 9)
        filters["l0"] = torch.tensor([
            [0.00341614, -0.01551246, -0.03848215, -0.01551246, 0.00341614],
            [-0.01551246, 0.05586982, 0.15925570, 0.05586982, -0.01551246],
            [-0.03848215, 0.15925570, 0.40304148, 0.15925570, -0.03848215],
            [-0.01551246, 0.05586982, 0.15925570, 0.05586982, -0.01551246],
            [0.00341614, -0.01551246, -0.03848215, -0.01551246, 0.00341614]]
        ).reshape(1, 1, 5, 5)
        filters["h0"] = torch.tensor([
            [-0.00033429, -0.00113093, -0.00171484, -0.00133542, -
                0.00080639, -0.00133542, -0.00171484, -0.00113093, -0.00033429],
            [-0.00113093, -0.00350017, -0.00243812, 0.00631653, 0.01261227,
                0.00631653, -0.00243812, -0.00350017, -0.00113093],
            [-0.00171484, -0.00243812, -0.00290081, -0.00673482, -
                0.00981051, -0.00673482, -0.00290081, -0.00243812, -0.00171484],
            [-0.00133542, 0.00631653, -0.00673482, -0.07027679, -
                0.11435863, -0.07027679, -0.00673482, 0.00631653, -0.00133542],
            [-0.00080639, 0.01261227, -0.00981051, -0.11435863,
                0.81380200, -0.11435863, -0.00981051, 0.01261227, -0.00080639],
            [-0.00133542, 0.00631653, -0.00673482, -0.07027679, -
                0.11435863, -0.07027679, -0.00673482, 0.00631653, -0.00133542],
            [-0.00171484, -0.00243812, -0.00290081, -0.00673482, -
                0.00981051, -0.00673482, -0.00290081, -0.00243812, -0.00171484],
            [-0.00113093, -0.00350017, -0.00243812, 0.00631653, 0.01261227,
                0.00631653, -0.00243812, -0.00350017, -0.00113093],
            [-0.00033429, -0.00113093, -0.00171484, -0.00133542, -0.00080639, -0.00133542, -0.00171484, -0.00113093, -0.00033429]]
        ).reshape(1, 1, 9, 9)
        filters["b"] = []
        filters["b"].append(torch.tensor([
            0.00277643, 0.00496194, 0.01026699, 0.01455399, 0.01026699, 0.00496194, 0.00277643,
            -0.00986904, -0.00893064, 0.01189859, 0.02755155, 0.01189859, -0.00893064, -0.00986904,
            -0.01021852, -0.03075356, -0.08226445, -
            0.11732297, -0.08226445, -0.03075356, -0.01021852,
            0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000,
            0.01021852, 0.03075356, 0.08226445, 0.11732297, 0.08226445, 0.03075356, 0.01021852,
            0.00986904, 0.00893064, -0.01189859, -
            0.02755155, -0.01189859, 0.00893064, 0.00986904,
            -0.00277643, -0.00496194, -0.01026699, -0.01455399, -0.01026699, -0.00496194, -0.00277643]
        ).reshape(1, 1, 7, 7).permute(0, 1, 3, 2))
        filters["b"].append(torch.tensor([
            -0.00343249, -0.00640815, -0.00073141, 0.01124321, 0.00182078, 0.00285723, 0.01166982,
            -0.00358461, -0.01977507, -0.04084211, -
            0.00228219, 0.03930573, 0.01161195, 0.00128000,
            0.01047717, 0.01486305, -0.04819057, -
            0.12227230, -0.05394139, 0.00853965, -0.00459034,
            0.00790407, 0.04435647, 0.09454202, -0.00000000, -
            0.09454202, -0.04435647, -0.00790407,
            0.00459034, -0.00853965, 0.05394139, 0.12227230, 0.04819057, -0.01486305, -0.01047717,
            -0.00128000, -0.01161195, -0.03930573, 0.00228219, 0.04084211, 0.01977507, 0.00358461,
            -0.01166982, -0.00285723, -0.00182078, -0.01124321, 0.00073141, 0.00640815, 0.00343249]
        ).reshape(1, 1, 7, 7).permute(0, 1, 3, 2))
        filters["b"].append(torch.tensor([
            0.00343249, 0.00358461, -0.01047717, -
            0.00790407, -0.00459034, 0.00128000, 0.01166982,
            0.00640815, 0.01977507, -0.01486305, -
            0.04435647, 0.00853965, 0.01161195, 0.00285723,
            0.00073141, 0.04084211, 0.04819057, -
            0.09454202, -0.05394139, 0.03930573, 0.00182078,
            -0.01124321, 0.00228219, 0.12227230, -
            0.00000000, -0.12227230, -0.00228219, 0.01124321,
            -0.00182078, -0.03930573, 0.05394139, 0.09454202, -
            0.04819057, -0.04084211, -0.00073141,
            -0.00285723, -0.01161195, -0.00853965, 0.04435647, 0.01486305, -0.01977507, -0.00640815,
            -0.01166982, -0.00128000, 0.00459034, 0.00790407, 0.01047717, -0.00358461, -0.00343249]
        ).reshape(1, 1, 7, 7).permute(0, 1, 3, 2))
        filters["b"].append(torch.tensor(
            [-0.00277643, 0.00986904, 0.01021852, -0.00000000, -0.01021852, -0.00986904, 0.00277643,
             -0.00496194, 0.00893064, 0.03075356, -
             0.00000000, -0.03075356, -0.00893064, 0.00496194,
             -0.01026699, -0.01189859, 0.08226445, -
             0.00000000, -0.08226445, 0.01189859, 0.01026699,
             -0.01455399, -0.02755155, 0.11732297, -
             0.00000000, -0.11732297, 0.02755155, 0.01455399,
             -0.01026699, -0.01189859, 0.08226445, -
             0.00000000, -0.08226445, 0.01189859, 0.01026699,
             -0.00496194, 0.00893064, 0.03075356, -
             0.00000000, -0.03075356, -0.00893064, 0.00496194,
             -0.00277643, 0.00986904, 0.01021852, -0.00000000, -0.01021852, -0.00986904, 0.00277643]
        ).reshape(1, 1, 7, 7).permute(0, 1, 3, 2))
        filters["b"].append(torch.tensor([
            -0.01166982, -0.00128000, 0.00459034, 0.00790407, 0.01047717, -0.00358461, -0.00343249,
            -0.00285723, -0.01161195, -0.00853965, 0.04435647, 0.01486305, -0.01977507, -0.00640815,
            -0.00182078, -0.03930573, 0.05394139, 0.09454202, -
            0.04819057, -0.04084211, -0.00073141,
            -0.01124321, 0.00228219, 0.12227230, -
            0.00000000, -0.12227230, -0.00228219, 0.01124321,
            0.00073141, 0.04084211, 0.04819057, -
            0.09454202, -0.05394139, 0.03930573, 0.00182078,
            0.00640815, 0.01977507, -0.01486305, -
            0.04435647, 0.00853965, 0.01161195, 0.00285723,
            0.00343249, 0.00358461, -0.01047717, -0.00790407, -0.00459034, 0.00128000, 0.01166982]
        ).reshape(1, 1, 7, 7).permute(0, 1, 3, 2))
        filters["b"].append(torch.tensor([
            -0.01166982, -0.00285723, -0.00182078, -
            0.01124321, 0.00073141, 0.00640815, 0.00343249,
            -0.00128000, -0.01161195, -0.03930573, 0.00228219, 0.04084211, 0.01977507, 0.00358461,
            0.00459034, -0.00853965, 0.05394139, 0.12227230, 0.04819057, -0.01486305, -0.01047717,
            0.00790407, 0.04435647, 0.09454202, -0.00000000, -
            0.09454202, -0.04435647, -0.00790407,
            0.01047717, 0.01486305, -0.04819057, -
            0.12227230, -0.05394139, 0.00853965, -0.00459034,
            -0.00358461, -0.01977507, -0.04084211, -
            0.00228219, 0.03930573, 0.01161195, 0.00128000,
            -0.00343249, -0.00640815, -0.00073141, 0.01124321, 0.00182078, 0.00285723, 0.01166982]
        ).reshape(1, 1, 7, 7).permute(0, 1, 3, 2))

    else:
        raise Exception(
            "Steerable filters not implemented for %d orientations" % n_orientations)

    if filter_type == "trained":
        if size == 5:
            # TODO maybe also train h0 and l0 filters
            filters = crop_steerable_pyramid_filters(filters, 5)
            filters["b"][0] = torch.tensor([
                [-0.0356752239, -0.0223877281, -0.0009542659,
                    0.0244821459, 0.0322226137],
                [-0.0593218654,  0.1245803162, -
                    0.0023863907, -0.1230178699, 0.0589442067],
                [-0.0281576272,  0.2976626456, -
                    0.0020888755, -0.2953369915, 0.0284542721],
                [-0.0586092323,  0.1251581162, -
                    0.0024624448, -0.1227868199, 0.0587830991],
                [-0.0327464789, -0.0223652460, -
                    0.0042342511,  0.0245472137, 0.0359398536]
            ]).reshape(1, 1, 5, 5)
            filters["b"][1] = torch.tensor([
                [3.9758663625e-02,  6.0679119080e-02,  3.0146904290e-02,
                    6.1198268086e-02,  3.6218870431e-02],
                [2.3255519569e-02, -1.2505133450e-01, -
                    2.9738345742e-01, -1.2518258393e-01,  2.3592948914e-02],
                [-1.3602430699e-03, -1.2058277935e-04,  2.6399988565e-04, -
                    2.3791544663e-04,  1.8450465286e-03],
                [-2.1563466638e-02,  1.2572696805e-01,  2.9745018482e-01,
                    1.2458638102e-01, -2.3847281933e-02],
                [-3.7941932678e-02, -6.1060950160e-02, -
                    2.9489086941e-02, -6.0411967337e-02, -3.8459088653e-02]
            ]).reshape(1, 1, 5, 5)

            # Below filters were optimised on 09/02/2021
            # 20K iterations with multiple images at more scales.
            filters["b"][0] = torch.tensor([
                [-4.5508436859e-02, -2.1767273545e-02, -1.9399923622e-04,
                    2.1200872958e-02,  4.5475799590e-02],
                [-6.3554823399e-02,  1.2832683325e-01, -
                    5.3858719184e-05, -1.2809979916e-01,  6.3842624426e-02],
                [-3.4809380770e-02,  2.9954621196e-01,  2.9066693969e-05, -
                    2.9957753420e-01,  3.4806568176e-02],
                [-6.3934154809e-02,  1.2806062400e-01,  9.0917674243e-05, -
                    1.2832444906e-01,  6.3572973013e-02],
                [-4.5492250472e-02, -2.1125273779e-02,  4.2229349492e-04,
                    2.1804777905e-02,  4.5236673206e-02]
            ]).reshape(1, 1, 5, 5)
            filters["b"][1] = torch.tensor([
                [4.8947390169e-02,  6.3575074077e-02,  3.4955859184e-02,
                    6.4085893333e-02,  4.9838040024e-02],
                [2.2061849013e-02, -1.2936264277e-01, -
                    3.0093491077e-01, -1.2997294962e-01,  2.0597217605e-02],
                [-5.1290717238e-05, -1.7305796064e-05,  2.0256420612e-05, -
                    1.1864109547e-04,  7.3973249528e-05],
                [-2.0749464631e-02,  1.2988376617e-01,  3.0080935359e-01,
                    1.2921217084e-01, -2.2159902379e-02],
                [-4.9614857882e-02, -6.4021714032e-02, -
                    3.4676689655e-02, -6.3446544111e-02, -4.8282280564e-02]
            ]).reshape(1, 1, 5, 5)

            # Trained on 17/02/2021 to match fourier pyramid in spatial domain
            filters["b"][0] = torch.tensor([
                [3.3370e-02,  9.3934e-02, -3.5810e-04, -9.4038e-02, -3.3115e-02],
                [1.7716e-01,  3.9378e-01,  6.8461e-05, -3.9343e-01, -1.7685e-01],
                [2.9213e-01,  6.1042e-01,  7.0654e-04, -6.0939e-01, -2.9177e-01],
                [1.7684e-01,  3.9392e-01,  1.0517e-03, -3.9268e-01, -1.7668e-01],
                [3.3000e-02,  9.4029e-02,  7.3565e-04, -9.3366e-02, -3.3008e-02]
            ]).reshape(1, 1, 5, 5) * 0.1

            filters["b"][1] = torch.tensor([
                [0.0331,  0.1763,  0.2907,  0.1753,  0.0325],
                [0.0941,  0.3932,  0.6079,  0.3904,  0.0922],
                [0.0008,  0.0009, -0.0010, -0.0025, -0.0015],
                [-0.0929, -0.3919, -0.6097, -0.3944, -0.0946],
                [-0.0328, -0.1760, -0.2915, -0.1768, -0.0333]
            ]).reshape(1, 1, 5, 5) * 0.1

        else:
            raise Exception(
                "Trained filters not implemented for size %d" % size)

    if filter_type == "cropped":
        filters = crop_steerable_pyramid_filters(filters, size)

    return filters
