# Copyright 2016 Osvaldo Santana Neto
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


from decimal import Decimal
from typing import Dict, Tuple  # noqa

from ..utils import RangeSet

TRACKING_PREFIX = {
    "AL": "Agentes de leitura",
    "AR": "Avisos de recebimento",
    "AS": "PAC - Ação Social",
    "CA": "Encomenda Internacional - Colis",
    "CB": "Encomenda Internacional - Colis",
    "CC": "Encomenda Internacional - Colis",
    "CD": "Encomenda Internacional - Colis",
    "CE": "Encomenda Internacional - Colis",
    "CF": "Encomenda Internacional - Colis",
    "CG": "Encomenda Internacional - Colis",
    "CH": "Encomenda Internacional - Colis",
    "CI": "Encomenda Internacional - Colis",
    "CJ": "Encomenda Internacional - Colis",
    "CK": "Encomenda Internacional - Colis",
    "CL": "Encomenda Internacional - Colis",
    "CM": "Encomenda Internacional - Colis",
    "CN": "Encomenda Internacional - Colis",
    "CO": "Encomenda Internacional - Colis",
    "CP": "Encomenda Internacional - Colis",
    "CQ": "Encomenda Internacional - Colis",
    "CR": "Carta registrada sem Valor Declarado",
    "CS": "Encomenda Internacional - Colis",
    "CT": "Encomenda Internacional - Colis",
    "CU": "Encomenda internacional - Colis",
    "CV": "Encomenda Internacional - Colis",
    "CW": "Encomenda Internacional - Colis",
    "CX": "Encomenda internacional - Colis ou Selo Lacre para Caixetas",
    "CY": "Encomenda Internacional - Colis",
    "CZ": "Encomenda Internacional - Colis",
    "DA": "SEDEX ou Remessa Expressa com AR Digital",
    "DB": "SEDEX ou Remessa Expressa com AR Digital (Bradesco)",
    "DC": "Remessa Expressa CRLV/CRV/CNH e Notificações",
    "DD": "Devolução de documentos",
    "DE": "Remessa Expressa Talão/Cartão com AR",
    "DF": "e-SEDEX",
    "DG": "SEDEX",
    "DI": "SEDEX ou Remessa Expressa com AR Digital (Itaú)",
    "DJ": "SEDEX",
    "DK": "PAC Extra Grande",
    "DL": "SEDEX",
    "DM": "e-SEDEX",
    "DN": "SEDEX",
    "DO": "SEDEX ou Remessa Expressa com AR Digital (Itaú)",
    "DP": "SEDEX Pagamento na Entrega",
    "DQ": "SEDEX ou Remessa Expressa com AR Digital (Santander)",
    "DR": "Remessa Expressa com AR Digital (Santander)",
    "DS": "SEDEX ou Remessa Expressa com AR Digital (Santander)",
    "DT": "Remessa econômica com AR Digital (DETRAN)",
    "DU": "e-SEDEX",
    "DV": "SEDEX c/ AR digital",
    "DW": "Encomenda SEDEX (Etiqueta Lógica)",
    "DX": "SEDEX 10",
    "EA": "Encomenda Internacional - EMS",
    "EB": "Encomenda Internacional - EMS",
    "EC": "PAC",
    "ED": "Packet Express",
    "EE": "Encomenda Internacional - EMS",
    "EF": "Encomenda Internacional - EMS",
    "EG": "Encomenda Internacional - EMS",
    "EH": "Encomenda Internacional - EMS ou Encomenda com AR Digital",
    "EI": "Encomenda Internacional - EMS",
    "EJ": "Encomenda Internacional - EMS",
    "EK": "Encomenda Internacional - EMS",
    "EL": "Encomenda Internacional - EMS",
    "EM": "Encomenda Internacional - SEDEX Mundi",  # BR Suffix
    # "EM": "Encomenda Internacional - EMS Importação",
    "EN": "Encomenda Internacional - EMS",
    "EO": "Encomenda Internacional - EMS",
    "EP": "Encomenda Internacional - EMS",
    "EQ": "Encomenda de serviço não expressa (ECT)",
    "ER": "Objeto registrado",
    "ES": "e-SEDEX ou EMS",
    "ET": "Encomenda Internacional - EMS",
    "EU": "Encomenda Internacional - EMS",
    "EV": "Encomenda Internacional - EMS",
    "EW": "Encomenda Internacional - EMS",
    "EX": "Encomenda Internacional - EMS",
    "EY": "Encomenda Internacional - EMS",
    "EZ": "Encomenda Internacional - EMS",
    "FA": "FAC registrado",
    "FE": "Encomenda FNDE",
    "FF": "Objeto registrado (DETRAN)",
    "FH": "FAC registrado com AR Digital",
    "FM": "FAC monitorado",
    "FR": "FAC registrado",
    "IA": "Logística Integrada (agendado / avulso)",
    "IC": "Logística Integrada (a cobrar)",
    "ID": "Logística Integrada (devolução de documento)",
    "IE": "Logística Integrada (Especial)",
    "IF": "CPF",
    "II": "Logística Integrada (ECT)",
    "IK": "Logística Integrada com Coleta Simultânea",
    "IM": "Logística Integrada (Medicamentos)",
    "IN": "Correspondência e EMS recebido do Exterior",
    "IP": "Logística Integrada (Programada)",
    "IR": "Impresso Registrado",
    "IS": "Logística integrada standard (medicamentos)",
    "IT": "Remessa Expressa Medicamentos / Logística Integrada Termolábil",
    "IU": "Logística Integrada (urgente)",
    "IX": "EDEI Expresso",
    "JA": "Remessa econômica com AR Digital",
    "JB": "Remessa econômica com AR Digital",
    "JC": "Remessa econômica com AR Digital",
    "JD": "Remessa econômica Talão/Cartão",
    "JE": "Remessa econômica com AR Digital",
    "JF": "Remessa econômica com AR Digital",
    "JG": "Objeto registrado urgente/prioritário",
    "JH": "Objeto registrado urgente / prioritário",
    "JI": "Remessa econômica Talão/Cartão",
    "JJ": "Objeto registrado (Justiça)",
    "JK": "Remessa econômica Talão/Cartão",
    "JL": "Objeto registrado",
    "JM": "Mala Direta Postal Especial",
    "JN": "Objeto registrado econômico",
    "JO": "Objeto registrado urgente",
    "JP": "Receita Federal",
    "JQ": "Remessa econômica com AR Digital",
    "JR": "Objeto registrado urgente / prioritário",
    "JS": "Objeto registrado",
    "JT": "Objeto Registrado Urgente",
    "JV": "Remessa Econômica (c/ AR DIGITAL)",
    "LA": "SEDEX com Logística Reversa Simultânea em Agência",
    "LB": "e-SEDEX com Logística Reversa Simultânea em Agência",
    "LC": "Objeto Internacional (Prime)",
    "LE": "Logística Reversa Econômica",
    "LF": "Objeto Internacional (Prime)",
    "LI": "Objeto Internacional (Prime)",
    "LJ": "Objeto Internacional (Prime)",
    "LK": "Objeto Internacional (Prime)",
    "LM": "Objeto Internacional (Prime)",
    "LN": "Objeto Internacional (Prime)",
    "LP": "PAC com Logística Reversa Simultânea em Agência",
    "LS": "SEDEX Logística Reversa",
    "LV": "Logística Reversa Expressa",
    "LX": "Packet Standard / Econômica",
    "LZ": "Objeto Internacional (Prime)",
    "MA": "Serviços adicionais do Telegrama",
    "MB": "Telegrama (balcão)",
    "MC": "Telegrama (Fonado)",
    "MD": "SEDEX Mundi (Documento interno)",
    "ME": "Telegrama",
    "MF": "Telegrama (Fonado)",
    "MK": "Telegrama (corporativo)",
    "ML": "Fecha Malas (Rabicho)",
    "MM": "Telegrama (Grandes clientes)",
    "MP": "Telegrama (Pré-pago)",
    "MR": "AR digital",
    "MS": "Encomenda Saúde",
    "MT": "Telegrama (Telemail)",
    "MY": "Telegrama internacional (entrante)",
    "MZ": "Telegrama (Correios Online)",
    "NE": "Tele Sena resgatada",
    "NX": "EDEI Econômico (não urgente)",
    "PA": "Passaporte",
    "PB": "PAC",
    "PC": "PAC a Cobrar",
    "PD": "PAC",
    "PE": "PAC",
    "PF": "Passaporte",
    "PG": "PAC",
    "PH": "PAC",
    "PI": "PAC",
    "PJ": "PAC",
    "PK": "PAC Extra Grande",
    "PL": "PAC",
    "PN": "PAC Normal",
    "PR": "Reembolso Postal",
    "QQ": "Objeto de teste (SIGEP Web)",
    "RA": "Objeto registrado / prioritário",
    "RB": "Carta registrada",
    "RC": "Carta registrada com Valor Declarado",
    "RD": "Remessa econômica ou objeto registrado (DETRAN)",
    "RE": "Objeto registrado econômico",
    "RF": "Receita Federal",
    "RG": "Objeto registrado",
    "RH": "Objeto registrado com AR Digital",
    "RI": "Objeto registrado internacional prioritário",
    "RJ": "Objeto registrado",
    "RK": "Objeto registrado",
    "RL": "Objeto registrado",
    "RM": "Objeto registrado urgente",
    "RN": "Objeto registrado (SIGEPWEB ou Agência)",
    "RO": "Objeto registrado",
    "RP": "Reembolso Postal",
    "RQ": "Objeto registrado",
    "RR": "Objeto registrado",
    "RS": "Objeto registrado",
    "RT": "Remessa econômica Talão/Cartão",
    "RU": "Objeto registrado (ECT)",
    "RV": "Remessa econômica CRLV/CRV/CNH e Notificações com AR Digital",
    "RW": "Objeto internacional",
    "RX": "Objeto internacional",
    "RY": "Remessa econômica Talão/Cartão com AR Digital",
    "RZ": "Objeto registrado",
    "SA": "SEDEX",
    "SB": "SEDEX 10",
    "SC": "SEDEX a cobrar",
    "SD": "SEDEX ou Remessa Expressa (DETRAN)",
    "SE": "SEDEX",
    "SF": "SEDEX",
    "SG": "SEDEX",
    "SH": "SEDEX com AR Digital / SEDEX ou AR Digital",
    "SI": "SEDEX",
    "SJ": "SEDEX Hoje",
    "SK": "SEDEX",
    "SL": "SEDEX",
    "SM": "SEDEX 12",
    "SN": "SEDEX",
    "SO": "SEDEX",
    "SP": "SEDEX Pré-franqueado",
    "SQ": "SEDEX",
    "SR": "SEDEX",
    "SS": "SEDEX",
    "ST": "Remessa Expressa Talão/Cartão",
    "SU": "Encomenda de serviço expressa (ECT)",
    "SV": "Remessa Expressa CRLV/CRV/CNH e Notificações com AR Digital",
    "SW": "e-SEDEX",
    "SX": "SEDEX 10",
    "SY": "Remessa Expressa Talão/Cartão com AR Digital",
    "SZ": "SEDEX",
    "TC": "Objeto para treinamento",
    "TE": "Objeto para treinamento",
    "TS": "Objeto para treinamento",
    "VA": "Encomendas com valor declarado",
    "VC": "Encomendas",
    "VD": "Encomendas com valor declarado",
    "VE": "Encomendas",
    "VF": "Encomendas com valor declarado",
    "VV": "Objeto internacional",
    "XA": "Aviso de chegada (internacional)",
    "XM": "SEDEX Mundi",
    "XR": "Encomenda SUR Postal Expresso",
    "XX": "Encomenda SUR Postal 24 horas",
}

EXTRA_SERVICES = {
    1: {'code': "AR", 'name': "Aviso de Recebimento", 'display_on_label': True},
    2: {'code': "MP", 'name': "Mão Própria Nacional", 'display_on_label': True},
    19: {'code': "VD", 'name': "Valor Declarado (Encomendas)", 'display_on_label': True},
    25: {'code': "RR", 'name': "Registro Nacional", 'display_on_label': False},
}

EXTRA_SERVICE_AR = 1
EXTRA_SERVICE_MP = 2
EXTRA_SERVICE_VD = 19
EXTRA_SERVICE_RR = 25

SERVICES = {
    '40215': {
        'id': 104707,
        'description': 'SEDEX 10',
        'category': 'SERVICO_COM_RESTRICAO',
        'max_weight': 10000,
        'display_name': 'SEDEX 10',
        'symbol': "premium",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '81019': {
        'id': 104672,
        'description': 'E-SEDEX STANDARD',
        'category': 'SERVICO_COM_RESTRICAO',
        'max_weight': 15000,
        'display_name': 'E-SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '41068': {
        'id': 109819,
        'description': 'PAC CONTRATO AGENCIA',
        'category': 'PAC',
        'display_name': 'PAC',
        'max_weight': 30000,
        'symbol': "standard",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("3000.00"),
    },
    '04669': {
        'id': 124884,
        'description': 'PAC',
        'category': 'PAC',
        'display_name': 'PAC',
        'max_weight': 30000,
        'symbol': "standard",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("3000.00"),
    },
    '40444': {
        'id': 109811,
        'description': 'SEDEX - CONTRATO',
        'category': 'SEDEX',
        'max_weight': 30000,
        'display_name': 'SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '40436': {
        'id': 109810,
        'description': 'SEDEX - CONTRATO',
        'category': 'SEDEX',
        'max_weight': 30000,
        'display_name': 'SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '40096': {
        'id': 104625,
        'description': 'SEDEX (CONTRATO)',
        'category': 'SEDEX',
        'max_weight': 30000,
        'display_name': 'SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '04162': {
        'id': 124849,
        'description': 'SEDEX CONTRATO AGENCIA',
        'category': 'SEDEX',
        'max_weight': 30000,
        'display_name': 'SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '40380': {
        'id': 109806,
        'description': 'SEDEX REVERSO 40096',
        'category': 'REVERSO',
        'max_weight': 30000,
        'display_name': 'SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '40010': {
        'id': 104295,
        'description': 'SEDEX A VISTA',
        'category': 'SEDEX',
        'max_weight': 30000,
        'display_name': 'SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '41211': {
        'id': 113546,
        'description': 'PAC - CONTRATO',
        'category': 'PAC',
        'display_name': 'PAC',
        'max_weight': 30000,
        'symbol': "standard",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("3000.00"),
    },
    '40630': {
        'id': 114976,
        'description': 'SEDEX PAGAMENTO NA ENTREGA -',
        'category': 'SEDEX',
        'max_weight': 30000,
        'display_name': 'SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '40916': {
        'id': 118568,
        'description': 'SEDEX AGRUPADO II',
        'category': 'SEDEX',
        'max_weight': 30000,
        'display_name': 'SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '40908': {
        'id': 118567,
        'description': 'SEDEX AGRUPADO I',
        'category': 'SEDEX',
        'max_weight': 30000,
        'display_name': 'SEDEX',
        'symbol': "express",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '41300': {
        'id': 120366,
        'description': 'PAC GRANDES FORMATOS',
        'category': 'SERVICO_COM_RESTRICAO',
        'max_weight': 50000,
        'display_name': 'PAC',
        'symbol': "standard",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("3000.00"),
    },
    '40169': {
        'id': 115218,
        'description': 'SEDEX 12',
        'category': 'SERVICO_COM_RESTRICAO',
        'max_weight': 10000,
        'display_name': 'SEDEX 12',
        'symbol': "premium",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '40290': {
        'id': 108934,
        'description': 'SEDEX HOJE',
        'category': 'SERVICO_COM_RESTRICAO',
        'max_weight': 10000,
        'display_name': 'SEDEX Hoje',
        'symbol': "premium",
        'default_extra_services': [EXTRA_SERVICE_RR],
        'min_declared_value': Decimal("18.50"),
        'max_declared_value': Decimal("10000.00"),
    },
    '10154': {
        'id': 118424,
        'description': 'CARTA COMERCIAL  REGISTRADA',
        'category': 'CARTA_REGISTRADA',
        'display_name': 'Carta Registrada',
    },
    '41246': {
        'id': 115487,
        'description': 'REM. CAMPANHA PAPAI NOEL DOS',
        'category': 'SEM_CATEGORIA',
        'display_name': 'Papai Noel dos Correios'},
    '40150': {
        'id': 115136,
        'description': 'SERVICO DE PROTOCOLO POSTAL -',
        'category': 'SEDEX',
        'display_name': 'Protocolo',
    },
    '10065': {
        'id': 109480,
        'description': 'CARTA COMERCIAL A FATURAR',
        'category': 'CARTA_REGISTRADA',
        'display_name': 'Carta Comercial',
    },
}  # type: Dict[str, dict]

SERVICE_PAC = '04669'
SERVICE_SEDEX = '04162'
SERVICE_SEDEX10 = '40215'
SERVICE_SEDEX12 = '40169'
SERVICE_E_SEDEX = '81019'

INSURANCE_VALUE_THRESHOLD_PAC = Decimal("50.00")  # R$
INSURANCE_VALUE_THRESHOLD_SEDEX = Decimal("75.00")  # R$
INSURANCE_PERCENTUAL_COST = Decimal("0.007")  # 0.7%

REGIONAL_DIRECTIONS = {
    1: {'code': "AC", 'name': "AC - ADMINISTRAÇAO CENTRAL"},
    3: {'code': "ACR", 'name': "DR - ACRE"},
    4: {'code': "AL", 'name': "DR - ALAGOAS"},
    6: {'code': "AM", 'name': "DR - AMAZONAS"},
    5: {'code': "AP", 'name': "DR - AMAPÁ"},
    8: {'code': "BA", 'name': "DR - BAHIA"},
    10: {'code': "BSB", 'name': "DR - BRASÍLIA"},
    12: {'code': "CE", 'name': "DR - CEARÁ"},
    14: {'code': "ES", 'name': "DR - ESPIRITO SANTO"},
    16: {'code': "GO", 'name': "DR - GOIÁS"},
    18: {'code': "MA", 'name': "DR - MARANHÃO"},
    20: {'code': "MG", 'name': "DR - MINAS GERAIS"},
    22: {'code': "MS", 'name': "DR - MATO GROSSO DO SUL"},
    24: {'code': "MT", 'name': "DR - MATO GROSSO"},
    28: {'code': "PA", 'name': "DR - PARÁ"},
    30: {'code': "PB", 'name': "DR - PARAÍBA"},
    32: {'code': "PE", 'name': "DR - PERNAMBUCO"},
    34: {'code': "PI", 'name': "DR - PIAUÍ"},
    36: {'code': "PR", 'name': "DR - PARANÁ"},
    50: {'code': "RJ", 'name': "DR - RIO DE JANEIRO"},
    60: {'code': "RN", 'name': "DR - RIO GRANDE DO NORTE"},
    26: {'code': "RO", 'name': "DR - RONDONIA"},
    65: {'code': "RR", 'name': "DR - RORAIMA"},
    64: {'code': "RS", 'name': "DR - RIO GRANDE DO SUL"},
    68: {'code': "SC", 'name': "DR - SANTA CATARINA"},
    70: {'code': "SE", 'name': "DR - SERGIPE"},
    74: {'code': "SPI", 'name': "DR - SÃO PAULO INTERIOR"},
    72: {'code': "SPM", 'name': "DR - SÃO PAULO"},
    75: {'code': "TO", 'name': "DR - TOCANTINS"},
}

TRACKING_EVENT_TYPES = {
    "ERROR": "Evento de erro",  # custom event type for "Not Found" error
    "BDE": "Baixa de distribuição externa",
    "BDI": "Baixa de distribuição interna",
    "BDR": "Baixa corretiva",
    "BLQ": "Bloqueio de objetos",
    # "CAR": "Conferência de lista de registro",  # usage not found in SRO documentation
    "CD": "Conferência de nota de despacho",
    "CMT": "Chegada de um meio de transporte",
    "CO": "Coleta de objetos",
    "CUN": "Conferência de lista de registro",
    "DO": "Expedição de nota de despacho",
    "EST": "Estorno",
    "FC": "Função complementar",
    "IDC": "Indenização de objetos",
    "LDI": "Lista de distribuição interna",
    "LDE": "Lista de distribuição externa",
    "OEC": "Lista de Objetos Entregues ao Carteiro",
    "PAR": "Conferência Unidade Internacional",
    "PMT": "Partida Meio de Transporte",
    "PO": "Postagem (exceção)",
    "RO": "Expedição de Lista de Registro",
    "TRI": "Triagem",
    # "CMR": "Evento Desconhecido",  # usage not found in SRO documentation
}

TRACKING_STATUS = {
    ('ERROR', 0): ('error', 'Objeto não encontrado', '', ''),  # custom status for "Not Found" error
    ('BDE', 0): (
        'delivered',
        'Objeto entregue ao destinatário',
        'Recebido por:',
        'Finalizar a entrega. Não é mais necessário prosseguir com o acompanhamento.',
    ),
    ('BDI', 0): (
        'delivered',
        'Objeto entregue ao destinatário',
        'Recebido por:',
        'Finalizar a entrega. Não é mais necessário prosseguir com o acompanhamento.',
    ),
    ('BDR', 0): (
        'delivered',
        'Objeto entregue ao destinatário',
        'Recebido por:',
        'Finalizar a entrega. Não é mais necessário prosseguir com o acompanhamento.',
    ),
    ('CD', 0): (
        'shipped',
        'Objeto recebido na Unidade dos Correios',
        '',
        'Acompanhar',
    ),
    ('CMT', 0): (
        'shipped',
        'Objeto recebido na Unidade dos Correios',
        '',
        'Acompanhar',
    ),
    ('CUN', 0): (
        'shipped',
        'Objeto recebido na Unidade dos Correios',
        '',
        'Acompanhar',
    ),
    ('DO', 0): (
        'shipped',
        'Objeto encaminhado para',
        '<nome da cidade>',
        'Acompanhar',
    ),
    ('LDE', 0): (
        'shipped',
        'Objeto saiu para entrega ao remetente',
        '',
        'Acompanhar',
    ),
    ('LDI', 0): (
        'waiting_retrieval',
        'Objeto aguardando retirada no endereço indicado',
        'Endereço:',
        'Acompanhar. O interessado deverá buscar o objeto em uma Unidade dos Correios.',
    ),
    ('OEC', 0): (
        'shipped',
        'Objeto saiu para entrega ao destinatário',
        '',
        'Acompanhar. O interessado deverá buscar o objeto em uma Unidade dos Correios.',
    ),
    ('PO', 0): (
        'shipped',
        'Objeto postado',
        '',
        'Acompanhar',
    ),
    ('RO', 0): (
        'shipped',
        'Objeto encaminhado para',
        '<nome da cidade>',
        'Acompanhar',
    ),
    ('TRI', 0): (
        'shipped',
        'Objeto encaminhado para',
        '<nome da cidade>',
        'Acompanhar',
    ),
    ('BDE', 1): (
        'delivered',
        'Objeto entregue ao destinatário',
        'Recebido por:',
        'Finalizar a entrega. Não é mais necessário prosseguir com o acompanhamento.',
    ),
    ('BDI', 1): (
        'delivered',
        'Objeto entregue ao destinatário',
        'Recebido por:',
        'Finalizar a entrega. Não é mais necessário prosseguir com o acompanhamento.',
    ),
    ('BDR', 1): (
        'delivered',
        'Objeto entregue ao destinatário',
        'Recebido por:',
        'Finalizar a entrega. Não é mais necessário prosseguir com o acompanhamento.',
    ),
    ('BLQ', 1): (
        'shipped',
        'Entrega de objeto bloqueada a pedido do remetente',
        'Objeto em análise de destinação',
        'Acompanhar',
    ),
    ('CD', 1): (
        'shipped',
        'Objeto recebido na Unidade dos Correios',
        '',
        'Acompanhar',
    ),
    ('CO', 1): (
        'shipped',
        'Objeto coletado',
        '',
        'Acompanhar',
    ),
    ('CUN', 1): (
        'shipped',
        'Objeto recebido na Unidade dos Correios',
        '',
        'Acompanhar',
    ),
    ('DO', 1): (
        'shipped',
        'Objeto encaminhado para',
        '<nome da cidade>',
        'Acompanhar',
    ),
    ('EST', 1): (
        'ignore_last_entry',
        'Favor desconsiderar a informação anterior',
        '',
        'Acompanhar',
    ),
    ('FC', 1): (
        'shipped',
        'Objeto será devolvido por solicitação do remetente',
        '',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('IDC', 1): (
        'lost',
        'Objeto não localizado',
        'Houve indenização dos valores correspondentes',
        'Acompanhar',
    ),
    ('LDI', 1): (
        'waiting_retrieval',
        'Objeto aguardando retirada no endereço indicado',
        'Endereço:',
        'Acompanhar. O interessado deverá buscar o objeto em uma Unidade dos Correios.',
    ),
    ('OEC', 1): (
        'shipped',
        'Objeto saiu para entrega ao destinatário',
        '',
        'Acompanhar',
    ),
    ('PMT', 1): (
        'shipped',
        'Objeto encaminhado para',
        '<nome da cidade>',
        'Acompanhar',
    ),
    ('PO', 1): (
        'shipped',
        'Objeto postado',
        '',
        'Acompanhar',
    ),
    ('RO', 1): (
        'shipped',
        'Objeto encaminhado para',
        '<nome da cidade>',
        'Acompanhar',
    ),
    ('BDE', 2): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Carteiro não atendido',
        'Aguarde: Objeto estará disponível para retirada na unidade a ser informada.',
        'Acompanhar. O interessado deverá buscar o objeto em uma Unidade dos Correios.',
    ),
    ('BDI', 2): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Carteiro não atendido',
        'Aguarde: Objeto estará disponível para retirada na unidade a ser informada.',
        'Acompanhar. O interessado deverá buscar o objeto em uma Unidade dos Correios.',
    ),
    ('BDR', 2): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Carteiro não atendido',
        'Aguarde: Objeto estará disponível para retirada na unidade a ser informada.',
        'Acompanhar. O interessado deverá buscar o objeto em uma Unidade dos Correios.',
    ),
    ('CD', 2): (
        'shipped',
        'Objeto recebido na Unidade dos Correios',
        '',
        'Acompanhar',
    ),
    ('DO', 2): (
        'shipped',
        'Objeto encaminhado para',
        '<nome da cidade>',
        'Acompanhar',
    ),
    ('EST', 2): (
        'ignore_last_entry',
        'Favor desconsiderar a informação anterior',
        '',
        'Acompanhar',
    ),
    ('FC', 2): (
        'shipped',
        'Objeto com data de entrega agendada',
        '',
        'Acompanhar',
    ),
    ('IDC', 2): (
        'lost',
        'Objeto não localizado',
        'Houve indenização dos valores correspondentes',
        'Acompanhar',
    ),
    ('LDI', 2): (
        'waiting_retrieval',
        'Objeto disponível para retirada em Caixa Postal',
        '',
        'Acompanhar. O interessado deverá buscar o objeto em uma Unidade dos Correios.',
    ),
    ('BDE', 3): (
        'waiting_retrieval',
        'Remetente não retirou objeto na Unidade dos Correios',
        'Objeto em análise de destinação',
        'Acompanhar. O interessado não buscou o objeto na unidade dos Correios durante o período de guarda.',
    ),
    ('BDI', 3): (
        'waiting_retrieval',
        'Remetente não retirou objeto na Unidade dos Correios',
        'Objeto em análise de destinação',
        'Acompanhar. O interessado não buscou o objeto na unidade dos Correios durante o período de guarda.',
    ),
    ('BDR', 3): (
        'waiting_retrieval',
        'Remetente não retirou objeto na Unidade dos Correios',
        'Objeto em análise de destinação',
        'Acompanhar. O interessado não buscou o objeto na unidade dos Correios durante o período de guarda.',
    ),
    ('CD', 3): (
        'shipped',
        'Objeto recebido na Unidade dos Correios',
        '',
        'Acompanhar',
    ),
    ('EST', 3): (
        'ignore_last_entry',
        'Favor desconsiderar a informação anterior',
        '',
        'Acompanhar',
    ),
    ('FC', 3): (
        'shipped',
        'Objeto mal encaminhado',
        'Encaminhamento a ser corrigido.',
        'Acompanhar',
    ),
    ('IDC', 3): (
        'lost',
        'Objeto não localizado',
        'Houve indenização dos valores correspondentes',
        'Acompanhar',
    ),
    ('LDI', 3): (
        'waiting_retrieval',
        'Objeto aguardando retirada no endereço indicado',
        'Endereço:',
        'Acompanhar. O interessado deverá buscar o objeto em uma Unidade dos Correios.',
    ),
    ('BDE', 4): (
        'delivery_rejected',
        'A entrega não pode ser efetuada - Cliente recusou-se a receber',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 4): (
        'delivery_rejected',
        'A entrega não pode ser efetuada - Cliente recusou-se a receber',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 4): (
        'delivery_rejected',
        'A entrega não pode ser efetuada - Cliente recusou-se a receber',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('EST', 4): (
        'ignore_last_entry',
        'Favor desconsiderar a informação anterior',
        '',
        'Acompanhar',
    ),
    ('FC', 4): (
        'shipped',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar',
    ),
    ('IDC', 4): (
        'lost',
        'Objeto não localizado',
        'Houve indenização dos valores correspondentes',
        'Acompanhar',
    ),
    ('BDE', 5): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 5): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 5): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('EST', 5): (
        'ignore_last_entry',
        'Favor desconsiderar a informação anterior',
        '',
        'Acompanhar',
    ),
    ('FC', 5): (
        'shipped',
        'Objeto devolvido aos Correios',
        '',
        'Acompanhar',
    ),
    ('IDC', 5): (
        'lost',
        'Objeto não localizado',
        'Houve indenização dos valores correspondentes',
        'Acompanhar',
    ),
    ('BDE', 6): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Cliente desconhecido no local',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 6): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Cliente desconhecido no local',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 6): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Cliente desconhecido no local',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('EST', 6): (
        'ignore_last_entry',
        'Favor desconsiderar a informação anterior',
        '',
        'Acompanhar',
    ),
    ('IDC', 6): (
        'lost',
        'Objeto não localizado',
        'Houve indenização dos valores correspondentes',
        'Acompanhar',
    ),
    ('BDE', 7): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar',
    ),
    ('BDI', 7): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar',
    ),
    ('BDR', 7): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar',
    ),
    ('FC', 7): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Empresa sem expediente',
        'A entrega deverá ocorrer no próximo dia útil',
        'Acompanhar',
    ),
    ('IDC', 7): (
        'lost',
        'Objeto não localizado',
        'Houve indenização dos valores correspondentes',
        'Acompanhar',
    ),
    ('BDE', 8): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 8): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 8): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 9): (
        'lost',
        'Objeto não localizado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDI', 9): (
        'lost',
        'Objeto não localizado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDR', 9): (
        'lost',
        'Objeto não localizado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('EST', 9): (
        'ignore_last_entry',
        'Favor desconsiderar a informação anterior',
        '',
        'Acompanhar',
    ),
    ('LDE', 9): (
        'shipped',
        'Objeto saiu para entrega ao remetente',
        '',
        'Acompanhar',
    ),
    ('OEC', 9): (
        'shipped',
        'Objeto saiu para entrega ao remetente',
        '',
        'Acompanhar',
    ),
    ('PO', 9): (
        'shipped',
        'Objeto postado após o horário limite da agência',
        'Objeto sujeito a encaminhamento no próximo dia útil',
        'Acompanhar',
    ),
    ('BDE', 10): (
        'shipped_wrong_address',
        'A entrega não pode ser efetuada - Cliente mudou-se',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 10): (
        'shipped_wrong_address',
        'A entrega não pode ser efetuada - Cliente mudou-se',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 10): (
        'shipped_wrong_address',
        'A entrega não pode ser efetuada - Cliente mudou-se',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 12): (
        'waiting_retrieval',
        'Remetente não retirou objeto na Unidade dos Correios',
        'Objeto em análise de destinação',
        'Acionar atendimento dos Correios.',
    ),
    ('BDI', 12): (
        'waiting_retrieval',
        'Remetente não retirou objeto na Unidade dos Correios',
        'Objeto em análise de destinação',
        'Acionar atendimento dos Correios.',
    ),
    ('BDR', 12): (
        'waiting_retrieval',
        'Remetente não retirou objeto na Unidade dos Correios',
        'Objeto em análise de destinação',
        'Acionar atendimento dos Correios.',
    ),
    ('LDI', 14): (
        'waiting_retrieval',
        'Objeto aguardando retirada no endereço indicado',
        'Endereço:',
        'Acompanhar. O interessado deverá buscar o objeto em uma Unidade dos Correios.',
    ),
    ('PAR', 15): (
        'delivered',
        'Objeto recebido em <destino>',
        '',
        'Acompanhar',
    ),
    ('PAR', 16): (
        'customs_control',
        'Objeto recebido no Brasil',
        'Objeto sujeito à fiscalização e atraso na entrega',
        'Acompanhar',
    ),
    ('PAR', 17): (
        'customs_control',
        'Objeto liberado pela alfândega',
        '',
        'Acompanhar',
    ),
    ('PAR', 18): (
        'shipped',
        'Objeto recebido na unidade de exportação',
        '',
        'Acompanhar',
    ),
    ('BDE', 19): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 19): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 19): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Endereço incorreto',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 20): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Carteiro não atendido',
        'Será realizada nova tentativa de entrega',
        'Acompanhar',
    ),
    ('BDI', 20): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Carteiro não atendido',
        'Será realizada nova tentativa de entrega',
        'Acompanhar',
    ),
    ('BDR', 20): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Carteiro não atendido',
        'Será realizada nova tentativa de entrega',
        'Acompanhar',
    ),
    ('BDE', 21): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Carteiro não atendido',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 21): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Carteiro não atendido',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 21): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Carteiro não atendido',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 22): (
        'shipped',
        'Objeto devolvido aos Correios',
        '',
        'Acompanhar',
    ),
    ('BDI', 22): (
        'shipped',
        'Objeto devolvido aos Correios',
        '',
        'Acompanhar',
    ),
    ('BDR', 22): (
        'shipped',
        'Objeto devolvido aos Correios',
        '',
        'Acompanhar',
    ),
    ('BDE', 23): (
        'returned',
        'Objeto devolvido ao remetente',
        'Recebido por:',
        'Acompanhar',
    ),
    ('BDI', 23): (
        'returned',
        'Objeto devolvido ao remetente',
        'Recebido por:',
        'Acompanhar',
    ),
    ('BDR', 23): (
        'returned',
        'Objeto devolvido ao remetente',
        'Recebido por:',
        'Acompanhar',
    ),
    ('BDE', 24): (
        'waiting_retrieval',
        'Objeto disponível para retirada em Caixa Postal',
        '',
        'Acompanhar',
    ),
    ('BDI', 24): (
        'waiting_retrieval',
        'Objeto disponível para retirada em Caixa Postal',
        '',
        'Acompanhar',
    ),
    ('BDR', 24): (
        'waiting_retrieval',
        'Objeto disponível para retirada em Caixa Postal',
        '',
        'Acompanhar',
    ),
    ('BDE', 25): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Empresa sem expediente',
        'A entrega deverá ocorrer no próximo dia útil',
        'Acompanhar',
    ),
    ('BDI', 25): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Empresa sem expediente',
        'A entrega deverá ocorrer no próximo dia útil',
        'Acompanhar',
    ),
    ('BDR', 25): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Empresa sem expediente',
        'A entrega deverá ocorrer no próximo dia útil',
        'Acompanhar',
    ),
    ('BDE', 26): (
        'waiting_retrieval',
        'Destinatário não retirou objeto na Unidade dos Correios',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 26): (
        'waiting_retrieval',
        'Destinatário não retirou objeto na Unidade dos Correios',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 26): (
        'waiting_retrieval',
        'Destinatário não retirou objeto na Unidade dos Correios',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 28): (
        'damaged',
        'Objeto e/ou conteúdo avariado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDI', 28): (
        'damaged',
        'Objeto e/ou conteúdo avariado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDR', 28): (
        'damaged',
        'Objeto e/ou conteúdo avariado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDE', 32): (
        'shipped',
        'Objeto com data de entrega agendada',
        '',
        'Acompanhar',
    ),
    ('BDI', 32): (
        'shipped',
        'Objeto com data de entrega agendada',
        '',
        'Acompanhar',
    ),
    ('BDR', 32): (
        'shipped',
        'Objeto com data de entrega agendada',
        '',
        'Acompanhar',
    ),
    ('BDE', 33): (
        'delivery_rejected',
        'A entrega não pode ser efetuada - Destinatário não apresentou documento exigido',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 33): (
        'delivery_rejected',
        'A entrega não pode ser efetuada - Destinatário não apresentou documento exigido',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 33): (
        'delivery_rejected',
        'A entrega não pode ser efetuada - Destinatário não apresentou documento exigido',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 34): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Logradouro com numeração irregular',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar',
    ),
    ('BDI', 34): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Logradouro com numeração irregular',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar',
    ),
    ('BDR', 34): (
        'delivery_unsuccessful',
        'A entrega não pode ser efetuada - Logradouro com numeração irregular',
        'Objeto sujeito a atraso na entrega ou a devolução ao remetente',
        'Acompanhar',
    ),
    ('BDE', 35): (
        'delivery_unsuccessful',
        'Coleta ou entrega de objeto não efetuada',
        'Será realizada nova tentativa de coleta ou entrega',
        'Acompanhar',
    ),
    ('BDI', 35): (
        'delivery_unsuccessful',
        'Coleta ou entrega de objeto não efetuada',
        'Será realizada nova tentativa de coleta ou entrega',
        'Acompanhar',
    ),
    ('BDR', 35): (
        'delivery_unsuccessful',
        'Coleta ou entrega de objeto não efetuada',
        'Será realizada nova tentativa de coleta ou entrega',
        'Acompanhar',
    ),
    ('BDE', 36): (
        'delivery_unsuccessful',
        'Coleta ou entrega de objeto não efetuada',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 36): (
        'delivery_unsuccessful',
        'Coleta ou entrega de objeto não efetuada',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 36): (
        'delivery_unsuccessful',
        'Coleta ou entrega de objeto não efetuada',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 37): (
        'damaged',
        'Objeto e/ou conteúdo avariado por acidente com veículo',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDI', 37): (
        'damaged',
        'Objeto e/ou conteúdo avariado por acidente com veículo',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDR', 37): (
        'damaged',
        'Objeto e/ou conteúdo avariado por acidente com veículo',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDE', 38): (
        'delivery_unsuccessful',
        'Objeto endereçado à empresa falida',
        'Objeto será encaminhado para entrega ao administrador judicial',
        'Acompanhar',
    ),
    ('BDI', 38): (
        'delivery_unsuccessful',
        'Objeto endereçado à empresa falida',
        'Objeto será encaminhado para entrega ao administrador judicial',
        'Acompanhar',
    ),
    ('BDR', 38): (
        'delivery_unsuccessful',
        'Objeto endereçado à empresa falida',
        'Objeto será encaminhado para entrega ao administrador judicial',
        'Acompanhar',
    ),
    ('BDE', 40): (
        'customs_control',
        'A importação do objeto/conteúdo não foi autorizada pelos órgãos fiscalizadores',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 40): (
        'customs_control',
        'A importação do objeto/conteúdo não foi autorizada pelos órgãos fiscalizadores',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 40): (
        'customs_control',
        'A importação do objeto/conteúdo não foi autorizada pelos órgãos fiscalizadores',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 41): (
        'unknown',
        'A entrega do objeto está condicionada à composição do lote',
        '',
        'Acompanhar',
    ),
    ('BDI', 41): (
        'unknown',
        'A entrega do objeto está condicionada à composição do lote',
        '',
        'Acompanhar',
    ),
    ('BDR', 41): (
        'unknown',
        'A entrega do objeto está condicionada à composição do lote',
        '',
        'Acompanhar',
    ),
    ('BDE', 42): (
        'unknown',
        'Lote de objetos incompleto',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 42): (
        'unknown',
        'Lote de objetos incompleto',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 42): (
        'unknown',
        'Lote de objetos incompleto',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 43): (
        'customs_control',
        'Objeto apreendido por órgão de fiscalização ou outro órgão anuente',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDI', 43): (
        'customs_control',
        'Objeto apreendido por órgão de fiscalização ou outro órgão anuente',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDR', 43): (
        'customs_control',
        'Objeto apreendido por órgão de fiscalização ou outro órgão anuente',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDE', 45): (
        'shipped',
        'Objeto recebido na unidade de distribuição',
        'Entrega prevista para o próximo dia útil',
        'Acompanhar',
    ),
    ('BDI', 45): (
        'shipped',
        'Objeto recebido na unidade de distribuição',
        'Entrega prevista para o próximo dia útil',
        'Acompanhar',
    ),
    ('BDR', 45): (
        'shipped',
        'Objeto recebido na unidade de distribuição',
        'Entrega prevista para o próximo dia útil',
        'Acompanhar',
    ),
    ('BDE', 46): (
        'delivery_unsuccessful',
        'Tentativa de entrega não efetuada',
        'Entrega prevista para o próximo dia útil',
        'Acompanhar',
    ),
    ('BDI', 46): (
        'delivery_unsuccessful',
        'Tentativa de entrega não efetuada',
        'Entrega prevista para o próximo dia útil',
        'Acompanhar',
    ),
    ('BDR', 46): (
        'delivery_unsuccessful',
        'Tentativa de entrega não efetuada',
        'Entrega prevista para o próximo dia útil',
        'Acompanhar',
    ),
    ('BDE', 47): (
        'delivery_unsuccessful',
        'Saída para entrega cancelada',
        'Será efetuado novo lançamento para entrega',
        'Acompanhar',
    ),
    ('BDI', 47): (
        'delivery_unsuccessful',
        'Saída para entrega cancelada',
        'Será efetuado novo lançamento para entrega',
        'Acompanhar',
    ),
    ('BDR', 47): (
        'delivery_unsuccessful',
        'Saída para entrega cancelada',
        'Será efetuado novo lançamento para entrega',
        'Acompanhar',
    ),
    ('BDE', 48): (
        'delivery_rejected',
        'Retirada em Unidade dos Correios não autorizada pelo remetente',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 48): (
        'delivery_rejected',
        'Retirada em Unidade dos Correios não autorizada pelo remetente',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 48): (
        'delivery_rejected',
        'Retirada em Unidade dos Correios não autorizada pelo remetente',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 49): (
        'delivery_unsuccessful',
        'As dimensões do objeto impossibilitam o tratamento e a entrega',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 49): (
        'delivery_unsuccessful',
        'As dimensões do objeto impossibilitam o tratamento e a entrega',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 49): (
        'delivery_unsuccessful',
        'As dimensões do objeto impossibilitam o tratamento e a entrega',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 50): (
        'lost',
        'Objeto roubado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDI', 50): (
        'lost',
        'Objeto roubado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDR', 50): (
        'lost',
        'Objeto roubado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDE', 51): (
        'lost',
        'Objeto roubado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDI', 51): (
        'lost',
        'Objeto roubado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDR', 51): (
        'lost',
        'Objeto roubado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDE', 52): (
        'lost',
        'Objeto roubado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDI', 52): (
        'lost',
        'Objeto roubado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDR', 52): (
        'lost',
        'Objeto roubado',
        'Favor entrar em contato com os Correios.',
        'Acionar atendimento dos Correios.',
    ),
    ('BDE', 53): (
        'shipped',
        'Objeto reimpresso e reenviado',
        '',
        'Acompanhar. O objeto impresso pelos Correios precisou ser refeito e reenviado.',
    ),
    ('BDI', 53): (
        'shipped',
        'Objeto reimpresso e reenviado',
        '',
        'Acompanhar. O objeto impresso pelos Correios precisou ser refeito e reenviado.',
    ),
    ('BDR', 53): (
        'shipped',
        'Objeto reimpresso e reenviado',
        '',
        'Acompanhar. O objeto impresso pelos Correios precisou ser refeito e reenviado.',
    ),
    ('BDE', 54): (
        'customs_control',
        'Para recebimento do objeto, é necessário o pagamento do ICMS Importação',
        '',
        'Acompanhar. O interessado deverá pagar o imposto devido para retirar o objeto em uma Unidade dos Correios.',
    ),
    ('BDI', 54): (
        'customs_control',
        'Para recebimento do objeto, é necessário o pagamento do ICMS Importação',
        '',
        'Acompanhar. O interessado deverá pagar o imposto devido para retirar o objeto em uma Unidade dos Correios.',
    ),
    ('BDR', 54): (
        'customs_control',
        'Para recebimento do objeto, é necessário o pagamento do ICMS Importação',
        '',
        'Acompanhar. O interessado deverá pagar o imposto devido para retirar o objeto em uma Unidade dos Correios.',
    ),
    ('BDE', 55): (
        'customs_control',
        'Solicitada revisão do tributo estabelecido',
        '',
        'Acompanhar',
    ),
    ('BDI', 55): (
        'customs_control',
        'Solicitada revisão do tributo estabelecido',
        '',
        'Acompanhar',
    ),
    ('BDR', 55): (
        'customs_control',
        'Solicitada revisão do tributo estabelecido',
        '',
        'Acompanhar',
    ),
    ('BDE', 56): (
        'customs_control',
        'Declaração aduaneira ausente ou incorreta',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDI', 56): (
        'customs_control',
        'Declaração aduaneira ausente ou incorreta',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDR', 56): (
        'customs_control',
        'Declaração aduaneira ausente ou incorreta',
        'Objeto será devolvido ao remetente',
        'Acompanhar o retorno do objeto ao remetente.',
    ),
    ('BDE', 57): (
        'customs_control',
        'Revisão de tributo concluída - Objeto liberado',
        '',
        'Acompanhar',
    ),
    ('BDI', 57): (
        'customs_control',
        'Revisão de tributo concluída - Objeto liberado',
        '',
        'Acompanhar',
    ),
    ('BDR', 57): (
        'customs_control',
        'Revisão de tributo concluída - Objeto liberado',
        '',
        'Acompanhar',
    ),
    ('BDE', 58): (
        'customs_control',
        'Revisão de tributo concluída - Tributo alterado',
        'O valor do tributo pode ter aumentado ou diminuído',
        'Acompanhar',
    ),
    ('BDI', 58): (
        'customs_control',
        'Revisão de tributo concluída - Tributo alterado',
        'O valor do tributo pode ter aumentado ou diminuído',
        'Acompanhar',
    ),
    ('BDR', 58): (
        'customs_control',
        'Revisão de tributo concluída - Tributo alterado',
        'O valor do tributo pode ter aumentado ou diminuído',
        'Acompanhar',
    ),
    ('BDE', 59): (
        'customs_control',
        'Revisão de tributo concluída - Tributo mantido',
        'Poderá haver incidência de juros e multa.',
        'Acompanhar',
    ),
    ('BDI', 59): (
        'customs_control',
        'Revisão de tributo concluída - Tributo mantido',
        'Poderá haver incidência de juros e multa.',
        'Acompanhar',
    ),
    ('BDR', 59): (
        'customs_control',
        'Revisão de tributo concluída - Tributo mantido',
        'Poderá haver incidência de juros e multa.',
        'Acompanhar',
    ),
    ('BDE', 66): (
        'shipped_delayed',
        'Área com distribuição sujeita a prazo diferenciado',
        'Restrição de entrega domiciliar temporária',
        'Acompanhar',
    ),
    ('BDI', 66): (
        'shipped_delayed',
        'Área com distribuição sujeita a prazo diferenciado',
        'Restrição de entrega domiciliar temporária',
        'Acompanhar',
    ),
    ('BDR', 66): (
        'shipped_delayed',
        'Área com distribuição sujeita a prazo diferenciado',
        'Restrição de entrega domiciliar temporária',
        'Acompanhar',
    ),
    ('BDE', 69): (
        'shipped_delayed',
        'Objeto com atraso na entrega',
        '',
        'Acompanhar',
    ),
    ('BDI', 69): (
        'shipped_delayed',
        'Objeto com atraso na entrega',
        '',
        'Acompanhar',
    ),
    ('BDR', 69): (
        'shipped_delayed',
        'Objeto com atraso na entrega',
        '',
        'Acompanhar',
    ),
}

ZIP_CODE_MAP = {
    'SP': (
        RangeSet((1000, 20000)),  # all
        RangeSet((1000, 6000), (8000, 8500)),  # capital
    ),
    'RJ': (
        RangeSet((20000, 29000)),
        RangeSet((20000, 23800)),
    ),
    'ES': (
        RangeSet((29000, 30000)),
        RangeSet((29000, 29100)),
    ),
    'MG': (
        RangeSet((30000, 40000)),
        RangeSet((30000, 32000)),
    ),
    'BA': (
        RangeSet((40000, 49000)),
        RangeSet((40000, 42600)),
    ),
    'SE': (
        RangeSet((49000, 50000)),
        RangeSet((49000, 49100)),
    ),
    'PE': (
        RangeSet((50000, 57000)),
        RangeSet((50000, 53000)),
    ),
    'AL': (
        RangeSet((57000, 58000)),
        RangeSet((57000, 57100)),
    ),
    'PB': (
        RangeSet((58000, 59000)),
        RangeSet((58000, 58100)),
    ),
    'RN': (
        RangeSet((59000, 60000)),
        RangeSet((59000, 59140)),
    ),
    'CE': (
        RangeSet((60000, 64000)),
        RangeSet((60000, 61600)),
    ),
    'PI': (
        RangeSet((64000, 65000)),
        RangeSet((64000, 64100)),
    ),
    'MA': (
        RangeSet((65000, 66000)),
        RangeSet((65000, 65110)),
    ),
    'PA': (
        RangeSet((66000, 68900)),
        RangeSet((66000, 67000)),
    ),
    'AP': (
        RangeSet((68900, 69000)),
        RangeSet((68900, 68915)),
    ),
    'AM': (
        RangeSet((69000, 69300), (69400, 69900)),
        RangeSet((69000, 69100), (69000, 69100)),
    ),
    'RR': (
        RangeSet((69300, 69400)),
        RangeSet((69300, 69340)),
    ),
    'AC': (
        RangeSet((69900, 70000)),
        RangeSet((69900, 69925)),
    ),
    'DF': (
        RangeSet((70000, 72800), (73000, 73700)),
        RangeSet((70000, 72800), (73000, 73700)),
    ),
    'GO': (
        RangeSet((72800, 73000), (73700, 76800)),
        RangeSet((74000, 74900)),
    ),
    'TO': (
        RangeSet((77000, 78000)),
        RangeSet((77000, 77300)),
    ),
    'MT': (
        RangeSet((78000, 78900)),
        RangeSet((78000, 78110)),
    ),
    'RO': (
        RangeSet((76800, 77000)),
        RangeSet((76800, 76850)),
    ),
    'MS': (
        RangeSet((79000, 80000)),
        RangeSet((79000, 79130)),
    ),
    'PR': (
        RangeSet((80000, 88000)),
        RangeSet((80000, 83000)),
    ),
    'SC': (
        RangeSet((88000, 90000)),
        RangeSet((88000, 88100)),
    ),
    'RS': (
        RangeSet((90000, 100000)),
        RangeSet((90000, 92000)),
    ),
}  # type: Dict[str, Tuple[RangeSet, RangeSet]]

ZIP_CODES = RangeSet(*(p[0] for p in ZIP_CODE_MAP.values()))

FREIGHT_ERROR_RESPONSES = {
    0: 'Processamento com sucesso',
    -1: 'Código de serviço inválido',
    -2: 'CEP de origem inválido',
    -3: 'CEP de destino inválido',
    -4: 'Peso excedido',
    -5: 'O Valor Declarado não deve exceder R$ 10.000,00',
    -6: 'Serviço indisponível para o trecho informado',
    -7: 'O Valor Declarado é obrigatório para este serviço',
    -8: 'Este serviço não aceita Mão Própria',
    -9: 'Este serviço não aceita Aviso de Recebimento',
    -10: 'Precificação indisponível para o trecho informado',
    -11: 'Para definição do preço deverão ser informados, também, o comprimento, '
         'a largura e altura do objeto em centímetros (cm).',
    -12: 'Comprimento inválido.',
    -13: 'Largura inválida.',
    -14: 'Altura inválida.',
    -15: 'O comprimento não pode ser maior que 105 cm.',
    -16: 'A largura não pode ser maior que 105 cm.',
    -17: 'A altura não pode ser maior que 105 cm.',
    -18: 'A altura não pode ser inferior a 2 cm.',
    -20: 'A largura não pode ser inferior a 11 cm.',
    -22: 'O comprimento não pode ser inferior a 16 cm.',
    -23: 'A soma resultante do comprimento + largura + altura não deve superar a 200 cm.',
    -24: 'Comprimento inválido.',
    -25: 'Diâmetro inválido',
    -26: 'Informe o comprimento.',
    -27: 'Informe o diâmetro.',
    -28: 'O comprimento não pode ser maior que 105 cm.',
    -29: 'O diâmetro não pode ser maior que 91 cm.',
    -30: 'O comprimento não pode ser inferior a 18 cm.',
    -31: 'O diâmetro não pode ser inferior a 5 cm.',
    -32: 'A soma resultante do comprimento + o dobro do diâmetro não deve superar a 200 cm.',
    -33: 'Sistema temporariamente fora do ar. Favor tentar mais tarde.',
    -34: 'Código Administrativo ou Senha inválidos.',
    -35: 'Senha incorreta.',
    -36: 'Cliente não possui contrato vigente com os Correios.',
    -37: 'Cliente não possui serviço ativo em seu contrato.',
    -38: 'Serviço indisponível para este código administrativo.',
    -39: 'Peso excedido para o formato envelope',
    -40: ('Para definicao do preco deverao ser informados, tambem, o comprimento e a '
          'largura e altura do objeto em centimetros (cm).'),
    -41: 'O comprimento nao pode ser maior que 60 cm.',
    -42: 'O comprimento nao pode ser inferior a 16 cm.',
    -43: 'A soma resultante do comprimento + largura nao deve superar a 120 cm.',
    -44: 'A largura nao pode ser inferior a 11 cm.',
    -45: 'A largura nao pode ser maior que 60 cm.',
    -888: 'Erro ao calcular a tarifa',
    6: 'Localidade de origem não abrange o serviço informado',
    # 7: 'Localidade de destino não abrange o serviço informado',
    7: 'Serviço indisponível, tente mais tarde',  #
    8: 'Serviço indisponível para o trecho informado',
    9: 'CEP inicial pertencente a Área de Risco.',
    10: 'Área com entrega temporariamente sujeita a prazo diferenciado.',
    11: 'CEP inicial e final pertencentes a Área de Risco',
    99: 'Outros erros diversos do .Net',
}

FREIGHT_ERROR_INITIAL_ZIPCODE_RESTRICTED = 9
FREIGHT_ERROR_FINAL_ZIPCODE_RESTRICTED = 10
FREIGHT_ERROR_INITIAL_AND_FINAL_ZIPCODE_RESTRICTED = 11
