"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsDeploymentProvider = void 0;
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const cr = require("aws-cdk-lib/custom-resources");
const is_complete_function_1 = require("./is-complete-function");
const on_event_function_1 = require("./on-event-function");
/**
 * A custom resource provider to handle creation of new {@link EcsDeployment}.
 */
class EcsDeploymentProvider extends cr.Provider {
    constructor(scope, id, props) {
        const functionNamePrefix = [
            'EcsDeploymentProvider',
            props.deploymentGroup.application.applicationName,
            props.deploymentGroup.deploymentGroupName,
        ].join('-');
        const eventLambda = new on_event_function_1.OnEventFunction(scope, `${id}OnEventLambda`, {
            functionName: `${functionNamePrefix}-onEvent`,
            timeout: cdk.Duration.seconds(60),
        });
        eventLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'codedeploy:GetApplicationRevision',
                'codedeploy:RegisterApplicationRevision',
            ],
            resources: [
                props.deploymentGroup.application.applicationArn,
            ],
        }));
        eventLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'codedeploy:CreateDeployment',
                'codedeploy:StopDeployment',
                'codedeploy:GetDeployment',
            ],
            resources: [
                props.deploymentGroup.deploymentGroupArn,
            ],
        }));
        eventLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'codedeploy:GetDeploymentConfig',
            ],
            resources: [
                props.deploymentGroup.deploymentConfig.deploymentConfigArn,
            ],
        }));
        const completeLambda = new is_complete_function_1.IsCompleteFunction(scope, `${id}IsCompleteLambda`, {
            functionName: `${functionNamePrefix}-isComplete`,
            timeout: cdk.Duration.seconds(60),
        });
        completeLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'codedeploy:GetDeployment',
            ],
            resources: [
                props.deploymentGroup.deploymentGroupArn,
            ],
        }));
        super(scope, id, {
            providerFunctionName: `${functionNamePrefix}-provider`,
            onEventHandler: eventLambda,
            isCompleteHandler: completeLambda,
            queryInterval: props.queryInterval || cdk.Duration.seconds(15),
            totalTimeout: props.timeout,
        });
    }
}
exports.EcsDeploymentProvider = EcsDeploymentProvider;
//# sourceMappingURL=data:application/json;base64,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