"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = exports.DeploymentStatus = void 0;
const AWS = require("aws-sdk");
const logger_1 = require("./logger");
var DeploymentStatus;
(function (DeploymentStatus) {
    DeploymentStatus["CREATED"] = "Created";
    DeploymentStatus["QUEUED"] = "Queued";
    DeploymentStatus["IN_PROGRESS"] = "InProgress";
    DeploymentStatus["BAKING"] = "Baking";
    DeploymentStatus["SUCCEEDED"] = "Succeeded";
    DeploymentStatus["FAILED"] = "Failed";
    DeploymentStatus["STOPPED"] = "Stopped";
    DeploymentStatus["READY"] = "Ready";
})(DeploymentStatus = exports.DeploymentStatus || (exports.DeploymentStatus = {}));
/**
 * The lambda function called from CloudFormation for this custom resource.
 *
 * @param event
 * @returns whether the deployment is complete
 */
async function handler(event) {
    const logger = new logger_1.Logger();
    const codedeployClient = new AWS.CodeDeploy();
    try {
        const resp = await codedeployClient.getDeployment({ deploymentId: event.PhysicalResourceId }).promise();
        let rollbackResp = {};
        if (resp.deploymentInfo?.rollbackInfo?.rollbackDeploymentId) {
            rollbackResp = await codedeployClient.getDeployment({ deploymentId: resp.deploymentInfo?.rollbackInfo?.rollbackDeploymentId }).promise();
        }
        logger.appendKeys({
            stackEvent: event.RequestType,
            deploymentId: event.PhysicalResourceId,
            deploymentStatus: resp.deploymentInfo?.status,
            rollbackStatus: rollbackResp?.deploymentInfo?.status,
        });
        logger.info('Checking deployment');
        // check if deployment id is complete
        switch (event.RequestType) {
            case 'Create':
            case 'Update':
                switch (resp.deploymentInfo?.status) {
                    case DeploymentStatus.SUCCEEDED:
                        logger.info('Deployment finished successfully', { complete: true });
                        return { IsComplete: true };
                    case DeploymentStatus.FAILED:
                    case DeploymentStatus.STOPPED:
                        if (rollbackResp.deploymentInfo?.status) {
                            if (rollbackResp.deploymentInfo?.status == DeploymentStatus.SUCCEEDED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.FAILED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.STOPPED) {
                                const errInfo = resp.deploymentInfo.errorInformation;
                                const error = new Error(`Deployment ${resp.deploymentInfo.status}: [${errInfo?.code}] ${errInfo?.message}`);
                                logger.error('Deployment failed', { complete: true, error });
                                throw error;
                            }
                            logger.info('Waiting for final status from a rollback', { complete: false });
                            return { IsComplete: false }; // waiting for final status from rollback
                        }
                        else {
                            const errInfo = resp.deploymentInfo.errorInformation;
                            const error = new Error(`Deployment ${resp.deploymentInfo.status}: [${errInfo?.code}] ${errInfo?.message}`);
                            logger.error('No rollback to wait for', { complete: true, error });
                            throw error;
                        }
                    default:
                        logger.info('Waiting for final status from deployment', { complete: false });
                        return { IsComplete: false };
                }
            case 'Delete':
                switch (resp.deploymentInfo?.status) {
                    case DeploymentStatus.SUCCEEDED:
                        logger.info('Deployment finished successfully - nothing to delete', { complete: true });
                        return { IsComplete: true };
                    case DeploymentStatus.FAILED:
                    case DeploymentStatus.STOPPED:
                        if (rollbackResp.deploymentInfo?.status) {
                            if (rollbackResp.deploymentInfo?.status == DeploymentStatus.SUCCEEDED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.FAILED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.STOPPED) {
                                logger.info('Rollback in final status', { complete: true });
                                return { IsComplete: true }; // rollback finished, we're deleted
                            }
                            logger.info('Waiting for final status from a rollback', { complete: false });
                            return { IsComplete: false }; // waiting for rollback
                        }
                        logger.info('No rollback to wait for', { complete: true });
                        return { IsComplete: true };
                    default:
                        logger.info('Waiting for final status from deployment', { complete: false });
                        return { IsComplete: false };
                }
            default:
                logger.error('Unknown request type');
                throw new Error(`Unknown request type: ${event.RequestType}`);
        }
    }
    catch (e) {
        logger.error('Unable to determine deployment status', { error: e });
        if (event.RequestType === 'Delete') {
            logger.warn('Ignoring error - nothing to do', { complete: true });
            return { IsComplete: true };
        }
        throw e;
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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