"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const AWS = require("aws-sdk");
const logger_1 = require("./logger");
/**
 * The lambda function called from CloudFormation for this custom resource.
 *
 * @param event
 * @returns attribues of the deployment that was created
 */
async function handler(event) {
    const logger = new logger_1.Logger();
    const codedeployClient = new AWS.CodeDeploy();
    logger.appendKeys({
        stackEvent: event.RequestType,
    });
    switch (event.RequestType) {
        case 'Create':
        case 'Update': {
            // create deployment
            const props = event.ResourceProperties;
            let autoRollbackConfiguration;
            if (props.autoRollbackConfigurationEnabled === 'true') {
                autoRollbackConfiguration = {
                    enabled: true,
                    events: props.autoRollbackConfigurationEvents.split(','),
                };
            }
            else if (props.autoRollbackConfigurationEnabled === 'false') {
                autoRollbackConfiguration = {
                    enabled: false,
                };
            }
            const resp = await codedeployClient.createDeployment({
                applicationName: props.applicationName,
                deploymentConfigName: props.deploymentConfigName,
                deploymentGroupName: props.deploymentGroupName,
                autoRollbackConfiguration,
                description: props.description,
                revision: {
                    revisionType: 'AppSpecContent',
                    appSpecContent: {
                        content: props.revisionAppSpecContent,
                    },
                },
            }).promise();
            if (!resp.deploymentId) {
                throw new Error('No deploymentId received from call to CreateDeployment');
            }
            logger.appendKeys({
                deploymentId: resp.deploymentId,
            });
            logger.info('Created new deployment');
            return {
                PhysicalResourceId: resp.deploymentId,
                Data: {
                    deploymentId: resp.deploymentId,
                },
            };
        }
        case 'Delete':
            logger.appendKeys({
                deploymentId: event.PhysicalResourceId,
            });
            // cancel deployment and rollback
            try {
                const resp = await codedeployClient.stopDeployment({
                    deploymentId: event.PhysicalResourceId,
                    autoRollbackEnabled: true,
                }).promise();
                logger.info(`Stopped deployment: ${resp.status} ${resp.statusMessage}`);
            }
            catch (e) {
                logger.warn('Ignoring error', { error: e });
            }
            return {
                PhysicalResourceId: event.PhysicalResourceId,
                Data: {
                    deploymentId: event.PhysicalResourceId,
                },
            };
        default:
            logger.error('Unknown stack event');
            throw new Error(`Unknown request type: ${event.RequestType}`);
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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