# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': '.'}

packages = \
['gwa']

package_data = \
{'': ['*']}

install_requires = \
['toml[tomli]>=0.10.2,<0.11.0']

setup_kwargs = {
    'name': 'incolumepy.gwa',
    'version': '0.1.18',
    'description': 'Prospections from Github Workflows Actions',
    'long_description': '[![Tests CI/CD](https://github.com/incolumepy-prospections/incolumepy.gwa/actions/workflows/tests_ci_cd.yaml/badge.svg)](https://github.com/incolumepy-prospections/incolumepy.gwa/actions/workflows/tests_ci_cd.yaml)\n[![publish automatically](https://github.com/incolumepy-prospections/incolumepy.gwa/actions/workflows/publish-automatically.yml/badge.svg)](https://github.com/incolumepy-prospections/incolumepy.gwa/actions/workflows/publish-automatically.yml)\n# incolumepy.gwa\nProspection GitHub Workflows Actions\n\nContains:\n+ poetry\n+ pytest\n+ coverage\n+ Python multiversions (pyenv)\n+ tox\n+ black\n+ isort\n+ flake8\n+ pylint\n+ pydocstyle\n+ mypy\n+ semver\n\n---\n## Start Project\n1. Create a github project on github.com;\n2. Clone the project, on local host;\n3. Create the branch dev and branches of control when necessary:\n   ```bash\n   $ git br dev\n   $ git co -b feature#1 dev\n   $ git co -b feature#2 dev\n   $ git co -b bug#2 dev\n   ```\n4. Configure project structure:\n   1. Set Editor configuration:\n      ```bash\n      $ curl https://pastebin.com/raw/TrDhuvFZ -o .editorconfig\n      ```\n   2. Update gitignore:\n      ```bash\n      $ curl https://pastebin.com/raw/C910Dqze -o .gitignore\n      ```\n    3. Commit changes\n5. Configure poetry:\n   1. Init poetry\n      ```bash\n      $ poetry init    # response that need\n      ```\n   2. Adding package path in configuration:\n    ```bash\n    [tool.poetry]\n      ...\n    packages=[\n        {include = \'incolumepy/gwa\', from=\'\'},\n    ]\n    ```\n   3. Define Python version (3.6.8+):\n       ```bash\n      Change..\n      [tool.poetry.dependencies]\n      python = "~3.10"\n\n      To ..\n      [tool.poetry.dependencies]\n      python = ">=3.6.8,<4"\n       ```\n   4. Commit changes\n6. Configure unit tests with pytest:\n   1. Install pytest\n      ```bash\n      $ poetry add -D pytest     ## adding pytest\n      $ poetry add -D pytest-cov     ## adding pytest-cov\n      ```\n   2. Create tests structure\n      ```bash\n      $ cd incolumepy.gwa;    # home_project\n      $ mkdir tests       # create directory\n      $ > conftest.py     # create file\n      $ > __init__.py     # create file\n      ```\n   3. Add test environment\n      ```bash\n      $  git add -f poetry.lock pyproject.toml  tests/\n      ```\n   4. Commit changes\n\n## Linters and Checkers\n1. Adding lint/control tools\n   1. ```bash\n      $ poetry add -D isort pydocstyle pylint mypy black flake8 tox\n      $ git ci -m "Adding tox and lint tools" poetry.lock pyproject.toml\n      ```\n2. Configure code coverage\n   1. Edit _pyproject.toml_, add on end file.\n   ```bash\n   $ ...\n   [tool.pytest.ini_options]\n   addopts = "--cov=incolumepy.gwa"\n   testpaths = [\n      "tests",\n   ]\n\n   ```\n3. Configure pylint tool\n   1. Edit _pyproject.toml_, add on end file.\n   ```bash\n   ...\n   [tool.pylint.format]\n   # Maximum number of characters on a single line.\n   max-line-length = 160\n\n   [tool.pylint.basic]\n   # Allow shorter and longer variable names than the default.\n   argument-rgx = "[a-z_][a-z0-9_]*$"\n   attr-rgx = "[a-z_][a-z0-9_]*$"\n   variable-rgx = "[a-z_][a-z0-9_]*$"\n\n   # Ensure that orjson is analysed as a C extension by pylint.\n   extension-pkg-whitelist = "orjson"\n\n   [tool.pylint.messages_control]\n   disable = [\n       # Disable too many and too few checks.\n       "too-many-ancestors",\n       "too-many-arguments",\n       "too-many-boolean-expressions",\n       "too-many-branches",\n       "too-many-function-args",\n       "too-many-instance-attributes",\n       "too-many-lines",\n       "too-many-locals",\n       "too-many-nested-blocks",\n       "too-many-public-methods",\n       "too-many-return-statements",\n       "too-many-statements",\n       "too-few-public-methods",\n\n       # Similar lines in files (often the case in tests).\n       "duplicate-code",\n\n       # Many functions (e.g. callbacks) will naturally have unused arguments.\n       "unused-argument",\n\n       # Disable checking that method could be a function in classes (often used for organisation).\n       "no-self-use",\n\n       # Disable failure for TODO items in the codebase (code will always have TODOs).\n       "fixme",\n\n       # Disable docstrings checks as we don\'t require excessive documentation.\n       "missing-docstring",\n\n       "no-member",\n       "unspecified-encoding",\n   ]\n   ```\n   2. Commit\n      ```bash\n      $ git ci -m "Configure pylint tool" pyproject.toml\n      ```\n4. Configure mypy tool\n    1. Edit _pyproject.toml_, add on end file.\n    ```bash\n    ...\n       [tool.mypy]\n       mypy_path = "incolumepy"\n       check_untyped_defs = true\n       disallow_any_generics = true\n       ignore_missing_imports = true\n       no_implicit_optional = true\n       show_error_codes = true\n       strict_equality = true\n       warn_redundant_casts = true\n       warn_return_any = true\n       warn_unreachable = true\n       warn_unused_configs = true\n       no_implicit_reexport = true\n    ```\n   2. Commit\n   ```bash\n   $ git ci -m "Configure mypy tool" pyproject.toml\n   ```\n5. Configure isort tool\n    1. Edit _pyproject.toml_, add on end file.\n    ```bash\n    ...\n    [tool.isort]\n    multi_line_output = 3\n    line_length = 160\n    include_trailing_comma = true\n    ```\n    2. Commit\n      ```bash\n      $ git ci -m "Configure isort tool" pyproject.toml\n      ```\n6. Configure flake8 tool\n    1. Edit _pyproject.toml_, add on end file.\n    ```bash\n    ...\n    [flake8]\n    max-line-length = 160\n    ```\n    2. Commit\n      ```bash\n      $ git ci -m "Configure flake8 tool" pyproject.toml\n      ```\n7. Configure black tool:\n    1. Edit _pyproject.toml_, add on end file.\n    ```bash\n    ...\n    [tool.black]\n    line_length = 160\n    ```\n    2. Commit\n      ```bash\n      $ git ci -m "Configure black tool" pyproject.toml\n      ```\n8. Configure tox tool:\n    1. Edit _pyproject.toml_, add on end file.\n    ```bash\n    [tool.tox]\n    legacy_tox_ini = """\n    [tox]\n    minversion = 3.3.0\n    isolated_build = True\n    envlist =\n    black\n    flake8\n    isort\n    mypy\n    pydocstyle\n    pylint\n    py{36,37,38,39,310}\n\n    ;[tox:.package]\n    ;basepython = python3\n\n    [testenv]\n    whitelist_externals = poetry\n    skip_install = true\n    commands =\n    poetry env use {envpython}\n    poetry install -vv --no-root\n    poetry run pytest {posargs} tests/\n\n    [testenv:flake8]\n    deps = flake8\n    commands = flake8 --config pyproject.toml incolumepy/ tests/\n\n    [testenv:mypy]\n    deps =\n    mypy\n    types-toml\n    commands = mypy incolumepy/\n\n    [testenv:pydocstyle]\n    commands = poetry run pydocstyle incolumepy/ tests/\n\n    [testenv:isort]\n    commands = poetry run isort --check --atomic --py all incolumepy/ tests/\n\n    [testenv:pylint]\n    commands = poetry run pylint incolumepy/ tests/\n\n    [testenv:black]\n    commands = poetry run black --check incolumepy/ tests/\n    """\n    ```\n    2. Commit\n      ```bash\n      $ git ci -m "Configure tox tool" pyproject.toml\n      ```\n\n## Testting lint configurate\n1. Test pydocstyle:\n   ```bash\n   $ pydocstyle incolumepy/ tests/\n   ```\n2. Test pylint\n   ```bash\n   $ pylint incolumepy/ tests/\n   ```\n3. Test mypy\n   ```bash\n   $ mypy incolumepy/ tests/\n   ```\n4. Test isort\n   ```bash\n   $ isort --check incolumepy/ tests/\n   ```\n5. Test flake8\n   ```bash\n   $ flake8 --config pyproject.toml incolumepy/ tests/\n   ```\n6. Test black\n   ```bash\n   $ black --check incolumepy/ tests/\n   ```\n\n## Testting tox configurate\n6. Text tox\n   ```bash\n   $ tox\n   ```\n\n## Control version with poetry\nUnlike bumpversion, poetry changes the version of the main file only.\nTo work around this situation and update the version of pyproject.toml,\n__init__.py and version.txt, we will do the following procedure.\n\n```bash\n$ cat > incolumepy/gwa/__init__.py << eof\nimport toml\nfrom pathlib import Path\n__root__ = Path(__file__).parents[0]\nversion_file = __root__.joinpath(\'version.txt\')\nversion_file.write_text(f"{toml.load(Path(__file__).parents[2].joinpath(\'pyproject.toml\'))[\'tool\'][\'poetry\'][\'version\']}\\n")\n__version__ = version_file.read_text().strip()\neof\n\n$ v=`poetry version prerelease`; pytest tests/ && git ci -m "$v" pyproject.toml $(find -name "version.txt")  #sem tag\n$ s=`poetry version patch`; pytest tests/ && git ci -m "`echo $s`" pyproject.toml `find -name "version.txt"`; git tag -f `poetry version -s` -m "$(echo $s)"  #com tag\n\n```\n## Facility with make\nUse make to easy various commands.\n```bash\n$ curl https://pastebin.com/raw/eTHpL70G -o Makefile\n```\n### Makefile help\n```bash\n$ make help\n```\n### New prerelease with Makefile\n```bash\n$ git co dev\n$ git merge --no-ff [branch] --autostash\n$ make prerelease\n```\n### New release with Makefile\n```bash\n$ git co dev\n$ make release\n```\n### Checking lint\n```bash\n$ make lint\n```\n### Run tox over make\n```bash\n$ make tox\n```\n',
    'author': 'Britodfbr',
    'author_email': 'britodfbr@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.8,<4',
}


setup(**setup_kwargs)
