"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyDocument = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const policy_statement_1 = require("./policy-statement");
/**
 * A PolicyDocument is a collection of statements.
 *
 * @stability stable
 */
class PolicyDocument {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        this.statements = new Array();
        this.creationStack = cdk.captureStackTrace();
        this.autoAssignSids = !!props.assignSids;
        this.addStatements(...props.statements || []);
    }
    /**
     * Creates a new PolicyDocument based on the object provided.
     *
     * This will accept an object created from the `.toJSON()` call
     *
     * @param obj the PolicyDocument in object form.
     * @stability stable
     */
    static fromJson(obj) {
        var _b;
        const newPolicyDocument = new PolicyDocument();
        const statement = (_b = obj.Statement) !== null && _b !== void 0 ? _b : [];
        if (statement && !Array.isArray(statement)) {
            throw new Error('Statement must be an array');
        }
        newPolicyDocument.addStatements(...obj.Statement.map((s) => policy_statement_1.PolicyStatement.fromJson(s)));
        return newPolicyDocument;
    }
    /**
     * Produce the Token's value at resolution time.
     *
     * @stability stable
     */
    resolve(context) {
        context.registerPostProcessor(new RemoveDuplicateStatements(this.autoAssignSids));
        return this.render();
    }
    /**
     * Whether the policy document contains any statements.
     *
     * @stability stable
     */
    get isEmpty() {
        return this.statements.length === 0;
    }
    /**
     * The number of statements already added to this policy.
     *
     * Can be used, for example, to generate unique "sid"s within the policy.
     *
     * @stability stable
     */
    get statementCount() {
        return this.statements.length;
    }
    /**
     * Adds a statement to the policy document.
     *
     * @param statement the statement to add.
     * @stability stable
     */
    addStatements(...statement) {
        this.statements.push(...statement);
    }
    /**
     * Encode the policy document as a string.
     *
     * @stability stable
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyDocument',
        });
    }
    /**
     * JSON-ify the document.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toJSON() {
        return this.render();
    }
    /**
     * Validate that all policy statements in the policy document satisfies the requirements for any policy.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#access_policies-json
     * @stability stable
     */
    validateForAnyPolicy() {
        const errors = new Array();
        for (const statement of this.statements) {
            errors.push(...statement.validateForAnyPolicy());
        }
        return errors;
    }
    /**
     * Validate that all policy statements in the policy document satisfies the requirements for a resource-based policy.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#access_policies-json
     * @stability stable
     */
    validateForResourcePolicy() {
        const errors = new Array();
        for (const statement of this.statements) {
            errors.push(...statement.validateForResourcePolicy());
        }
        return errors;
    }
    /**
     * Validate that all policy statements in the policy document satisfies the requirements for an identity-based policy.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#access_policies-json
     * @stability stable
     */
    validateForIdentityPolicy() {
        const errors = new Array();
        for (const statement of this.statements) {
            errors.push(...statement.validateForIdentityPolicy());
        }
        return errors;
    }
    render() {
        if (this.isEmpty) {
            return undefined;
        }
        const doc = {
            Statement: this.statements.map(s => s.toStatementJson()),
            Version: '2012-10-17',
        };
        return doc;
    }
}
exports.PolicyDocument = PolicyDocument;
_a = JSII_RTTI_SYMBOL_1;
PolicyDocument[_a] = { fqn: "@aws-cdk/aws-iam.PolicyDocument", version: "1.109.0" };
/**
 * Removes duplicate statements and assign Sids if necessary
 */
class RemoveDuplicateStatements {
    constructor(autoAssignSids) {
        this.autoAssignSids = autoAssignSids;
    }
    postProcess(input, _context) {
        if (!input || !input.Statement) {
            return input;
        }
        const jsonStatements = new Set();
        const uniqueStatements = [];
        for (const statement of input.Statement) {
            const jsonStatement = JSON.stringify(statement);
            if (!jsonStatements.has(jsonStatement)) {
                uniqueStatements.push(statement);
                jsonStatements.add(jsonStatement);
            }
        }
        // assign unique SIDs (the statement index) if `autoAssignSids` is enabled
        const statements = uniqueStatements.map((s, i) => {
            if (this.autoAssignSids && !s.Sid) {
                s.Sid = i.toString();
            }
            return s;
        });
        return {
            ...input,
            Statement: statements,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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