import { Duration, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Grant } from './grant';
import { IIdentity } from './identity-base';
import { IManagedPolicy } from './managed-policy';
import { Policy } from './policy';
import { PolicyDocument } from './policy-document';
import { PolicyStatement } from './policy-statement';
import { AddToPrincipalPolicyResult, IPrincipal, PrincipalPolicyFragment } from './principals';
/**
 * Properties for defining an IAM Role.
 *
 * @stability stable
 */
export interface RoleProps {
    /**
     * The IAM principal (i.e. `new ServicePrincipal('sns.amazonaws.com')`) which can assume this role.
     *
     * You can later modify the assume role policy document by accessing it via
     * the `assumeRolePolicy` property.
     *
     * @stability stable
     */
    readonly assumedBy: IPrincipal;
    /**
     * (deprecated) ID that the role assumer needs to provide when assuming this role.
     *
     * If the configured and provided external IDs do not match, the
     * AssumeRole operation will fail.
     *
     * @default No external ID required
     * @deprecated see {@link externalIds}
     */
    readonly externalId?: string;
    /**
     * List of IDs that the role assumer needs to provide one of when assuming this role.
     *
     * If the configured and provided external IDs do not match, the
     * AssumeRole operation will fail.
     *
     * @default No external ID required
     * @stability stable
     */
    readonly externalIds?: string[];
    /**
     * A list of managed policies associated with this role.
     *
     * You can add managed policies later using
     * `addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName(policyName))`.
     *
     * @default - No managed policies.
     * @stability stable
     */
    readonly managedPolicies?: IManagedPolicy[];
    /**
     * A list of named policies to inline into this role.
     *
     * These policies will be
     * created with the role, whereas those added by ``addToPolicy`` are added
     * using a separate CloudFormation resource (allowing a way around circular
     * dependencies that could otherwise be introduced).
     *
     * @default - No policy is inlined in the Role resource.
     * @stability stable
     */
    readonly inlinePolicies?: {
        [name: string]: PolicyDocument;
    };
    /**
     * The path associated with this role.
     *
     * For information about IAM paths, see
     * Friendly Names and Paths in IAM User Guide.
     *
     * @default /
     * @stability stable
     */
    readonly path?: string;
    /**
     * AWS supports permissions boundaries for IAM entities (users or roles).
     *
     * A permissions boundary is an advanced feature for using a managed policy
     * to set the maximum permissions that an identity-based policy can grant to
     * an IAM entity. An entity's permissions boundary allows it to perform only
     * the actions that are allowed by both its identity-based policies and its
     * permissions boundaries.
     *
     * @default - No permissions boundary.
     * @stability stable
     * @link https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html
     */
    readonly permissionsBoundary?: IManagedPolicy;
    /**
     * A name for the IAM role.
     *
     * For valid values, see the RoleName parameter for
     * the CreateRole action in the IAM API Reference.
     *
     * IMPORTANT: If you specify a name, you cannot perform updates that require
     * replacement of this resource. You can perform updates that require no or
     * some interruption. If you must replace the resource, specify a new name.
     *
     * If you specify a name, you must specify the CAPABILITY_NAMED_IAM value to
     * acknowledge your template's capabilities. For more information, see
     * Acknowledging IAM Resources in AWS CloudFormation Templates.
     *
     * @default - AWS CloudFormation generates a unique physical ID and uses that ID
     * for the role name.
     * @stability stable
     */
    readonly roleName?: string;
    /**
     * The maximum session duration that you want to set for the specified role.
     *
     * This setting can have a value from 1 hour (3600sec) to 12 (43200sec) hours.
     *
     * Anyone who assumes the role from the AWS CLI or API can use the
     * DurationSeconds API parameter or the duration-seconds CLI parameter to
     * request a longer session. The MaxSessionDuration setting determines the
     * maximum duration that can be requested using the DurationSeconds
     * parameter.
     *
     * If users don't specify a value for the DurationSeconds parameter, their
     * security credentials are valid for one hour by default. This applies when
     * you use the AssumeRole* API operations or the assume-role* CLI operations
     * but does not apply when you use those operations to create a console URL.
     *
     * @default Duration.hours(1)
     * @stability stable
     * @link https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use.html
     */
    readonly maxSessionDuration?: Duration;
    /**
     * A description of the role.
     *
     * It can be up to 1000 characters long.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
}
/**
 * Options allowing customizing the behavior of {@link Role.fromRoleArn}.
 *
 * @stability stable
 */
export interface FromRoleArnOptions {
    /**
     * Whether the imported role can be modified by attaching policy resources to it.
     *
     * @default true
     * @stability stable
     */
    readonly mutable?: boolean;
}
/**
 * IAM Role.
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 *
 * @stability stable
 */
export declare class Role extends Resource implements IRole {
    /**
     * Import an external role by ARN.
     *
     * If the imported Role ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * role has a `path` (like `arn:...:role/AdminRoles/Alice`), the
     * `roleName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Role ARN should be supplied without the
     * `path` in order to resolve the correct role resource.
     *
     * @param scope construct scope.
     * @param id construct id.
     * @param roleArn the ARN of the role to import.
     * @param options allow customizing the behavior of the returned role.
     * @stability stable
     */
    static fromRoleArn(scope: Construct, id: string, roleArn: string, options?: FromRoleArnOptions): IRole;
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The AWS account ID of this principal.
     *
     * Can be undefined when the account is not known
     * (for example, for service principals).
     * Can be a Token - in that case,
     * it's assumed to be AWS::AccountId.
     *
     * @stability stable
     */
    readonly principalAccount: string | undefined;
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     *
     * @stability stable
     */
    readonly assumeRoleAction: string;
    /**
     * The assume role policy document associated with this role.
     *
     * @stability stable
     */
    readonly assumeRolePolicy?: PolicyDocument;
    /**
     * Returns the ARN of this role.
     *
     * @stability stable
     */
    readonly roleArn: string;
    /**
     * Returns the stable and unique string identifying the role.
     *
     * For example,
     * AIDAJQABLZS4A3QDU576Q.
     *
     * @stability stable
     * @attribute true
     */
    readonly roleId: string;
    /**
     * Returns the name of the role.
     *
     * @stability stable
     */
    readonly roleName: string;
    /**
     * Returns the role.
     *
     * @stability stable
     */
    readonly policyFragment: PrincipalPolicyFragment;
    /**
     * Returns the permissions boundary attached to this role.
     *
     * @stability stable
     */
    readonly permissionsBoundary?: IManagedPolicy;
    private defaultPolicy?;
    private readonly managedPolicies;
    private readonly attachedPolicies;
    private readonly inlinePolicies;
    private immutableRole?;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: RoleProps);
    /**
     * Adds a permission to the role's default policy document.
     *
     * If there is no default policy attached to this role, it will be created.
     *
     * @param statement The permission statement to add to the policy document.
     * @stability stable
     */
    addToPrincipalPolicy(statement: PolicyStatement): AddToPrincipalPolicyResult;
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPolicy(statement: PolicyStatement): boolean;
    /**
     * Attaches a managed policy to this role.
     *
     * @param policy The the managed policy to attach.
     * @stability stable
     */
    addManagedPolicy(policy: IManagedPolicy): void;
    /**
     * Attaches a policy to this role.
     *
     * @param policy The policy to attach.
     * @stability stable
     */
    attachInlinePolicy(policy: Policy): void;
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     *
     * @stability stable
     */
    grant(grantee: IPrincipal, ...actions: string[]): Grant;
    /**
     * Grant permissions to the given principal to pass this role.
     *
     * @stability stable
     */
    grantPassRole(identity: IPrincipal): Grant;
    /**
     * Return a copy of this Role object whose Policies will not be updated.
     *
     * Use the object returned by this method if you want this Role to be used by
     * a construct without it automatically updating the Role's Policies.
     *
     * If you do, you are responsible for adding the correct statements to the
     * Role's policies yourself.
     *
     * @stability stable
     */
    withoutPolicyUpdates(): IRole;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    protected validate(): string[];
}
/**
 * A Role object.
 *
 * @stability stable
 */
export interface IRole extends IIdentity {
    /**
     * Returns the ARN of this role.
     *
     * @stability stable
     * @attribute true
     */
    readonly roleArn: string;
    /**
     * Returns the name of this role.
     *
     * @stability stable
     * @attribute true
     */
    readonly roleName: string;
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     *
     * @stability stable
     */
    grant(grantee: IPrincipal, ...actions: string[]): Grant;
    /**
     * Grant permissions to the given principal to pass this role.
     *
     * @stability stable
     */
    grantPassRole(grantee: IPrincipal): Grant;
}
