"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.listExecutions = void 0;
exports.getExecutionHistory = getExecutionHistory;
const client_sfn_1 = require("@aws-sdk/client-sfn");
const sfnClient = new client_sfn_1.SFNClient({ region: 'us-east-2' });
/**
 * Fetches a list of executions for a given state machine.
 *
 * @param {string} stateMachineArn - The ARN of the state machine.
 * @return {Promise<Array>} - A promise that resolves to an array of execution objects.
 * @throws {Error} - If an error occurs while fetching the list of executions.
 * @example
 * const _stateMachineArn = 'arn:aws:states:{{Region}}:{{AccountID}}:stateMachine:MyStateMachine-oiye9p0di';
 * const _executionName = '3db0f8ae-fab8-4c77-bb61-3f3b48ab39e1';
 * listExecutions(_stateMachineArn).then(executions => {
 *   const filteredExecutions = executions!.filter(execution => execution.name!.includes(_executionName + '-'));
 *   console.log(filteredExecutions);
 * }).catch(error => {
 *   console.log(error);
 * });
 */
const listExecutions = async (stateMachineArn) => {
    const command = new client_sfn_1.ListExecutionsCommand({ stateMachineArn: stateMachineArn });
    try {
        const response = await sfnClient.send(command);
        return response.executions;
    }
    catch (err) {
        return [];
    }
};
exports.listExecutions = listExecutions;
/**
 * Retrieves the execution history of a specific execution using the given execution ARN.
 *
 * @param {string} executionArn - The ARN of the execution.
 *
 * @returns {Promise<string>} - The ID of the AMI associated with the execution if it succeeded, otherwise an empty string.
 */
async function getExecutionHistory(executionArn) {
    const params = {
        executionArn: executionArn,
        reverseOrder: true,
    };
    const command = new client_sfn_1.GetExecutionHistoryCommand(params);
    const history = await sfnClient.send(command);
    for (const event of history.events) {
        if (event.type === client_sfn_1.HistoryEventType.ExecutionSucceeded) {
            let outputJson = JSON.parse(event.executionSucceededEventDetails.output || '{}');
            if (outputJson.ImageBuildInformation.Image.OutputResources.Amis.length > 0) {
                let amiId = outputJson.ImageBuildInformation.Image.OutputResources.Amis[0].Image;
                return amiId;
            }
            else {
                return '';
            }
        }
    }
    return '';
}
//# sourceMappingURL=data:application/json;base64,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