"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/cdk");
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends cdk.Construct {
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context) {
        const root = context.node.stack;
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId);
        }
        return lambda.functionArn;
    }
    constructor(scope, id) {
        super(scope, id);
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicyArns: [
                this.node.stack.formatArn({
                    service: 'iam',
                    region: '',
                    account: 'aws',
                    resource: 'policy',
                    resourceName: 'service-role/AWSLambdaBasicExecutionRole',
                })
            ]
        });
        // handler allows to put bucket notification on s3 buckets.
        role.addToPolicy(new iam.PolicyStatement()
            .addAction('s3:PutBucketNotification')
            .addAllResources());
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.Standard, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: `exports.handler = ${handler.toString()};` },
                Handler: 'index.handler',
                Role: role.roleArn,
                Runtime: 'nodejs8.10',
                Timeout: 300,
            }
        });
        this.functionArn = resource.getAtt('Arn').toString();
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
// tslint:disable:no-console
/**
 * Lambda event handler for the custom resource. Bear in mind that we are going
 * to .toString() this function and inline it as Lambda code.
 *
 * The function will issue a putBucketNotificationConfiguration request for the
 * specified bucket.
 */
const handler = (event, context) => {
    const s3 = new (require('aws-sdk').S3)();
    const https = require("https");
    const url = require("url");
    log(JSON.stringify(event, undefined, 2));
    const props = event.ResourceProperties;
    if (event.RequestType === 'Delete') {
        props.NotificationConfiguration = {}; // this is how you clean out notifications
    }
    const req = {
        Bucket: props.BucketName,
        NotificationConfiguration: props.NotificationConfiguration
    };
    return s3.putBucketNotificationConfiguration(req, (err, data) => {
        log({ err, data });
        if (err) {
            return submitResponse("FAILED", err.message + `\nMore information in CloudWatch Log Stream: ${context.logStreamName}`);
        }
        else {
            return submitResponse("SUCCESS");
        }
    });
    function log(obj) {
        console.error(event.RequestId, event.StackId, event.LogicalResourceId, obj);
    }
    // tslint:disable-next-line:max-line-length
    // adapted from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-cfnresponsemodule
    // to allow sending an error messge as a reason.
    function submitResponse(responseStatus, reason) {
        const responseBody = JSON.stringify({
            Status: responseStatus,
            Reason: reason || "See the details in CloudWatch Log Stream: " + context.logStreamName,
            PhysicalResourceId: context.logStreamName,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            NoEcho: false,
        });
        log({ responseBody });
        const parsedUrl = url.parse(event.ResponseURL);
        const options = {
            hostname: parsedUrl.hostname,
            port: 443,
            path: parsedUrl.path,
            method: "PUT",
            headers: {
                "content-type": "",
                "content-length": responseBody.length
            }
        };
        const request = https.request(options, (r) => {
            log({ statusCode: r.statusCode, statusMessage: r.statusMessage });
            context.done();
        });
        request.on("error", (error) => {
            log({ sendError: error });
            context.done();
        });
        request.write(responseBody);
        request.end();
    }
};
//# sourceMappingURL=data:application/json;base64,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