"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_s3_notifications_1 = require("@aws-cdk/aws-s3-notifications");
const cdk = require("@aws-cdk/cdk");
const notifications_resource_handler_1 = require("./notifications-resource-handler");
/**
 * A custom CloudFormation resource that updates bucket notifications for a
 * bucket. The reason we need it is because the AWS::S3::Bucket notification
 * configuration is defined on the bucket itself, which makes it impossible to
 * provision notifications at the same time as the target (since
 * PutBucketNotifications validates the targets).
 *
 * Since only a single BucketNotifications resource is allowed for each Bucket,
 * this construct is not exported in the public API of this module. Instead, it
 * is created just-in-time by `s3.Bucket.onEvent`, so a 1:1 relationship is
 * ensured.
 *
 * @see
 * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-notificationconfig.html
 */
class BucketNotifications extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaNotifications = new Array();
        this.queueNotifications = new Array();
        this.topicNotifications = new Array();
        this.bucket = props.bucket;
    }
    /**
     * Adds a notification subscription for this bucket.
     * If this is the first notification, a BucketNotification resource is added to the stack.
     *
     * @param event The type of event
     * @param target The target construct
     * @param filters A set of S3 key filters
     */
    addNotification(event, target, ...filters) {
        const resource = this.createResourceOnce();
        // resolve target. this also provides an opportunity for the target to e.g. update
        // policies to allow this notification to happen.
        const targetProps = target.asBucketNotificationDestination(this.bucket.bucketArn, this.bucket.node.uniqueId);
        const commonConfig = {
            Events: [event],
            Filter: renderFilters(filters),
        };
        // if the target specifies any dependencies, add them to the custom resource.
        // for example, the SNS topic policy must be created /before/ the notification resource.
        // otherwise, S3 won't be able to confirm the subscription.
        if (targetProps.dependencies) {
            resource.node.addDependency(...targetProps.dependencies);
        }
        // based on the target type, add the the correct configurations array
        switch (targetProps.type) {
            case aws_s3_notifications_1.BucketNotificationDestinationType.Lambda:
                this.lambdaNotifications.push({ ...commonConfig, LambdaFunctionArn: targetProps.arn });
                break;
            case aws_s3_notifications_1.BucketNotificationDestinationType.Queue:
                this.queueNotifications.push({ ...commonConfig, QueueArn: targetProps.arn });
                break;
            case aws_s3_notifications_1.BucketNotificationDestinationType.Topic:
                this.topicNotifications.push({ ...commonConfig, TopicArn: targetProps.arn });
                break;
            default:
                throw new Error('Unsupported notification target type:' + aws_s3_notifications_1.BucketNotificationDestinationType[targetProps.type]);
        }
    }
    renderNotificationConfiguration() {
        return {
            LambdaFunctionConfigurations: this.lambdaNotifications.length > 0 ? this.lambdaNotifications : undefined,
            QueueConfigurations: this.queueNotifications.length > 0 ? this.queueNotifications : undefined,
            TopicConfigurations: this.topicNotifications.length > 0 ? this.topicNotifications : undefined
        };
    }
    /**
     * Defines the bucket notifications resources in the stack only once.
     * This is called lazily as we add notifications, so that if notifications are not added,
     * there is no notifications resource.
     */
    createResourceOnce() {
        if (!this.resource) {
            const handlerArn = notifications_resource_handler_1.NotificationsResourceHandler.singleton(this);
            this.resource = new cdk.CfnResource(this, 'Resource', {
                type: 'Custom::S3BucketNotifications',
                properties: {
                    ServiceToken: handlerArn,
                    BucketName: this.bucket.bucketName,
                    NotificationConfiguration: new cdk.Token(() => this.renderNotificationConfiguration())
                }
            });
        }
        return this.resource;
    }
}
exports.BucketNotifications = BucketNotifications;
function renderFilters(filters) {
    if (!filters || filters.length === 0) {
        return undefined;
    }
    const renderedRules = new Array();
    for (const rule of filters) {
        if (!rule.suffix && !rule.prefix) {
            throw new Error('NotificationKeyFilter must specify `prefix` and/or `suffix`');
        }
        if (rule.suffix) {
            renderedRules.push({ Name: 'suffix', Value: rule.suffix });
        }
        if (rule.prefix) {
            renderedRules.push({ Name: 'prefix', Value: rule.prefix });
        }
    }
    return {
        Key: {
            FilterRules: renderedRules
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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