"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3n = require("@aws-cdk/aws-s3-notifications");
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
const s3 = require("../lib");
const notification_dests_1 = require("./notification-dests");
module.exports = {
    'bucket without notifications'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                }
            }
        });
        test.done();
    },
    'when notification are added, a custom resource is provisioned + a lambda handler for it'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        const topic = new notification_dests_1.Topic(stack, 'MyTopic');
        bucket.onEvent(s3.EventType.ObjectCreated, topic);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', { Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)' }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications'));
        test.done();
    },
    'when notification are added, you can tag the lambda'(test) {
        const stack = new cdk.Stack();
        stack.node.apply(new cdk.Tag('Lambda', 'AreTagged'));
        const bucket = new s3.Bucket(stack, 'MyBucket');
        const topic = new notification_dests_1.Topic(stack, 'MyTopic');
        bucket.onEvent(s3.EventType.ObjectCreated, topic);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Tags: [{ Key: 'Lambda', Value: 'AreTagged' }],
            Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications'));
        test.done();
    },
    'bucketNotificationTarget is not called during synthesis'(test) {
        const stack = new cdk.Stack();
        // notice the order here - topic is defined before bucket
        // but this shouldn't impact the fact that the topic policy includes
        // the bucket information
        const topic = new notification_dests_1.Topic(stack, 'Topic');
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.onObjectCreated(topic);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::TopicPolicy', {
            "Topics": [
                {
                    "Ref": "TopicBFC7AF6E"
                }
            ],
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": "sns:Publish",
                        "Condition": {
                            "ArnLike": {
                                "aws:SourceArn": {
                                    "Fn::GetAtt": [
                                        "MyBucketF68F3FF0",
                                        "Arn"
                                    ]
                                }
                            }
                        },
                        "Effect": "Allow",
                        "Principal": {
                            "Service": { "Fn::Join": ["", ["s3.", { Ref: "AWS::URLSuffix" }]] }
                        },
                        "Resource": {
                            "Ref": "TopicBFC7AF6E"
                        },
                        "Sid": "sid0"
                    }
                ],
                "Version": "2012-10-17"
            }
        }));
        test.done();
    },
    'subscription types'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'TestBucket');
        const queueTarget = {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Queue,
                arn: 'arn:aws:sqs:...'
            })
        };
        const lambdaTarget = {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Lambda,
                arn: 'arn:aws:lambda:...'
            })
        };
        const topicTarget = {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Topic,
                arn: 'arn:aws:sns:...'
            })
        };
        bucket.onEvent(s3.EventType.ObjectCreated, queueTarget);
        bucket.onEvent(s3.EventType.ObjectCreated, lambdaTarget);
        bucket.onObjectRemoved(topicTarget, { prefix: 'prefix' });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691",
                    "Arn"
                ]
            },
            "BucketName": {
                "Ref": "TestBucket560B80BC"
            },
            "NotificationConfiguration": {
                "LambdaFunctionConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectCreated:*"
                        ],
                        "LambdaFunctionArn": "arn:aws:lambda:..."
                    }
                ],
                "QueueConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectCreated:*"
                        ],
                        "QueueArn": "arn:aws:sqs:..."
                    }
                ],
                "TopicConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectRemoved:*"
                        ],
                        "TopicArn": "arn:aws:sns:...",
                        "Filter": {
                            "Key": {
                                "FilterRules": [
                                    {
                                        "Name": "prefix",
                                        "Value": "prefix"
                                    }
                                ]
                            }
                        }
                    }
                ]
            }
        }));
        test.done();
    },
    'multiple subscriptions of the same type'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'TestBucket');
        bucket.onEvent(s3.EventType.ObjectRemovedDelete, {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Queue,
                arn: 'arn:aws:sqs:...:queue1'
            })
        });
        bucket.onEvent(s3.EventType.ObjectRemovedDelete, {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Queue,
                arn: 'arn:aws:sqs:...:queue2'
            })
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691",
                    "Arn"
                ]
            },
            "BucketName": {
                "Ref": "TestBucket560B80BC"
            },
            "NotificationConfiguration": {
                "QueueConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectRemoved:Delete"
                        ],
                        "QueueArn": "arn:aws:sqs:...:queue1"
                    },
                    {
                        "Events": [
                            "s3:ObjectRemoved:Delete"
                        ],
                        "QueueArn": "arn:aws:sqs:...:queue2"
                    }
                ]
            }
        }));
        test.done();
    },
    'prefix/suffix filters'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'TestBucket');
        const bucketNotificationTarget = {
            type: s3n.BucketNotificationDestinationType.Queue,
            arn: 'arn:aws:sqs:...'
        };
        bucket.onEvent(s3.EventType.ObjectRemovedDelete, { asBucketNotificationDestination: _ => bucketNotificationTarget }, { prefix: 'images/', suffix: '.jpg' });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691",
                    "Arn"
                ]
            },
            "BucketName": {
                "Ref": "TestBucket560B80BC"
            },
            "NotificationConfiguration": {
                "QueueConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectRemoved:Delete"
                        ],
                        "Filter": {
                            "Key": {
                                "FilterRules": [
                                    {
                                        "Name": "suffix",
                                        "Value": ".jpg"
                                    },
                                    {
                                        "Name": "prefix",
                                        "Value": "images/"
                                    }
                                ]
                            }
                        },
                        "QueueArn": "arn:aws:sqs:..."
                    }
                ]
            }
        }));
        test.done();
    },
    'a notification destination can specify a set of dependencies that must be resolved before the notifications resource is created'(test) {
        const stack = new cdk_1.Stack();
        const bucket = new s3.Bucket(stack, 'Bucket');
        const dependent = new cdk.CfnResource(stack, 'Dependent', { type: 'DependOnMe' });
        const dest = {
            asBucketNotificationDestination: () => ({
                arn: 'arn',
                type: s3n.BucketNotificationDestinationType.Queue,
                dependencies: [dependent]
            })
        };
        bucket.onObjectCreated(dest);
        stack.node.prepareTree();
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Resources.BucketNotifications8F2E257D, {
            Type: 'Custom::S3BucketNotifications',
            Properties: {
                ServiceToken: { 'Fn::GetAtt': ['BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691', 'Arn'] },
                BucketName: { Ref: 'Bucket83908E77' },
                NotificationConfiguration: { QueueConfigurations: [{ Events: ['s3:ObjectCreated:*'], QueueArn: 'arn' }] }
            },
            DependsOn: ['Dependent']
        });
        test.done();
    },
    'CloudWatch Events': {
        'onPutItem contains the Bucket ARN itself when path is undefined'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
                bucketName: 'MyBucket',
            });
            bucket.onPutObject('PutRule');
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.s3",
                    ],
                    "detail": {
                        "eventSource": [
                            "s3.amazonaws.com",
                        ],
                        "eventName": [
                            "PutObject",
                        ],
                        "resources": {
                            "ARN": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":s3:::MyBucket",
                                        ],
                                    ],
                                },
                            ],
                        },
                    },
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        "onPutItem contains the path when it's provided"(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
                bucketName: 'MyBucket',
            });
            bucket.onPutObject('PutRule', undefined, 'my/path.zip');
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.s3",
                    ],
                    "detail": {
                        "eventSource": [
                            "s3.amazonaws.com",
                        ],
                        "eventName": [
                            "PutObject",
                        ],
                        "resources": {
                            "ARN": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":s3:::MyBucket/my/path.zip"
                                        ],
                                    ],
                                },
                            ],
                        },
                    },
                },
                "State": "ENABLED",
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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