"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
const ESBUILD_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        this.props = props;
        Bundling.runsLocally = (_c = (_a = Bundling.runsLocally) !== null && _a !== void 0 ? _a : (_b = util_1.getEsBuildVersion()) === null || _b === void 0 ? void 0 : _b.startsWith(ESBUILD_VERSION)) !== null && _c !== void 0 ? _c : false;
        const projectRoot = path.dirname(props.depsLockFilePath);
        this.relativeEntryPath = path.relative(projectRoot, path.resolve(props.entry));
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(projectRoot, path.resolve(props.tsconfig));
        }
        this.externals = [
            ...(_d = props.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'],
            ...(_e = props.nodeModules) !== null && _e !== void 0 ? _e : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? (_f = props.dockerImage) !== null && _f !== void 0 ? _f : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_g = props.buildArgs) !== null && _g !== void 0 ? _g : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    ESBUILD_VERSION: (_h = props.esbuildVersion) !== null && _h !== void 0 ? _h : ESBUILD_VERSION,
                },
            }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    var _a;
                    if (Bundling.runsLocally === false) {
                        process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...(_a = props.environment) !== null && _a !== void 0 ? _a : {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: path.dirname(props.entry),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(path.dirname(options.depsLockFilePath), {
            assetHashType: cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        const pathJoin = osPathJoin(osPlatform);
        const npx = osPlatform === 'win32' ? 'npx.cmd' : 'npx';
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const defines = Object.entries((_b = this.props.define) !== null && _b !== void 0 ? _b : {});
        const esbuildCommand = [
            npx, 'esbuild',
            '--bundle', `"${pathJoin(inputDir, this.relativeEntryPath)}"`,
            `--target=${(_c = this.props.target) !== null && _c !== void 0 ? _c : toTarget(this.props.runtime)}`,
            '--platform=node',
            `--outfile="${pathJoin(outputDir, 'index.js')}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...this.props.sourceMap ? ['--sourcemap'] : [],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
        ].join(' ');
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            let installer = Installer.NPM;
            let lockFile = util_1.LockFile.NPM;
            if (this.props.depsLockFilePath.endsWith(util_1.LockFile.YARN)) {
                lockFile = util_1.LockFile.YARN;
                installer = Installer.YARN;
            }
            const osCommand = new OsCommand(osPlatform);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(outputDir, 'package.json'), { dependencies }),
                osCommand.copy(pathJoin(inputDir, lockFile), pathJoin(outputDir, lockFile)),
                osCommand.changeDirectory(outputDir),
                `${installer} install`,
            ]);
        }
        return chain([
            ...(_e = (_d = this.props.commandHooks) === null || _d === void 0 ? void 0 : _d.beforeBundling(inputDir, outputDir)) !== null && _e !== void 0 ? _e : [],
            esbuildCommand,
            ...(_g = (this.props.nodeModules && ((_f = this.props.commandHooks) === null || _f === void 0 ? void 0 : _f.beforeInstall(inputDir, outputDir)))) !== null && _g !== void 0 ? _g : [],
            depsCommand,
            ...(_j = (_h = this.props.commandHooks) === null || _h === void 0 ? void 0 : _h.afterBundling(inputDir, outputDir)) !== null && _j !== void 0 ? _j : [],
        ]);
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > ${filePath}`;
        }
        return `echo '${stringifiedData}' > ${filePath}`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy ${src} ${dest}`;
        }
        return `cp ${src} ${dest}`;
    }
    changeDirectory(dir) {
        return `cd ${dir}`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLG9EQUErRDtBQUMvRCxxQ0FBcUM7QUFFckMsaUNBQXdGO0FBRXhGLE1BQU0sZUFBZSxHQUFHLEdBQUcsQ0FBQztBQXNCNUI7O0dBRUc7QUFDSCxNQUFhLFFBQVE7SUE0Qm5CLFlBQTZCLEtBQW9COztRQUFwQixVQUFLLEdBQUwsS0FBSyxDQUFlO1FBQy9DLFFBQVEsQ0FBQyxXQUFXLGVBQUcsUUFBUSxDQUFDLFdBQVcseUNBQ3RDLHdCQUFpQixFQUFFLDBDQUFFLFVBQVUsQ0FBQyxlQUFlLG9DQUMvQyxLQUFLLENBQUM7UUFFWCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBRS9FLElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNsQixJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztTQUN0RjtRQUVELElBQUksQ0FBQyxTQUFTLEdBQUc7WUFDZixTQUFHLEtBQUssQ0FBQyxlQUFlLG1DQUFJLENBQUMsU0FBUyxDQUFDO1lBQ3ZDLFNBQUcsS0FBSyxDQUFDLFdBQVcsbUNBQUksRUFBRTtTQUMzQixDQUFDO1FBRUYsa0JBQWtCO1FBQ2xCLE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQztRQUM1RSxJQUFJLENBQUMsS0FBSyxHQUFHLGdCQUFnQjtZQUMzQixDQUFDLE9BQUMsS0FBSyxDQUFDLFdBQVcsbUNBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLEVBQUU7Z0JBQy9FLFNBQVMsRUFBRTtvQkFDVCxTQUFHLEtBQUssQ0FBQyxTQUFTLG1DQUFJLEVBQUU7b0JBQ3hCLEtBQUssRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLG1CQUFtQixDQUFDLEtBQUs7b0JBQzlDLGVBQWUsUUFBRSxLQUFLLENBQUMsY0FBYyxtQ0FBSSxlQUFlO2lCQUN6RDthQUNGLENBQUMsQ0FDRixDQUFDLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxtQ0FBbUM7UUFFOUUsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDLHFCQUFxQixDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlILElBQUksQ0FBQyxPQUFPLEdBQUcsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyw2RUFBNkU7UUFDN0Usa0VBQWtFO1FBQ2xFLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxHQUFHLENBQUM7UUFFNUIsaUJBQWlCO1FBQ2pCLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLEVBQUUsRUFBRSwrQkFBK0I7WUFDL0QsTUFBTSxVQUFVLEdBQUcsRUFBRSxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2pDLE1BQU0sa0JBQWtCLEdBQUcsQ0FBQyxTQUFpQixFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsV0FBVyxFQUFFLFNBQVMsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUVqSCxJQUFJLENBQUMsS0FBSyxHQUFHO2dCQUNYLFNBQVMsQ0FBQyxTQUFpQjs7b0JBQ3pCLElBQUksUUFBUSxDQUFDLFdBQVcsS0FBSyxLQUFLLEVBQUU7d0JBQ2xDLE9BQU8sQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLDZEQUE2RCxDQUFDLENBQUM7d0JBQ3BGLE9BQU8sS0FBSyxDQUFDO3FCQUNkO29CQUVELE1BQU0sWUFBWSxHQUFHLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxDQUFDO29CQUVuRCxXQUFJLENBQ0YsVUFBVSxLQUFLLE9BQU8sQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxNQUFNLEVBQ3ZDO3dCQUNFLFVBQVUsS0FBSyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSTt3QkFDcEMsWUFBWTtxQkFDYixFQUNEO3dCQUNFLEdBQUcsRUFBRSxFQUFFLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxTQUFHLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEVBQUUsRUFBRTt3QkFDbkQsS0FBSyxFQUFFOzRCQUNMLFFBQVE7NEJBQ1IsT0FBTyxDQUFDLE1BQU07NEJBQ2QsU0FBUzt5QkFDVjt3QkFDRCxHQUFHLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDO3dCQUM5Qix3QkFBd0IsRUFBRSxVQUFVLEtBQUssT0FBTztxQkFDakQsQ0FBQyxDQUFDO29CQUVMLE9BQU8sSUFBSSxDQUFDO2dCQUNkLENBQUM7YUFDRixDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBbEdEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUFzQjtRQUN6QyxPQUFPLGlCQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLEVBQUU7WUFDNUQsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTTtZQUN2QyxRQUFRLEVBQUUsSUFBSSxRQUFRLENBQUMsT0FBTyxDQUFDO1NBQ2hDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSxNQUFNLENBQUMscUJBQXFCO1FBQ2pDLElBQUksQ0FBQyxXQUFXLEdBQUcsU0FBUyxDQUFDO0lBQy9CLENBQUM7SUF3Rk0scUJBQXFCLENBQUMsUUFBZ0IsRUFBRSxTQUFpQixFQUFFLGFBQThCLE9BQU87O1FBQ3JHLE1BQU0sUUFBUSxHQUFHLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUV4QyxNQUFNLEdBQUcsR0FBRyxVQUFVLEtBQUssT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQztRQUN2RCxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsT0FBTyxPQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxFQUFFLENBQUMsQ0FBQztRQUN4RCxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsT0FBTyxPQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxFQUFFLENBQUMsQ0FBQztRQUV4RCxNQUFNLGNBQWMsR0FBVztZQUM3QixHQUFHLEVBQUUsU0FBUztZQUNkLFVBQVUsRUFBRSxJQUFJLFFBQVEsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEdBQUc7WUFDN0QsWUFBWSxNQUFBLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUMvRCxpQkFBaUI7WUFDakIsY0FBYyxRQUFRLENBQUMsU0FBUyxFQUFFLFVBQVUsQ0FBQyxHQUFHO1lBQ2hELEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDeEMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUM5QyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsY0FBYyxRQUFRLEVBQUUsQ0FBQztZQUMzRCxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsWUFBWSxHQUFHLElBQUksSUFBSSxFQUFFLENBQUM7WUFDMUQsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDLFlBQVksR0FBRyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztZQUM1RSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ3BFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDL0MsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxRQUFRLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUNuRyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsUUFBUSxDQUFDLFNBQVMsRUFBRSxpQkFBaUIsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUN0RixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUNoRixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtTQUNqRixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVaLElBQUksV0FBVyxHQUFHLEVBQUUsQ0FBQztRQUNyQixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQzFCLDJFQUEyRTtZQUMzRSw0QkFBNEI7WUFDNUIsTUFBTSxPQUFPLEdBQUcsYUFBTSxDQUFDLGNBQWMsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUN2RSxJQUFJLENBQUMsT0FBTyxFQUFFO2dCQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsOEZBQThGLENBQUMsQ0FBQzthQUNqSDtZQUVELDJEQUEyRDtZQUMzRCxNQUFNLFlBQVksR0FBRywwQkFBbUIsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUMxRSxJQUFJLFNBQVMsR0FBRyxTQUFTLENBQUMsR0FBRyxDQUFDO1lBQzlCLElBQUksUUFBUSxHQUFHLGVBQVEsQ0FBQyxHQUFHLENBQUM7WUFDNUIsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUU7Z0JBQ3ZELFFBQVEsR0FBRyxlQUFRLENBQUMsSUFBSSxDQUFDO2dCQUN6QixTQUFTLEdBQUcsU0FBUyxDQUFDLElBQUksQ0FBQzthQUM1QjtZQUVELE1BQU0sU0FBUyxHQUFHLElBQUksU0FBUyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBRTVDLG9FQUFvRTtZQUNwRSxXQUFXLEdBQUcsS0FBSyxDQUFDO2dCQUNsQixTQUFTLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsY0FBYyxDQUFDLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQztnQkFDMUUsU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQyxFQUFFLFFBQVEsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQUM7Z0JBQzNFLFNBQVMsQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDO2dCQUNwQyxHQUFHLFNBQVMsVUFBVTthQUN2QixDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sS0FBSyxDQUFDO1lBQ1gsZUFBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksMENBQUUsY0FBYyxDQUFDLFFBQVEsRUFBRSxTQUFTLG9DQUFLLEVBQUU7WUFDckUsY0FBYztZQUNkLFNBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsV0FBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksMENBQUUsYUFBYSxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUMsQ0FBQyxtQ0FBSSxFQUFFO1lBQ2hHLFdBQVc7WUFDWCxlQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSwwQ0FBRSxhQUFhLENBQUMsUUFBUSxFQUFFLFNBQVMsb0NBQUssRUFBRTtTQUNyRSxDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUFwS0QsNEJBb0tDO0FBRUQsSUFBSyxTQUdKO0FBSEQsV0FBSyxTQUFTO0lBQ1osd0JBQVcsQ0FBQTtJQUNYLDBCQUFhLENBQUE7QUFDZixDQUFDLEVBSEksU0FBUyxLQUFULFNBQVMsUUFHYjtBQUVEOztHQUVHO0FBQ0gsTUFBTSxTQUFTO0lBQ2IsWUFBNkIsVUFBMkI7UUFBM0IsZUFBVSxHQUFWLFVBQVUsQ0FBaUI7SUFBRyxDQUFDO0lBRXJELFNBQVMsQ0FBQyxRQUFnQixFQUFFLElBQVM7UUFDMUMsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QyxJQUFJLElBQUksQ0FBQyxVQUFVLEtBQUssT0FBTyxFQUFFO1lBQy9CLE9BQU8sU0FBUyxlQUFlLE9BQU8sUUFBUSxFQUFFLENBQUM7U0FDbEQ7UUFFRCxPQUFPLFNBQVMsZUFBZSxPQUFPLFFBQVEsRUFBRSxDQUFDO0lBQ25ELENBQUM7SUFFTSxJQUFJLENBQUMsR0FBVyxFQUFFLElBQVk7UUFDbkMsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLE9BQU8sRUFBRTtZQUMvQixPQUFPLFFBQVEsR0FBRyxJQUFJLElBQUksRUFBRSxDQUFDO1NBQzlCO1FBRUQsT0FBTyxNQUFNLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztJQUM3QixDQUFDO0lBRU0sZUFBZSxDQUFDLEdBQVc7UUFDaEMsT0FBTyxNQUFNLEdBQUcsRUFBRSxDQUFDO0lBQ3JCLENBQUM7Q0FDRjtBQUVEOztHQUVHO0FBQ0gsU0FBUyxLQUFLLENBQUMsUUFBa0I7SUFDL0IsT0FBTyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUNoRCxDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLFVBQVUsQ0FBQyxRQUF5QjtJQUMzQyxPQUFPLFVBQVMsR0FBRyxLQUFlO1FBQ2hDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQztRQUNuQyxnREFBZ0Q7UUFDaEQsSUFBSSxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssT0FBTyxJQUFJLFFBQVEsS0FBSyxPQUFPLEVBQUU7WUFDckQsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztTQUNuQztRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUMsQ0FBQztBQUNKLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsUUFBUSxDQUFDLE9BQWdCO0lBQ2hDLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBRWhELElBQUksQ0FBQyxLQUFLLEVBQUU7UUFDVixNQUFNLElBQUksS0FBSyxDQUFDLHNDQUFzQyxDQUFDLENBQUM7S0FDekQ7SUFFRCxPQUFPLE9BQU8sS0FBSyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7QUFDM0IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIG9zIGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBBc3NldENvZGUsIENvZGUsIFJ1bnRpbWUgfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEJ1bmRsaW5nT3B0aW9ucyB9IGZyb20gJy4vdHlwZXMnO1xuaW1wb3J0IHsgZXhlYywgZXh0cmFjdERlcGVuZGVuY2llcywgZmluZFVwLCBnZXRFc0J1aWxkVmVyc2lvbiwgTG9ja0ZpbGUgfSBmcm9tICcuL3V0aWwnO1xuXG5jb25zdCBFU0JVSUxEX1ZFUlNJT04gPSAnMCc7XG5cbi8qKlxuICogQnVuZGxpbmcgcHJvcGVydGllc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEJ1bmRsaW5nUHJvcHMgZXh0ZW5kcyBCdW5kbGluZ09wdGlvbnMge1xuICAvKipcbiAgICogUGF0aCB0byBsb2NrIGZpbGVcbiAgICovXG4gIHJlYWRvbmx5IGRlcHNMb2NrRmlsZVBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogRW50cnkgZmlsZVxuICAgKi9cbiAgcmVhZG9ubHkgZW50cnk6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHJ1bnRpbWUgb2YgdGhlIGxhbWJkYSBmdW5jdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgcnVudGltZTogUnVudGltZTtcbn1cblxuLyoqXG4gKiBCdW5kbGluZyB3aXRoIGVzYnVpbGRcbiAqL1xuZXhwb3J0IGNsYXNzIEJ1bmRsaW5nIGltcGxlbWVudHMgY2RrLkJ1bmRsaW5nT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBlc2J1aWxkIGJ1bmRsZWQgTGFtYmRhIGFzc2V0IGNvZGVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYnVuZGxlKG9wdGlvbnM6IEJ1bmRsaW5nUHJvcHMpOiBBc3NldENvZGUge1xuICAgIHJldHVybiBDb2RlLmZyb21Bc3NldChwYXRoLmRpcm5hbWUob3B0aW9ucy5kZXBzTG9ja0ZpbGVQYXRoKSwge1xuICAgICAgYXNzZXRIYXNoVHlwZTogY2RrLkFzc2V0SGFzaFR5cGUuT1VUUFVULFxuICAgICAgYnVuZGxpbmc6IG5ldyBCdW5kbGluZyhvcHRpb25zKSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgY2xlYXJSdW5zTG9jYWxseUNhY2hlKCk6IHZvaWQge1xuICAgIHRoaXMucnVuc0xvY2FsbHkgPSB1bmRlZmluZWQ7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBydW5zTG9jYWxseT86IGJvb2xlYW47XG5cbiAgLy8gQ29yZSBidW5kbGluZyBvcHRpb25zXG4gIHB1YmxpYyByZWFkb25seSBpbWFnZTogY2RrLkRvY2tlckltYWdlO1xuICBwdWJsaWMgcmVhZG9ubHkgY29tbWFuZDogc3RyaW5nW107XG4gIHB1YmxpYyByZWFkb25seSBlbnZpcm9ubWVudD86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG4gIHB1YmxpYyByZWFkb25seSB3b3JraW5nRGlyZWN0b3J5OiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBsb2NhbD86IGNkay5JTG9jYWxCdW5kbGluZztcblxuICBwcml2YXRlIHJlYWRvbmx5IHJlbGF0aXZlRW50cnlQYXRoOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVsYXRpdmVUc2NvbmZpZ1BhdGg/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgZXh0ZXJuYWxzOiBzdHJpbmdbXTtcblxuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBCdW5kbGluZ1Byb3BzKSB7XG4gICAgQnVuZGxpbmcucnVuc0xvY2FsbHkgPSBCdW5kbGluZy5ydW5zTG9jYWxseVxuICAgICAgPz8gZ2V0RXNCdWlsZFZlcnNpb24oKT8uc3RhcnRzV2l0aChFU0JVSUxEX1ZFUlNJT04pXG4gICAgICA/PyBmYWxzZTtcblxuICAgIGNvbnN0IHByb2plY3RSb290ID0gcGF0aC5kaXJuYW1lKHByb3BzLmRlcHNMb2NrRmlsZVBhdGgpO1xuICAgIHRoaXMucmVsYXRpdmVFbnRyeVBhdGggPSBwYXRoLnJlbGF0aXZlKHByb2plY3RSb290LCBwYXRoLnJlc29sdmUocHJvcHMuZW50cnkpKTtcblxuICAgIGlmIChwcm9wcy50c2NvbmZpZykge1xuICAgICAgdGhpcy5yZWxhdGl2ZVRzY29uZmlnUGF0aCA9IHBhdGgucmVsYXRpdmUocHJvamVjdFJvb3QsIHBhdGgucmVzb2x2ZShwcm9wcy50c2NvbmZpZykpO1xuICAgIH1cblxuICAgIHRoaXMuZXh0ZXJuYWxzID0gW1xuICAgICAgLi4ucHJvcHMuZXh0ZXJuYWxNb2R1bGVzID8/IFsnYXdzLXNkayddLCAvLyBNYXJrIGF3cy1zZGsgYXMgZXh0ZXJuYWwgYnkgZGVmYXVsdCAoYXZhaWxhYmxlIGluIHRoZSBydW50aW1lKVxuICAgICAgLi4ucHJvcHMubm9kZU1vZHVsZXMgPz8gW10sIC8vIE1hcmsgdGhlIG1vZHVsZXMgdGhhdCB3ZSBhcmUgZ29pbmcgdG8gaW5zdGFsbCBhcyBleHRlcm5hbHMgYWxzb1xuICAgIF07XG5cbiAgICAvLyBEb2NrZXIgYnVuZGxpbmdcbiAgICBjb25zdCBzaG91bGRCdWlsZEltYWdlID0gcHJvcHMuZm9yY2VEb2NrZXJCdW5kbGluZyB8fCAhQnVuZGxpbmcucnVuc0xvY2FsbHk7XG4gICAgdGhpcy5pbWFnZSA9IHNob3VsZEJ1aWxkSW1hZ2VcbiAgICAgID8gcHJvcHMuZG9ja2VySW1hZ2UgPz8gY2RrLkRvY2tlckltYWdlLmZyb21CdWlsZChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGliJyksIHtcbiAgICAgICAgYnVpbGRBcmdzOiB7XG4gICAgICAgICAgLi4ucHJvcHMuYnVpbGRBcmdzID8/IHt9LFxuICAgICAgICAgIElNQUdFOiBwcm9wcy5ydW50aW1lLmJ1bmRsaW5nRG9ja2VySW1hZ2UuaW1hZ2UsXG4gICAgICAgICAgRVNCVUlMRF9WRVJTSU9OOiBwcm9wcy5lc2J1aWxkVmVyc2lvbiA/PyBFU0JVSUxEX1ZFUlNJT04sXG4gICAgICAgIH0sXG4gICAgICB9KVxuICAgICAgOiBjZGsuRG9ja2VySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdkdW1teScpOyAvLyBEbyBub3QgYnVpbGQgaWYgd2UgZG9uJ3QgbmVlZCB0b1xuXG4gICAgY29uc3QgYnVuZGxpbmdDb21tYW5kID0gdGhpcy5jcmVhdGVCdW5kbGluZ0NvbW1hbmQoY2RrLkFzc2V0U3RhZ2luZy5CVU5ETElOR19JTlBVVF9ESVIsIGNkay5Bc3NldFN0YWdpbmcuQlVORExJTkdfT1VUUFVUX0RJUik7XG4gICAgdGhpcy5jb21tYW5kID0gWydiYXNoJywgJy1jJywgYnVuZGxpbmdDb21tYW5kXTtcbiAgICB0aGlzLmVudmlyb25tZW50ID0gcHJvcHMuZW52aXJvbm1lbnQ7XG4gICAgLy8gQnVuZGxpbmcgc2V0cyB0aGUgd29ya2luZyBkaXJlY3RvcnkgdG8gY2RrLkFzc2V0U3RhZ2luZy5CVU5ETElOR19JTlBVVF9ESVJcbiAgICAvLyBhbmQgd2Ugd2FudCB0byBmb3JjZSBucHggdG8gdXNlIHRoZSBnbG9iYWxseSBpbnN0YWxsZWQgZXNidWlsZC5cbiAgICB0aGlzLndvcmtpbmdEaXJlY3RvcnkgPSAnLyc7XG5cbiAgICAvLyBMb2NhbCBidW5kbGluZ1xuICAgIGlmICghcHJvcHMuZm9yY2VEb2NrZXJCdW5kbGluZykgeyAvLyBvbmx5IGlmIERvY2tlciBpcyBub3QgZm9yY2VkXG4gICAgICBjb25zdCBvc1BsYXRmb3JtID0gb3MucGxhdGZvcm0oKTtcbiAgICAgIGNvbnN0IGNyZWF0ZUxvY2FsQ29tbWFuZCA9IChvdXRwdXREaXI6IHN0cmluZykgPT4gdGhpcy5jcmVhdGVCdW5kbGluZ0NvbW1hbmQocHJvamVjdFJvb3QsIG91dHB1dERpciwgb3NQbGF0Zm9ybSk7XG5cbiAgICAgIHRoaXMubG9jYWwgPSB7XG4gICAgICAgIHRyeUJ1bmRsZShvdXRwdXREaXI6IHN0cmluZykge1xuICAgICAgICAgIGlmIChCdW5kbGluZy5ydW5zTG9jYWxseSA9PT0gZmFsc2UpIHtcbiAgICAgICAgICAgIHByb2Nlc3Muc3RkZXJyLndyaXRlKCdlc2J1aWxkIGNhbm5vdCBydW4gbG9jYWxseS4gU3dpdGNoaW5nIHRvIERvY2tlciBidW5kbGluZy5cXG4nKTtcbiAgICAgICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBjb25zdCBsb2NhbENvbW1hbmQgPSBjcmVhdGVMb2NhbENvbW1hbmQob3V0cHV0RGlyKTtcblxuICAgICAgICAgIGV4ZWMoXG4gICAgICAgICAgICBvc1BsYXRmb3JtID09PSAnd2luMzInID8gJ2NtZCcgOiAnYmFzaCcsXG4gICAgICAgICAgICBbXG4gICAgICAgICAgICAgIG9zUGxhdGZvcm0gPT09ICd3aW4zMicgPyAnL2MnIDogJy1jJyxcbiAgICAgICAgICAgICAgbG9jYWxDb21tYW5kLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgZW52OiB7IC4uLnByb2Nlc3MuZW52LCAuLi5wcm9wcy5lbnZpcm9ubWVudCA/PyB7fSB9LFxuICAgICAgICAgICAgICBzdGRpbzogWyAvLyBzaG93IG91dHB1dFxuICAgICAgICAgICAgICAgICdpZ25vcmUnLCAvLyBpZ25vcmUgc3RkaW9cbiAgICAgICAgICAgICAgICBwcm9jZXNzLnN0ZGVyciwgLy8gcmVkaXJlY3Qgc3Rkb3V0IHRvIHN0ZGVyclxuICAgICAgICAgICAgICAgICdpbmhlcml0JywgLy8gaW5oZXJpdCBzdGRlcnJcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgY3dkOiBwYXRoLmRpcm5hbWUocHJvcHMuZW50cnkpLFxuICAgICAgICAgICAgICB3aW5kb3dzVmVyYmF0aW1Bcmd1bWVudHM6IG9zUGxhdGZvcm0gPT09ICd3aW4zMicsXG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgICB9LFxuICAgICAgfTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgY3JlYXRlQnVuZGxpbmdDb21tYW5kKGlucHV0RGlyOiBzdHJpbmcsIG91dHB1dERpcjogc3RyaW5nLCBvc1BsYXRmb3JtOiBOb2RlSlMuUGxhdGZvcm0gPSAnbGludXgnKTogc3RyaW5nIHtcbiAgICBjb25zdCBwYXRoSm9pbiA9IG9zUGF0aEpvaW4ob3NQbGF0Zm9ybSk7XG5cbiAgICBjb25zdCBucHggPSBvc1BsYXRmb3JtID09PSAnd2luMzInID8gJ25weC5jbWQnIDogJ25weCc7XG4gICAgY29uc3QgbG9hZGVycyA9IE9iamVjdC5lbnRyaWVzKHRoaXMucHJvcHMubG9hZGVyID8/IHt9KTtcbiAgICBjb25zdCBkZWZpbmVzID0gT2JqZWN0LmVudHJpZXModGhpcy5wcm9wcy5kZWZpbmUgPz8ge30pO1xuXG4gICAgY29uc3QgZXNidWlsZENvbW1hbmQ6IHN0cmluZyA9IFtcbiAgICAgIG5weCwgJ2VzYnVpbGQnLFxuICAgICAgJy0tYnVuZGxlJywgYFwiJHtwYXRoSm9pbihpbnB1dERpciwgdGhpcy5yZWxhdGl2ZUVudHJ5UGF0aCl9XCJgLFxuICAgICAgYC0tdGFyZ2V0PSR7dGhpcy5wcm9wcy50YXJnZXQgPz8gdG9UYXJnZXQodGhpcy5wcm9wcy5ydW50aW1lKX1gLFxuICAgICAgJy0tcGxhdGZvcm09bm9kZScsXG4gICAgICBgLS1vdXRmaWxlPVwiJHtwYXRoSm9pbihvdXRwdXREaXIsICdpbmRleC5qcycpfVwiYCxcbiAgICAgIC4uLnRoaXMucHJvcHMubWluaWZ5ID8gWyctLW1pbmlmeSddIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLnNvdXJjZU1hcCA/IFsnLS1zb3VyY2VtYXAnXSA6IFtdLFxuICAgICAgLi4udGhpcy5leHRlcm5hbHMubWFwKGV4dGVybmFsID0+IGAtLWV4dGVybmFsOiR7ZXh0ZXJuYWx9YCksXG4gICAgICAuLi5sb2FkZXJzLm1hcCgoW2V4dCwgbmFtZV0pID0+IGAtLWxvYWRlcjoke2V4dH09JHtuYW1lfWApLFxuICAgICAgLi4uZGVmaW5lcy5tYXAoKFtrZXksIHZhbHVlXSkgPT4gYC0tZGVmaW5lOiR7a2V5fT0ke0pTT04uc3RyaW5naWZ5KHZhbHVlKX1gKSxcbiAgICAgIC4uLnRoaXMucHJvcHMubG9nTGV2ZWwgPyBbYC0tbG9nLWxldmVsPSR7dGhpcy5wcm9wcy5sb2dMZXZlbH1gXSA6IFtdLFxuICAgICAgLi4udGhpcy5wcm9wcy5rZWVwTmFtZXMgPyBbJy0ta2VlcC1uYW1lcyddIDogW10sXG4gICAgICAuLi50aGlzLnJlbGF0aXZlVHNjb25maWdQYXRoID8gW2AtLXRzY29uZmlnPSR7cGF0aEpvaW4oaW5wdXREaXIsIHRoaXMucmVsYXRpdmVUc2NvbmZpZ1BhdGgpfWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLm1ldGFmaWxlID8gW2AtLW1ldGFmaWxlPSR7cGF0aEpvaW4ob3V0cHV0RGlyLCAnaW5kZXgubWV0YS5qc29uJyl9YF0gOiBbXSxcbiAgICAgIC4uLnRoaXMucHJvcHMuYmFubmVyID8gW2AtLWJhbm5lcjpqcz0ke0pTT04uc3RyaW5naWZ5KHRoaXMucHJvcHMuYmFubmVyKX1gXSA6IFtdLFxuICAgICAgLi4udGhpcy5wcm9wcy5mb290ZXIgPyBbYC0tZm9vdGVyOmpzPSR7SlNPTi5zdHJpbmdpZnkodGhpcy5wcm9wcy5mb290ZXIpfWBdIDogW10sXG4gICAgXS5qb2luKCcgJyk7XG5cbiAgICBsZXQgZGVwc0NvbW1hbmQgPSAnJztcbiAgICBpZiAodGhpcy5wcm9wcy5ub2RlTW9kdWxlcykge1xuICAgICAgLy8gRmluZCAncGFja2FnZS5qc29uJyBjbG9zZXN0IHRvIGVudHJ5IGZvbGRlciwgd2UgYXJlIGdvaW5nIHRvIGV4dHJhY3QgdGhlXG4gICAgICAvLyBtb2R1bGVzIHZlcnNpb25zIGZyb20gaXQuXG4gICAgICBjb25zdCBwa2dQYXRoID0gZmluZFVwKCdwYWNrYWdlLmpzb24nLCBwYXRoLmRpcm5hbWUodGhpcy5wcm9wcy5lbnRyeSkpO1xuICAgICAgaWYgKCFwa2dQYXRoKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGZpbmQgYSBgcGFja2FnZS5qc29uYCBpbiB0aGlzIHByb2plY3QuIFVzaW5nIGBub2RlTW9kdWxlc2AgcmVxdWlyZXMgYSBgcGFja2FnZS5qc29uYC4nKTtcbiAgICAgIH1cblxuICAgICAgLy8gRGV0ZXJtaW5lIGRlcGVuZGVuY2llcyB2ZXJzaW9ucywgbG9jayBmaWxlIGFuZCBpbnN0YWxsZXJcbiAgICAgIGNvbnN0IGRlcGVuZGVuY2llcyA9IGV4dHJhY3REZXBlbmRlbmNpZXMocGtnUGF0aCwgdGhpcy5wcm9wcy5ub2RlTW9kdWxlcyk7XG4gICAgICBsZXQgaW5zdGFsbGVyID0gSW5zdGFsbGVyLk5QTTtcbiAgICAgIGxldCBsb2NrRmlsZSA9IExvY2tGaWxlLk5QTTtcbiAgICAgIGlmICh0aGlzLnByb3BzLmRlcHNMb2NrRmlsZVBhdGguZW5kc1dpdGgoTG9ja0ZpbGUuWUFSTikpIHtcbiAgICAgICAgbG9ja0ZpbGUgPSBMb2NrRmlsZS5ZQVJOO1xuICAgICAgICBpbnN0YWxsZXIgPSBJbnN0YWxsZXIuWUFSTjtcbiAgICAgIH1cblxuICAgICAgY29uc3Qgb3NDb21tYW5kID0gbmV3IE9zQ29tbWFuZChvc1BsYXRmb3JtKTtcblxuICAgICAgLy8gQ3JlYXRlIGR1bW15IHBhY2thZ2UuanNvbiwgY29weSBsb2NrIGZpbGUgaWYgYW55IGFuZCB0aGVuIGluc3RhbGxcbiAgICAgIGRlcHNDb21tYW5kID0gY2hhaW4oW1xuICAgICAgICBvc0NvbW1hbmQud3JpdGVKc29uKHBhdGhKb2luKG91dHB1dERpciwgJ3BhY2thZ2UuanNvbicpLCB7IGRlcGVuZGVuY2llcyB9KSxcbiAgICAgICAgb3NDb21tYW5kLmNvcHkocGF0aEpvaW4oaW5wdXREaXIsIGxvY2tGaWxlKSwgcGF0aEpvaW4ob3V0cHV0RGlyLCBsb2NrRmlsZSkpLFxuICAgICAgICBvc0NvbW1hbmQuY2hhbmdlRGlyZWN0b3J5KG91dHB1dERpciksXG4gICAgICAgIGAke2luc3RhbGxlcn0gaW5zdGFsbGAsXG4gICAgICBdKTtcbiAgICB9XG5cbiAgICByZXR1cm4gY2hhaW4oW1xuICAgICAgLi4udGhpcy5wcm9wcy5jb21tYW5kSG9va3M/LmJlZm9yZUJ1bmRsaW5nKGlucHV0RGlyLCBvdXRwdXREaXIpID8/IFtdLFxuICAgICAgZXNidWlsZENvbW1hbmQsXG4gICAgICAuLi4odGhpcy5wcm9wcy5ub2RlTW9kdWxlcyAmJiB0aGlzLnByb3BzLmNvbW1hbmRIb29rcz8uYmVmb3JlSW5zdGFsbChpbnB1dERpciwgb3V0cHV0RGlyKSkgPz8gW10sXG4gICAgICBkZXBzQ29tbWFuZCxcbiAgICAgIC4uLnRoaXMucHJvcHMuY29tbWFuZEhvb2tzPy5hZnRlckJ1bmRsaW5nKGlucHV0RGlyLCBvdXRwdXREaXIpID8/IFtdLFxuICAgIF0pO1xuICB9XG59XG5cbmVudW0gSW5zdGFsbGVyIHtcbiAgTlBNID0gJ25wbScsXG4gIFlBUk4gPSAneWFybicsXG59XG5cbi8qKlxuICogT1MgYWdub3N0aWMgY29tbWFuZFxuICovXG5jbGFzcyBPc0NvbW1hbmQge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IG9zUGxhdGZvcm06IE5vZGVKUy5QbGF0Zm9ybSkge31cblxuICBwdWJsaWMgd3JpdGVKc29uKGZpbGVQYXRoOiBzdHJpbmcsIGRhdGE6IGFueSk6IHN0cmluZyB7XG4gICAgY29uc3Qgc3RyaW5naWZpZWREYXRhID0gSlNPTi5zdHJpbmdpZnkoZGF0YSk7XG4gICAgaWYgKHRoaXMub3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJykge1xuICAgICAgcmV0dXJuIGBlY2hvIF4ke3N0cmluZ2lmaWVkRGF0YX1eID4gJHtmaWxlUGF0aH1gO1xuICAgIH1cblxuICAgIHJldHVybiBgZWNobyAnJHtzdHJpbmdpZmllZERhdGF9JyA+ICR7ZmlsZVBhdGh9YDtcbiAgfVxuXG4gIHB1YmxpYyBjb3B5KHNyYzogc3RyaW5nLCBkZXN0OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIGlmICh0aGlzLm9zUGxhdGZvcm0gPT09ICd3aW4zMicpIHtcbiAgICAgIHJldHVybiBgY29weSAke3NyY30gJHtkZXN0fWA7XG4gICAgfVxuXG4gICAgcmV0dXJuIGBjcCAke3NyY30gJHtkZXN0fWA7XG4gIH1cblxuICBwdWJsaWMgY2hhbmdlRGlyZWN0b3J5KGRpcjogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gYGNkICR7ZGlyfWA7XG4gIH1cbn1cblxuLyoqXG4gKiBDaGFpbiBjb21tYW5kc1xuICovXG5mdW5jdGlvbiBjaGFpbihjb21tYW5kczogc3RyaW5nW10pOiBzdHJpbmcge1xuICByZXR1cm4gY29tbWFuZHMuZmlsdGVyKGMgPT4gISFjKS5qb2luKCcgJiYgJyk7XG59XG5cbi8qKlxuICogUGxhdGZvcm0gc3BlY2lmaWMgcGF0aCBqb2luXG4gKi9cbmZ1bmN0aW9uIG9zUGF0aEpvaW4ocGxhdGZvcm06IE5vZGVKUy5QbGF0Zm9ybSkge1xuICByZXR1cm4gZnVuY3Rpb24oLi4ucGF0aHM6IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgICBjb25zdCBqb2luZWQgPSBwYXRoLmpvaW4oLi4ucGF0aHMpO1xuICAgIC8vIElmIHdlIGFyZSBvbiB3aW4zMiBidXQgbmVlZCBwb3NpeCBzdHlsZSBwYXRoc1xuICAgIGlmIChvcy5wbGF0Zm9ybSgpID09PSAnd2luMzInICYmIHBsYXRmb3JtICE9PSAnd2luMzInKSB7XG4gICAgICByZXR1cm4gam9pbmVkLnJlcGxhY2UoL1xcXFwvZywgJy8nKTtcbiAgICB9XG4gICAgcmV0dXJuIGpvaW5lZDtcbiAgfTtcbn1cblxuLyoqXG4gKiBDb252ZXJ0cyBhIHJ1bnRpbWUgdG8gYW4gZXNidWlsZCBub2RlIHRhcmdldFxuICovXG5mdW5jdGlvbiB0b1RhcmdldChydW50aW1lOiBSdW50aW1lKTogc3RyaW5nIHtcbiAgY29uc3QgbWF0Y2ggPSBydW50aW1lLm5hbWUubWF0Y2goL25vZGVqcyhcXGQrKS8pO1xuXG4gIGlmICghbWF0Y2gpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBleHRyYWN0IHZlcnNpb24gZnJvbSBydW50aW1lLicpO1xuICB9XG5cbiAgcmV0dXJuIGBub2RlJHttYXRjaFsxXX1gO1xufVxuIl19