"""
Module for handling HTTP Client Error responses (4xx).

This module provides a class to represent and handle client error HTTP status codes (4xx).
These status codes indicate that the client seems to have made an error in the request.

Attributes:
    VERSION (str): Version of the module.

Classes:
    BadRequest
    Unauthorized
    PaymentRequired
    Forbidden
    NotFound
    MethodNotAllowed
    NotAcceptable
    ProxyAuthenticationRequired
    RequestTimeout
    Conflict
    Gone
    LengthRequired
    PreconditionFailed
    PayloadTooLarge
    UriTooLong
    UnsupportedMediaType
    RangeNotSatisfiable
    ExpectationFailed
    ImATeapot
    MisdirectedRequest
    UnprocessableEntity
    Locked
    FailedDependency
    TooEarly
    UpgradeRequired
    PreconditionRequired
    TooManyRequests
    RequestHeaderFieldsTooLarge
    UnavailableForLegalReasons

"""

VERSION: str = "1.0.0"

from http import HTTPStatus
from typing import Self

import http_status_code_exception.base_class

class BadRequest(http_status_code_exception.base_class.HTTPException):
    """400 Bad Request: The server could not understand the request due to invalid syntax."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.BAD_REQUEST, message)


class Unauthorized(http_status_code_exception.base_class.HTTPException):
    """401 Unauthorized: Authentication is required and has failed or has not yet been provided."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.UNAUTHORIZED, message)


class PaymentRequired(http_status_code_exception.base_class.HTTPException):
    """402 Payment Required: Reserved for future use."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.PAYMENT_REQUIRED, message)


class Forbidden(http_status_code_exception.base_class.HTTPException):
    """403 Forbidden: The server understood the request but refuses to authorize it."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.FORBIDDEN, message)


class NotFound(http_status_code_exception.base_class.HTTPException):
    """404 Not Found: The server could not find the requested resource."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.NOT_FOUND, message)


class MethodNotAllowed(http_status_code_exception.base_class.HTTPException):
    """405 Method Not Allowed: The method specified in the request is not allowed for the resource."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.METHOD_NOT_ALLOWED, message)


class NotAcceptable(http_status_code_exception.base_class.HTTPException):
    """406 Not Acceptable: The server cannot produce a response matching the list of acceptable values."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.NOT_ACCEPTABLE, message)


class ProxyAuthenticationRequired(http_status_code_exception.base_class.HTTPException):
    """407 Proxy Authentication Required: Authentication by a proxy is required."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.PROXY_AUTHENTICATION_REQUIRED, message)


class RequestTimeout(http_status_code_exception.base_class.HTTPException):
    """408 Request Timeout: The server timed out waiting for the request."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.REQUEST_TIMEOUT, message)


class Conflict(http_status_code_exception.base_class.HTTPException):
    """409 Conflict: The request could not be completed due to a conflict with the current state of the resource."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.CONFLICT, message)


class Gone(http_status_code_exception.base_class.HTTPException):
    """410 Gone: The resource requested is no longer available and will not be available again."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.GONE, message)


class LengthRequired(http_status_code_exception.base_class.HTTPException):
    """411 Length Required: The request did not specify the length of its content."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.LENGTH_REQUIRED, message)


class PreconditionFailed(http_status_code_exception.base_class.HTTPException):
    """412 Precondition Failed: The server does not meet one of the preconditions specified in the request."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.PRECONDITION_FAILED, message)


class PayloadTooLarge(http_status_code_exception.base_class.HTTPException):
    """413 Payload Too Large: The server refuses to process the request because the payload is too large."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.PAYLOAD_TOO_LARGE, message)


class URITooLong(http_status_code_exception.base_class.HTTPException):
    """414 URI Too Long: The server refuses to process the request because the URI is too long."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.URI_TOO_LONG, message)


class UnsupportedMediaType(http_status_code_exception.base_class.HTTPException):
    """415 Unsupported Media Type: The server refuses to process the request because the media type is unsupported."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.UNSUPPORTED_MEDIA_TYPE, message)


class RangeNotSatisfiable(http_status_code_exception.base_class.HTTPException):
    """416 Range Not Satisfiable: The server cannot serve the requested byte range."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.RANGE_NOT_SATISFIABLE, message)


class ExpectationFailed(http_status_code_exception.base_class.HTTPException):
    """417 Expectation Failed: The server cannot meet the expectation set in the Expect request header."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.EXPECTATION_FAILED, message)


class IMATeapot(http_status_code_exception.base_class.HTTPException):
    """418 I'm a Teapot: A humorous status code indicating that the server refuses to brew coffee."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.IM_A_TEAPOT, message)


class MisdirectedRequest(http_status_code_exception.base_class.HTTPException):
    """421 Misdirected Request: The request was directed to a server that is unable to produce a response."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.MISDIRECTED_REQUEST, message)


class UnprocessableEntity(http_status_code_exception.base_class.HTTPException):
    """422 Unprocessable Entity: The server understands the content type but was unable to process the enclosed instructions."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.UNPROCESSABLE_ENTITY, message)


class Locked(http_status_code_exception.base_class.HTTPException):
    """423 Locked: The resource that is being accessed is locked."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.LOCKED, message)


class FailedDependency(http_status_code_exception.base_class.HTTPException):
    """424 Failed Dependency: The request failed due to failure of a previous request."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.FAILED_DEPENDENCY, message)


class TooEarly(http_status_code_exception.base_class.HTTPException):
    """425 Too Early: The server is unwilling to risk processing a request that might be replayed."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.TOO_EARLY, message)


class UpgradeRequired(http_status_code_exception.base_class.HTTPException):
    """426 Upgrade Required: The client should switch to a different protocol."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.UPGRADE_REQUIRED, message)


class PreconditionRequired(http_status_code_exception.base_class.HTTPException):
    """428 Precondition Required: The origin server requires the request to be conditional."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.PRECONDITION_REQUIRED, message)


class TooManyRequests(http_status_code_exception.base_class.HTTPException):
    """429 Too Many Requests: The user has sent too many requests in a given amount of time."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.TOO_MANY_REQUESTS, message)


class RequestHeaderFieldsTooLarge(http_status_code_exception.base_class.HTTPException):
    """431 Request Header Fields Too Large: The server refuses to process the request because the header fields are too large."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.REQUEST_HEADER_FIELDS_TOO_LARGE, message)


class UnavailableForLegalReasons(http_status_code_exception.base_class.HTTPException):
    """451 Unavailable For Legal Reasons: The server is denying access to the resource as a consequence of a legal demand."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.UNAVAILABLE_FOR_LEGAL_REASONS, message)
