"""
Module for retrieving HTTP exception classes based on status codes.

This module provides a function `get` that maps HTTP status codes to their corresponding exception classes.
It supports informational, success, redirection, client error, and server error status codes. If the provided
status code is not recognized or not supported, a `ValueError` is raised.

Attributes:
    VERSION (str): Version of the module.
    
Functions:
    get(status_code: int) -> type[HTTPException]: 
        Retrieves the appropriate HTTP exception class for the given status code.

Raises:
    ValueError: If the provided status code is not recognized or not supported.

"""

from http import HTTPStatus

import http_status_code_exception

def get(status_code: int) -> http_status_code_exception.base_class.HTTPException:
    """
    Get the appropriate HTTP exception class based on the provided status code.

    Args:
        status_code (int): The HTTP status code for which to retrieve the exception class.

    Returns:
        type[HTTPException]: The exception class corresponding to the given status code.

    Raises:
        ValueError: If the provided status code is not a recognized HTTP status code.

    Example:
        >>> exception = get(404)
        >>> raise exception("Page not found")
        NotFound: Page not found (404)
    """
    if status_code in (413, 414, 416):
        raise ValueError(f"http.HTTPStatus not supported status code: {status_code}")
    exceptions = {
        HTTPStatus.CONTINUE.value: http_status_code_exception.informational.Continue,
        HTTPStatus.SWITCHING_PROTOCOLS.value: http_status_code_exception.informational.SwitchingProtocols,
        HTTPStatus.PROCESSING.value: http_status_code_exception.informational.Processing,
        HTTPStatus.OK.value: http_status_code_exception.success.Ok,
        HTTPStatus.CREATED.value: http_status_code_exception.success.Created,
        HTTPStatus.ACCEPTED.value: http_status_code_exception.success.Accepted,
        HTTPStatus.NON_AUTHORITATIVE_INFORMATION.value: http_status_code_exception.success.NonAuthoritativeInformation,
        HTTPStatus.NO_CONTENT.value: http_status_code_exception.success.NoContent,
        HTTPStatus.RESET_CONTENT.value: http_status_code_exception.success.ResetContent,
        HTTPStatus.PARTIAL_CONTENT.value: http_status_code_exception.success.PartialContent,
        HTTPStatus.MULTI_STATUS.value: http_status_code_exception.success.MultiStatus,
        HTTPStatus.ALREADY_REPORTED.value: http_status_code_exception.success.AlreadyReported,
        HTTPStatus.IM_USED.value: http_status_code_exception.success.IMUsed,
        HTTPStatus.MULTIPLE_CHOICES.value: http_status_code_exception.redirection.MultipleChoices,
        HTTPStatus.MOVED_PERMANENTLY.value: http_status_code_exception.redirection.MovedPermanently,
        HTTPStatus.FOUND.value: http_status_code_exception.redirection.Found,
        HTTPStatus.SEE_OTHER.value: http_status_code_exception.redirection.SeeOther,
        HTTPStatus.NOT_MODIFIED.value: http_status_code_exception.redirection.NotModified,
        HTTPStatus.USE_PROXY.value: http_status_code_exception.redirection.UseProxy,
        HTTPStatus.TEMPORARY_REDIRECT.value: http_status_code_exception.redirection.TemporaryRedirect,
        HTTPStatus.PERMANENT_REDIRECT.value: http_status_code_exception.redirection.PermanentRedirect,
        HTTPStatus.BAD_REQUEST.value: http_status_code_exception.client_error.BadRequest,
        HTTPStatus.UNAUTHORIZED.value: http_status_code_exception.client_error.Unauthorized,
        HTTPStatus.PAYMENT_REQUIRED.value: http_status_code_exception.client_error.PaymentRequired,
        HTTPStatus.FORBIDDEN.value: http_status_code_exception.client_error.Forbidden,
        HTTPStatus.NOT_FOUND.value: http_status_code_exception.client_error.NotFound,
        HTTPStatus.METHOD_NOT_ALLOWED.value: http_status_code_exception.client_error.MethodNotAllowed,
        HTTPStatus.NOT_ACCEPTABLE.value: http_status_code_exception.client_error.NotAcceptable,
        HTTPStatus.PROXY_AUTHENTICATION_REQUIRED.value: http_status_code_exception.client_error.ProxyAuthenticationRequired,
        HTTPStatus.REQUEST_TIMEOUT.value: http_status_code_exception.client_error.RequestTimeout,
        HTTPStatus.CONFLICT.value: http_status_code_exception.client_error.Conflict,
        HTTPStatus.GONE.value: http_status_code_exception.client_error.Gone,
        HTTPStatus.LENGTH_REQUIRED.value: http_status_code_exception.client_error.LengthRequired,
        HTTPStatus.PRECONDITION_FAILED.value: http_status_code_exception.client_error.PreconditionFailed,
        # HTTPStatus.PAYLOAD_TOO_LARGE.value: http_status_code_exception.client_error_responses.PayloadTooLarge,
        # HTTPStatus.URI_TOO_LONG.value: http_status_code_exception.client_error_responses.URITooLong,
        HTTPStatus.UNSUPPORTED_MEDIA_TYPE.value: http_status_code_exception.client_error.UnsupportedMediaType,
        # HTTPStatus.RANGE_NOT_SATISFIABLE.value: http_status_code_exception.client_error_responses.RangeNotSatisfiable,
        HTTPStatus.EXPECTATION_FAILED.value: http_status_code_exception.client_error.ExpectationFailed,
        HTTPStatus.IM_A_TEAPOT.value: http_status_code_exception.client_error.IMATeapot,
        HTTPStatus.MISDIRECTED_REQUEST.value: http_status_code_exception.client_error.MisdirectedRequest,
        HTTPStatus.UNPROCESSABLE_ENTITY.value: http_status_code_exception.client_error.UnprocessableEntity,
        HTTPStatus.LOCKED.value: http_status_code_exception.client_error.Locked,
        HTTPStatus.FAILED_DEPENDENCY.value: http_status_code_exception.client_error.FailedDependency,
        HTTPStatus.TOO_EARLY.value: http_status_code_exception.client_error.TooEarly,
        HTTPStatus.UPGRADE_REQUIRED.value: http_status_code_exception.client_error.UpgradeRequired,
        HTTPStatus.PRECONDITION_REQUIRED.value: http_status_code_exception.client_error.PreconditionRequired,
        HTTPStatus.TOO_MANY_REQUESTS.value: http_status_code_exception.client_error.TooManyRequests,
        HTTPStatus.REQUEST_HEADER_FIELDS_TOO_LARGE.value: http_status_code_exception.client_error.RequestHeaderFieldsTooLarge,
        HTTPStatus.UNAVAILABLE_FOR_LEGAL_REASONS.value: http_status_code_exception.client_error.UnavailableForLegalReasons,
        HTTPStatus.INTERNAL_SERVER_ERROR.value: http_status_code_exception.server_error.InternalServerError,
        HTTPStatus.NOT_IMPLEMENTED.value: http_status_code_exception.server_error.NotImplemented,
        HTTPStatus.BAD_GATEWAY.value: http_status_code_exception.server_error.BadGateway,
        HTTPStatus.SERVICE_UNAVAILABLE.value: http_status_code_exception.server_error.ServiceUnavailable,
        HTTPStatus.GATEWAY_TIMEOUT.value: http_status_code_exception.server_error.GatewayTimeout,
        HTTPStatus.HTTP_VERSION_NOT_SUPPORTED.value: http_status_code_exception.server_error.HTTPVersionNotSupported,
        HTTPStatus.VARIANT_ALSO_NEGOTIATES.value: http_status_code_exception.server_error.VariantAlsoNegotiates,
        HTTPStatus.INSUFFICIENT_STORAGE.value: http_status_code_exception.server_error.InsufficientStorage,
        HTTPStatus.LOOP_DETECTED.value: http_status_code_exception.server_error.LoopDetected,
        HTTPStatus.NOT_EXTENDED.value: http_status_code_exception.server_error.NotExtended,
        HTTPStatus.NETWORK_AUTHENTICATION_REQUIRED.value: http_status_code_exception.server_error.NetworkAuthenticationRequired,
    }
    if status_code not in exceptions:
        raise ValueError(f"Unknown HTTP status code: {status_code}")
    return exceptions[status_code]
