"""
Module for handling HTTP Successful responses (2xx).

This module provides a class to represent and handle successful HTTP status codes (2xx).
These status codes indicate that the client's request was successfully received, understood, and accepted.

Attributes:
    VERSION (str): Version of the module.

Classes:
    Ok
    Created
    Accepted
    NonAuthoritativeInformation
    NoContent
    ResetContent
    PartialContent
    MultiStatus
    AlreadyReported
    ImUsed

"""

VERSION: str = "1.0.0"

from http import HTTPStatus
from typing import Self

import http_status_code_exception.base_class

class Ok(http_status_code_exception.base_class.HTTPException):
    """200 OK: The request succeeded."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.OK, message)


class Created(http_status_code_exception.base_class.HTTPException):
    """201 Created: The request was successful, and a new resource was created."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.CREATED, message)


class Accepted(http_status_code_exception.base_class.HTTPException):
    """202 Accepted: The request was accepted for processing, but it may not be completed yet."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.ACCEPTED, message)


class NonAuthoritativeInformation(http_status_code_exception.base_class.HTTPException):
    """203 Non-Authoritative Information: The response was successfully received, but from a third-party source."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.NON_AUTHORITATIVE_INFORMATION, message)


class NoContent(http_status_code_exception.base_class.HTTPException):
    """204 No Content: The request was successful, but there is no content to return."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.NO_CONTENT, message)


class ResetContent(http_status_code_exception.base_class.HTTPException):
    """205 Reset Content: The client should reset the document view."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.RESET_CONTENT, message)


class PartialContent(http_status_code_exception.base_class.HTTPException):
    """206 Partial Content: The server is returning part of the requested resource."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.PARTIAL_CONTENT, message)


class MultiStatus(http_status_code_exception.base_class.HTTPException):
    """207 Multi-Status: The response contains multiple status codes."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.MULTI_STATUS, message)


class AlreadyReported(http_status_code_exception.base_class.HTTPException):
    """208 Already Reported: The resource was already reported in a previous response."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.ALREADY_REPORTED, message)


class IMUsed(http_status_code_exception.base_class.HTTPException):
    """226 IM Used: The server has fulfilled the request using the specified instance-manipulations."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.IM_USED, message)
