"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ManagedPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * Managed policy.
 *
 * @stability stable
 */
class ManagedPolicy extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.managedPolicyName,
        });
        /**
         * The policy document.
         *
         * @stability stable
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        this.description = props.description || '';
        this.path = props.path || '/';
        if (props.document) {
            this.document = props.document;
        }
        const resource = new iam_generated_1.CfnManagedPolicy(this, 'Resource', {
            policyDocument: this.document,
            managedPolicyName: this.physicalName,
            description: this.description,
            path: this.path,
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatements(p));
        }
        // arn:aws:iam::123456789012:policy/teststack-CreateTestDBPolicy-16M23YE3CS700
        this.managedPolicyName = this.getResourceNameAttribute(core_1.Stack.of(this).parseArn(resource.ref, '/').resourceName);
        this.managedPolicyArn = this.getResourceArnAttribute(resource.ref, {
            region: '',
            service: 'iam',
            resource: 'policy',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a customer managed policy from the managedPolicyName.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     * @stability stable
     */
    static fromManagedPolicyName(scope, id, managedPolicyName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = core_1.Stack.of(scope).formatArn({
                    service: 'iam',
                    region: '',
                    account: core_1.Stack.of(scope).account,
                    resource: 'policy',
                    resourceName: managedPolicyName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an external managed policy by ARN.
     *
     * For this managed policy, you only need to know the ARN to be able to use it.
     * This can be useful if you got the ARN from a CloudFormation Export.
     *
     * If the imported Managed Policy ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * managed policy has a `path` (like `arn:...:policy/AdminPolicy/AdminAllow`), the
     * `managedPolicyName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Managed Policy ARN should be supplied without the
     * `path` in order to resolve the correct managed policy resource.
     *
     * @param scope construct scope.
     * @param id construct id.
     * @param managedPolicyArn the ARN of the managed policy to import.
     * @stability stable
     */
    static fromManagedPolicyArn(scope, id, managedPolicyArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = managedPolicyArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a managed policy from one of the policies that AWS manages.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     * Some managed policy names start with "service-role/", some start with
     * "job-function/", and some don't start with anything. Do include the
     * prefix when constructing this object.
     *
     * @stability stable
     */
    static fromAwsManagedPolicyName(managedPolicyName) {
        class AwsManagedPolicy {
            constructor() {
                this.managedPolicyArn = core_1.Lazy.uncachedString({
                    produce(ctx) {
                        return core_1.Stack.of(ctx.scope).formatArn({
                            service: 'iam',
                            region: '',
                            account: 'aws',
                            resource: 'policy',
                            resourceName: managedPolicyName,
                        });
                    },
                });
            }
        }
        return new AwsManagedPolicy();
    }
    /**
     * Adds a statement to the policy document.
     *
     * @stability stable
     */
    addStatements(...statement) {
        this.document.addStatements(...statement);
    }
    /**
     * Attaches this policy to a user.
     *
     * @stability stable
     */
    attachToUser(user) {
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
    }
    /**
     * Attaches this policy to a role.
     *
     * @stability stable
     */
    attachToRole(role) {
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
    }
    /**
     * Attaches this policy to a group.
     *
     * @stability stable
     */
    attachToGroup(group) {
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            result.push('Managed Policy is empty. You must add statements to the policy');
        }
        result.push(...this.document.validateForIdentityPolicy());
        return result;
    }
}
exports.ManagedPolicy = ManagedPolicy;
_a = JSII_RTTI_SYMBOL_1;
ManagedPolicy[_a] = { fqn: "@aws-cdk/aws-iam.ManagedPolicy", version: "1.112.0" };
//# sourceMappingURL=data:application/json;base64,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