from azureml.core.conda_dependencies import CondaDependencies


class Dependencies:
    # Please update the azure-pipelines.yml as well
    # when upgrade to a new python version
    DEFAULT_PYTHON_VERSION = '3.6.8'
    DEFAULT_NAME = 'project_environment'

    """A class wraps AzureML Service's CondaDependencies class"""
    def __init__(self, conda_content: dict = None):
        if not conda_content:
            # Create an dummy dict if not specified.
            #
            # Cannot use `{}` or `None` here, because they will be ignored by CondaDependencies,
            # and will generate a default CondaDependencies instance containing packages
            # like 'azureml-defaults', which is not expected by our use case.
            # Thus, we make an dummy dict only contains 'name' here as a workaround.
            #
            # NOTE: We must create this dummy dict EVERY TIME, and cannot define as a global variable.
            #       Otherwise the instances of Dependencies will interfere with each other.
            conda_content = {'name': self.DEFAULT_NAME}

        self._underlying = CondaDependencies(_underlying_structure=conda_content)

    @staticmethod
    def create_default():
        """Default conda environment configuration.

        Yaml description is as follows:

        name: project_environment
        channels:
        - defaults
        dependencies:
        - python=3.6.8
        """
        d = Dependencies()
        d.add_conda_packages(f'python={Dependencies.DEFAULT_PYTHON_VERSION}')
        return d

    @staticmethod
    def update_from_default(channels=None, conda_packages=None, pip_options=None, pip_packages=None):
        """Creates a Dependencies object, updating from the one generated by `create_default()`."""
        d = Dependencies.create_default()
        if channels:
            d.add_channels(*channels)
        if conda_packages:
            d.add_conda_packages(*conda_packages)
        if pip_options:
            d.add_pip_options(*pip_options)
        if pip_packages:
            d.add_pip_packages(*pip_packages)
        return d

    @property
    def channels(self):
        result = sorted(self._underlying.conda_channels)
        if not result:
            result = ['defaults']
        return result

    def add_channels(self, *channels):
        for channel in channels:
            self._underlying.add_channel(channel)

    @property
    def conda_packages(self):
        return sorted(self._underlying.conda_packages)

    def add_conda_packages(self, *packages):
        for package in packages:
            self._underlying.add_conda_package(package)

    @property
    def pip_options(self):
        def option_weight(option):
            if option.startswith('--index-url'):
                # Use this hack to make index url at top of the list
                # otherwise conda will failed to search extract index urls
                # if --extra-index-url is before --index-url.
                return '---'
            return option.lower()

        return sorted(self._underlying.pip_options, key=option_weight)

    def add_pip_options(self, *options):
        for option in options:
            self._underlying.set_pip_option(option)

    @property
    def pip_packages(self):
        return sorted(self._underlying.pip_packages)

    def add_pip_packages(self, *packages):
        for package in packages:
            self._underlying.add_pip_package(package)

    def remove_pip_package(self, pip_package):
        self._underlying.remove_pip_package(pip_package)

    def get_pip_package_with_name(self, pip_name):
        def get_package_name(pip_entry):
            return CondaDependencies._get_package_name_with_extras(pip_entry)

        def check_name_match(package_name):
            # Check if pip name match package name
            # eg: azureml-designer-cv-modules matches azureml-designer-cv-modules[pytorch]

            package_name_stripped = package_name[:package_name.find('[')]
            return pip_name in (package_name_stripped, package_name)

        return next((p for p in self.pip_packages if check_name_match(get_package_name(p))), None)

    def add_version_suffix_to_package(self, pip_name, version_suffix):
        if not pip_name or not version_suffix:
            return

        pip_name_with_version = self.get_pip_package_with_name(pip_name)
        if pip_name_with_version:
            if pip_name_with_version == pip_name:
                # If true, this entry contains no version numbers. We do not support to add version suffixes to it.
                # e.g
                #     expected pip example: alghost==0.0.100
                #     unexpected pip example: alghost
                raise ValueError(f"'{pip_name}': no version specified. Cannot add version suffix.")

            self.remove_pip_package(pip_name)
            self.add_pip_packages(f"{pip_name_with_version}.{version_suffix}")

    @property
    def flat_dict(self):
        """Output a flat dict describing the dependencies, for SMT/JES/DS usage."""
        return {
            'CondaChannels': self.channels,
            'CondaPackages': self.conda_packages,
            'PipOptions': self.pip_options,
            'PipPackages': self.pip_packages,
        }

    def to_dict(self):
        """This is called when extracting module json spec file."""
        return self.flat_dict

    @property
    def conda_dependency_dict(self):
        """Merge raw data lists into a conda environment compatible dict, which will be dumped to a YAML file.

        For conda environment file format please refer to:
        https://docs.conda.io/projects/conda/en/latest/user-guide/tasks/manage-environments.html#creating-an-environment-file-manually

        Rules for generating conda environment YAML file:
          1. Firstly, sort each of the lists alphabetically.
          2. Merge PipOptions and PipPackages lists into one list. PipOptions comes before PipPackages.
          3. Create a dictionary object with key 'pip', value is the list created in step 2.
          4. Append this dictionary to the end of CondaPackages. (Only if the list created in step 2 is not empty.)
          5. Create a dictionary with the following keys (make sure to keep the order):
             * 'name': is hard-coded 'project_environment'
             * 'channels': the CondaChannel list. if list is empty, set to a default list as ['defaults'].
             * 'dependencies': is the CondaPackages list with pip appended to the end (if any).
          6. Dump this dict to a yaml file.

        WARNING:
             Do NOT change this implementation.

             AzureML service will create a hash for the generated YAML file,
             which will be used as the key of the docker image cache.
             If the implementation changed, and the image cache may fail to match,
             causing performance problems.

        WARNING 2:
             There is also a C# version of this logic in the JES code base.

             ref: `ConstructCondaDependency` method in the following code:
             https://msdata.visualstudio.com/AzureML/_git/StudioCore?path=%2FProduct%2FSource%2FStudioCoreService%2FCommon%2FCommonHelper.cs&version=GBmaster

             Please keep the C# code synced in case the implementation here
             must be changed in the future.
        """

        dependencies = self.conda_packages
        pip_options_and_packages = self.pip_options + self.pip_packages
        if pip_options_and_packages:
            pip_entry = {'pip': self.pip_options + self.pip_packages}
            dependencies.append(pip_entry)

        return {
            'name': self.DEFAULT_NAME,
            'channels': self.channels,
            'dependencies': dependencies,
        }

    @property
    def conda_dependency_yaml(self):
        """Output a yaml file describing the dependencies, for AzureML Service usage."""
        cd = CondaDependencies(_underlying_structure=self.conda_dependency_dict)
        return cd.serialize_to_string()
