"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const path = require("path");
const assertions_1 = require("aws-cdk-lib/assertions");
const core_1 = require("@aws-solutions-constructs/core");
test('Simple deployment works', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiGateway).toBeDefined();
    expect(construct.apiGatewayCloudWatchRole).toBeDefined();
    expect(construct.apiGatewayLogGroup).toBeDefined();
    expect(construct.apiLambdaFunctions.length).toEqual(1);
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from Asset', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from S3 Bucket and Key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = core_1.CreateScrapBucket(stack, {});
    const apiDefinitionKey = 'api.yaml';
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionBucket,
        apiDefinitionKey,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('Throws error when both api definition asset and s3 object are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionBucket: new s3.Bucket(stack, 'ApiDefinitionBucket'),
            apiDefinitionKey: 'key',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: lambda.Runtime.NODEJS_18_X,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified, but not both');
});
test('Multiple Lambda Functions can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
});
test('Existing lambda function can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler',
        functionName: 'ExistingLambdaFunction',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                existingLambdaObj
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    functionName: 'NewLambdaFunction',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'ExistingLambdaFunction'
    });
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'NewLambdaFunction'
    });
});
test('Throws error when neither existingLambdaObj or lambdaFunctionProps is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiIntegrations: [
                {
                    id: 'MessagesHandler'
                }
            ]
        });
    };
    expect(app).toThrowError('One of existingLambdaObj or lambdaFunctionProps must be specified for the api integration with id: MessagesHandler');
});
test('Throws error when no api definition is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: lambda.Runtime.NODEJS_18_X,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when no api integration is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('At least one ApiIntegration must be specified in the apiIntegrations property');
});
test('Throws error when api definition s3 bucket is specified but s3 object key is missing', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = core_1.CreateScrapBucket(stack, {});
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionBucket,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when api definition s3 object key is specified but s3 bucket is missing', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when api is defined as asset and s3 bucket is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const apiDefinitionBucket = core_1.CreateScrapBucket(stack, {});
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionBucket,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when api is defined as asset and s3 key is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Two Constructs create APIs with different names', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const secondApiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'SecondOpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition-withCognitoAuth.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'second-test', {
        apiDefinitionAsset: secondApiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::ApiGateway::RestApi", 2);
    template.hasResourceProperties("AWS::ApiGateway::RestApi", {
        Name: {
            "Fn::Join": [
                "",
                [
                    "test-apigateway-lambda-",
                    {
                        "Fn::Select": [
                            2,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        Ref: "AWS::StackId"
                                    }
                                ]
                            }
                        ]
                    }
                ]
            ]
        }
    });
    template.hasResourceProperties("AWS::ApiGateway::RestApi", {
        Name: {
            "Fn::Join": [
                "",
                [
                    "second-test-",
                    {
                        "Fn::Select": [
                            2,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        Ref: "AWS::StackId"
                                    }
                                ]
                            }
                        ]
                    }
                ]
            ]
        }
    });
});
test('Confirm API definition uri is added to function name', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    const resources = JSON.parse(JSON.stringify(template)).Resources;
    expect(Object.keys(resources).find((element) => {
        return element.includes("MessagesHandler");
    })).toBeTruthy();
});
test('Confirm  that providing both lambdaFunction and functionProps is an error', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler',
        functionName: 'ExistingLambdaFunction',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const props = {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                },
                existingLambdaObj
            }
        ]
    };
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    expect(app).toThrowError(`Error - Cannot provide both lambdaFunctionProps and existingLambdaObj in an ApiIntegrationfor the api integration with id: MessagesHandler`);
});
//# sourceMappingURL=data:application/json;base64,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