"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stageDependencies = exports.bundle = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
function bundle(options) {
    const { entry, runtime, architecture, outputPathSuffix } = options;
    const stagedir = cdk.FileSystem.mkdtemp('python-bundling-');
    const hasDeps = stageDependencies(entry, stagedir);
    const depsCommand = chain([
        hasDeps ? `rsync -r ${exports.BUNDLER_DEPENDENCIES_CACHE}/. ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}` : '',
        `rsync -r . ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}`,
    ]);
    // Determine which dockerfile to use. When dependencies are present, we use a
    // Dockerfile that can create a cacheable layer. We can't use this Dockerfile
    // if there aren't dependencies or the Dockerfile will complain about missing
    // sources.
    const dockerfile = hasDeps
        ? 'Dockerfile.dependencies'
        : 'Dockerfile';
    // copy Dockerfile to workdir
    fs.copyFileSync(path.join(__dirname, dockerfile), path.join(stagedir, dockerfile));
    const image = cdk.DockerImage.fromBuild(stagedir, {
        buildArgs: {
            IMAGE: runtime.bundlingImage.image,
        },
        platform: architecture.dockerPlatform,
        file: dockerfile,
    });
    return lambda.Code.fromAsset(entry, {
        assetHashType: options.assetHashType,
        assetHash: options.assetHash,
        exclude: exports.DEPENDENCY_EXCLUDES,
        bundling: {
            image,
            command: ['bash', '-c', depsCommand],
        },
    });
}
exports.bundle = bundle;
/**
 * Checks to see if the `entry` directory contains a type of dependency that
 * we know how to install.
 */
function stageDependencies(entry, stagedir) {
    const prefixes = [
        'Pipfile',
        'pyproject',
        'poetry',
        'requirements.txt',
    ];
    let found = false;
    for (const file of fs.readdirSync(entry)) {
        for (const prefix of prefixes) {
            if (file.startsWith(prefix)) {
                fs.copyFileSync(path.join(entry, file), path.join(stagedir, file));
                found = true;
            }
        }
    }
    return found;
}
exports.stageDependencies = stageDependencies;
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLDhDQUE4QztBQUM5QyxxQ0FBcUM7QUFFckM7O0dBRUc7QUFDVSxRQUFBLG1CQUFtQixHQUFHLENBQUMsT0FBTyxDQUFDLENBQUM7QUFFN0M7O0dBRUc7QUFDVSxRQUFBLDBCQUEwQixHQUFHLG1CQUFtQixDQUFDO0FBbUU5RDs7R0FFRztBQUNILFNBQWdCLE1BQU0sQ0FBQyxPQUF3QjtJQUM3QyxNQUFNLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxZQUFZLEVBQUUsZ0JBQWdCLEVBQUUsR0FBRyxPQUFPLENBQUM7SUFFbkUsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUM1RCxNQUFNLE9BQU8sR0FBRyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFFbkQsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDO1FBQ3hCLE9BQU8sQ0FBQyxDQUFDLENBQUMsWUFBWSxrQ0FBMEIsTUFBTSxHQUFHLENBQUMsWUFBWSxDQUFDLG1CQUFtQixJQUFJLGdCQUFnQixFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7UUFDckgsY0FBYyxHQUFHLENBQUMsWUFBWSxDQUFDLG1CQUFtQixJQUFJLGdCQUFnQixFQUFFO0tBQ3pFLENBQUMsQ0FBQztJQUVILDZFQUE2RTtJQUM3RSw2RUFBNkU7SUFDN0UsNkVBQTZFO0lBQzdFLFdBQVc7SUFDWCxNQUFNLFVBQVUsR0FBRyxPQUFPO1FBQ3hCLENBQUMsQ0FBQyx5QkFBeUI7UUFDM0IsQ0FBQyxDQUFDLFlBQVksQ0FBQztJQUVqQiw2QkFBNkI7SUFDN0IsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxVQUFVLENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxVQUFVLENBQUMsQ0FBQyxDQUFDO0lBRW5GLE1BQU0sS0FBSyxHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRTtRQUNoRCxTQUFTLEVBQUU7WUFDVCxLQUFLLEVBQUUsT0FBTyxDQUFDLGFBQWEsQ0FBQyxLQUFLO1NBQ25DO1FBQ0QsUUFBUSxFQUFFLFlBQVksQ0FBQyxjQUFjO1FBQ3JDLElBQUksRUFBRSxVQUFVO0tBQ2pCLENBQUMsQ0FBQztJQUVILE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFO1FBQ2xDLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYTtRQUNwQyxTQUFTLEVBQUUsT0FBTyxDQUFDLFNBQVM7UUFDNUIsT0FBTyxFQUFFLDJCQUFtQjtRQUM1QixRQUFRLEVBQUU7WUFDUixLQUFLO1lBQ0wsT0FBTyxFQUFFLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUM7U0FDckM7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDO0FBdkNELHdCQXVDQztBQUVEOzs7R0FHRztBQUNILFNBQWdCLGlCQUFpQixDQUFDLEtBQWEsRUFBRSxRQUFnQjtJQUMvRCxNQUFNLFFBQVEsR0FBRztRQUNmLFNBQVM7UUFDVCxXQUFXO1FBQ1gsUUFBUTtRQUNSLGtCQUFrQjtLQUNuQixDQUFDO0lBRUYsSUFBSSxLQUFLLEdBQUcsS0FBSyxDQUFDO0lBQ2xCLEtBQUssTUFBTSxJQUFJLElBQUksRUFBRSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsRUFBRTtRQUN4QyxLQUFLLE1BQU0sTUFBTSxJQUFJLFFBQVEsRUFBRTtZQUM3QixJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQzNCLEVBQUUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztnQkFDbkUsS0FBSyxHQUFHLElBQUksQ0FBQzthQUNkO1NBQ0Y7S0FDRjtJQUVELE9BQU8sS0FBSyxDQUFDO0FBQ2YsQ0FBQztBQW5CRCw4Q0FtQkM7QUFFRCxTQUFTLEtBQUssQ0FBQyxRQUFrQjtJQUMvQixPQUFPLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBQ2hELENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4vKipcbiAqIERlcGVuZGVuY3kgZmlsZXMgdG8gZXhjbHVkZSBmcm9tIHRoZSBhc3NldCBoYXNoLlxuICovXG5leHBvcnQgY29uc3QgREVQRU5ERU5DWV9FWENMVURFUyA9IFsnKi5weWMnXTtcblxuLyoqXG4gKiBUaGUgbG9jYXRpb24gaW4gdGhlIGltYWdlIHRoYXQgdGhlIGJ1bmRsZXIgaW1hZ2UgY2FjaGVzIGRlcGVuZGVuY2llcy5cbiAqL1xuZXhwb3J0IGNvbnN0IEJVTkRMRVJfREVQRU5ERU5DSUVTX0NBQ0hFID0gJy92YXIvZGVwZW5kZW5jaWVzJztcblxuLyoqXG4gKiBPcHRpb25zIGZvciBidW5kbGluZ1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEJ1bmRsaW5nT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBFbnRyeSBwYXRoXG4gICAqL1xuICByZWFkb25seSBlbnRyeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcnVudGltZSBvZiB0aGUgbGFtYmRhIGZ1bmN0aW9uXG4gICAqL1xuICByZWFkb25seSBydW50aW1lOiBsYW1iZGEuUnVudGltZTtcblxuICAvKipcbiAgICogVGhlIHN5c3RlbSBhcmNoaXRlY3R1cmUgb2YgdGhlIGxhbWJkYSBmdW5jdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgYXJjaGl0ZWN0dXJlOiBsYW1iZGEuQXJjaGl0ZWN0dXJlO1xuXG4gIC8qKlxuICAgKiBPdXRwdXQgcGF0aCBzdWZmaXggKCdweXRob24nIGZvciBhIGxheWVyLCAnLicgb3RoZXJ3aXNlKVxuICAgKi9cbiAgcmVhZG9ubHkgb3V0cHV0UGF0aFN1ZmZpeDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEZXRlcm1pbmVzIGhvdyBhc3NldCBoYXNoIGlzIGNhbGN1bGF0ZWQuIEFzc2V0cyB3aWxsIGdldCByZWJ1aWxkIGFuZFxuICAgKiB1cGxvYWRlZCBvbmx5IGlmIHRoZWlyIGhhc2ggaGFzIGNoYW5nZWQuXG4gICAqXG4gICAqIElmIGFzc2V0IGhhc2ggaXMgc2V0IHRvIGBTT1VSQ0VgIChkZWZhdWx0KSwgdGhlbiBvbmx5IGNoYW5nZXMgdG8gdGhlIHNvdXJjZVxuICAgKiBkaXJlY3Rvcnkgd2lsbCBjYXVzZSB0aGUgYXNzZXQgdG8gcmVidWlsZC4gVGhpcyBtZWFucywgZm9yIGV4YW1wbGUsIHRoYXQgaW5cbiAgICogb3JkZXIgdG8gcGljayB1cCBhIG5ldyBkZXBlbmRlbmN5IHZlcnNpb24sIGEgY2hhbmdlIG11c3QgYmUgbWFkZSB0byB0aGVcbiAgICogc291cmNlIHRyZWUuIElkZWFsbHksIHRoaXMgY2FuIGJlIGltcGxlbWVudGVkIGJ5IGluY2x1ZGluZyBhIGRlcGVuZGVuY3lcbiAgICogbG9ja2ZpbGUgaW4geW91ciBzb3VyY2UgdHJlZSBvciB1c2luZyBmaXhlZCBkZXBlbmRlbmNpZXMuXG4gICAqXG4gICAqIElmIHRoZSBhc3NldCBoYXNoIGlzIHNldCB0byBgT1VUUFVUYCwgdGhlIGhhc2ggaXMgY2FsY3VsYXRlZCBhZnRlclxuICAgKiBidW5kbGluZy4gVGhpcyBtZWFucyB0aGF0IGFueSBjaGFuZ2UgaW4gdGhlIG91dHB1dCB3aWxsIGNhdXNlIHRoZSBhc3NldCB0b1xuICAgKiBiZSBpbnZhbGlkYXRlZCBhbmQgdXBsb2FkZWQuIEJlYXIgaW4gbWluZCB0aGF0IGBwaXBgIGFkZHMgdGltZXN0YW1wcyB0b1xuICAgKiBkZXBlbmRlbmNpZXMgaXQgaW5zdGFsbHMsIHdoaWNoIGltcGxpZXMgdGhhdCBpbiB0aGlzIG1vZGUgUHl0aG9uIGJ1bmRsZXNcbiAgICogd2lsbCBfYWx3YXlzXyBnZXQgcmVidWlsZCBhbmQgdXBsb2FkZWQuIE5vcm1hbGx5IHRoaXMgaXMgYW4gYW50aS1wYXR0ZXJuXG4gICAqIHNpbmNlIGJ1aWxkXG4gICAqXG4gICAqIEBkZWZhdWx0IEFzc2V0SGFzaFR5cGUuU09VUkNFIEJ5IGRlZmF1bHQsIGhhc2ggaXMgY2FsY3VsYXRlZCBiYXNlZCBvbiB0aGVcbiAgICogY29udGVudHMgb2YgdGhlIHNvdXJjZSBkaXJlY3RvcnkuIElmIGBhc3NldEhhc2hgIGlzIGFsc28gc3BlY2lmaWVkLCB0aGVcbiAgICogZGVmYXVsdCBpcyBgQ1VTVE9NYC4gVGhpcyBtZWFucyB0aGF0IG9ubHkgdXBkYXRlcyB0byB0aGUgc291cmNlIHdpbGwgY2F1c2VcbiAgICogdGhlIGFzc2V0IHRvIHJlYnVpbGQuXG4gICAqL1xuICByZWFkb25seSBhc3NldEhhc2hUeXBlPzogY2RrLkFzc2V0SGFzaFR5cGU7XG5cbiAgLyoqXG4gICAqIFNwZWNpZnkgYSBjdXN0b20gaGFzaCBmb3IgdGhpcyBhc3NldC4gSWYgYGFzc2V0SGFzaFR5cGVgIGlzIHNldCBpdCBtdXN0XG4gICAqIGJlIHNldCB0byBgQXNzZXRIYXNoVHlwZS5DVVNUT01gLiBGb3IgY29uc2lzdGVuY3ksIHRoaXMgY3VzdG9tIGhhc2ggd2lsbFxuICAgKiBiZSBTSEEyNTYgaGFzaGVkIGFuZCBlbmNvZGVkIGFzIGhleC4gVGhlIHJlc3VsdGluZyBoYXNoIHdpbGwgYmUgdGhlIGFzc2V0XG4gICAqIGhhc2guXG4gICAqXG4gICAqIE5PVEU6IHRoZSBoYXNoIGlzIHVzZWQgaW4gb3JkZXIgdG8gaWRlbnRpZnkgYSBzcGVjaWZpYyByZXZpc2lvbiBvZiB0aGUgYXNzZXQsIGFuZFxuICAgKiB1c2VkIGZvciBvcHRpbWl6aW5nIGFuZCBjYWNoaW5nIGRlcGxveW1lbnQgYWN0aXZpdGllcyByZWxhdGVkIHRvIHRoaXMgYXNzZXQgc3VjaCBhc1xuICAgKiBwYWNrYWdpbmcsIHVwbG9hZGluZyB0byBBbWF6b24gUzMsIGV0Yy4gSWYgeW91IGNob3NlIHRvIGN1c3RvbWl6ZSB0aGUgaGFzaCwgeW91IHdpbGxcbiAgICogbmVlZCB0byBtYWtlIHN1cmUgaXQgaXMgdXBkYXRlZCBldmVyeSB0aW1lIHRoZSBhc3NldCBjaGFuZ2VzLCBvciBvdGhlcndpc2UgaXQgaXNcbiAgICogcG9zc2libGUgdGhhdCBzb21lIGRlcGxveW1lbnRzIHdpbGwgbm90IGJlIGludmFsaWRhdGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGJhc2VkIG9uIGBhc3NldEhhc2hUeXBlYFxuICAgKi9cbiAgcmVhZG9ubHkgYXNzZXRIYXNoPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb2R1Y2UgYnVuZGxlZCBMYW1iZGEgYXNzZXQgY29kZVxuICovXG5leHBvcnQgZnVuY3Rpb24gYnVuZGxlKG9wdGlvbnM6IEJ1bmRsaW5nT3B0aW9ucyk6IGxhbWJkYS5Db2RlIHtcbiAgY29uc3QgeyBlbnRyeSwgcnVudGltZSwgYXJjaGl0ZWN0dXJlLCBvdXRwdXRQYXRoU3VmZml4IH0gPSBvcHRpb25zO1xuXG4gIGNvbnN0IHN0YWdlZGlyID0gY2RrLkZpbGVTeXN0ZW0ubWtkdGVtcCgncHl0aG9uLWJ1bmRsaW5nLScpO1xuICBjb25zdCBoYXNEZXBzID0gc3RhZ2VEZXBlbmRlbmNpZXMoZW50cnksIHN0YWdlZGlyKTtcblxuICBjb25zdCBkZXBzQ29tbWFuZCA9IGNoYWluKFtcbiAgICBoYXNEZXBzID8gYHJzeW5jIC1yICR7QlVORExFUl9ERVBFTkRFTkNJRVNfQ0FDSEV9Ly4gJHtjZGsuQXNzZXRTdGFnaW5nLkJVTkRMSU5HX09VVFBVVF9ESVJ9LyR7b3V0cHV0UGF0aFN1ZmZpeH1gIDogJycsXG4gICAgYHJzeW5jIC1yIC4gJHtjZGsuQXNzZXRTdGFnaW5nLkJVTkRMSU5HX09VVFBVVF9ESVJ9LyR7b3V0cHV0UGF0aFN1ZmZpeH1gLFxuICBdKTtcblxuICAvLyBEZXRlcm1pbmUgd2hpY2ggZG9ja2VyZmlsZSB0byB1c2UuIFdoZW4gZGVwZW5kZW5jaWVzIGFyZSBwcmVzZW50LCB3ZSB1c2UgYVxuICAvLyBEb2NrZXJmaWxlIHRoYXQgY2FuIGNyZWF0ZSBhIGNhY2hlYWJsZSBsYXllci4gV2UgY2FuJ3QgdXNlIHRoaXMgRG9ja2VyZmlsZVxuICAvLyBpZiB0aGVyZSBhcmVuJ3QgZGVwZW5kZW5jaWVzIG9yIHRoZSBEb2NrZXJmaWxlIHdpbGwgY29tcGxhaW4gYWJvdXQgbWlzc2luZ1xuICAvLyBzb3VyY2VzLlxuICBjb25zdCBkb2NrZXJmaWxlID0gaGFzRGVwc1xuICAgID8gJ0RvY2tlcmZpbGUuZGVwZW5kZW5jaWVzJ1xuICAgIDogJ0RvY2tlcmZpbGUnO1xuXG4gIC8vIGNvcHkgRG9ja2VyZmlsZSB0byB3b3JrZGlyXG4gIGZzLmNvcHlGaWxlU3luYyhwYXRoLmpvaW4oX19kaXJuYW1lLCBkb2NrZXJmaWxlKSwgcGF0aC5qb2luKHN0YWdlZGlyLCBkb2NrZXJmaWxlKSk7XG5cbiAgY29uc3QgaW1hZ2UgPSBjZGsuRG9ja2VySW1hZ2UuZnJvbUJ1aWxkKHN0YWdlZGlyLCB7XG4gICAgYnVpbGRBcmdzOiB7XG4gICAgICBJTUFHRTogcnVudGltZS5idW5kbGluZ0ltYWdlLmltYWdlLFxuICAgIH0sXG4gICAgcGxhdGZvcm06IGFyY2hpdGVjdHVyZS5kb2NrZXJQbGF0Zm9ybSxcbiAgICBmaWxlOiBkb2NrZXJmaWxlLFxuICB9KTtcblxuICByZXR1cm4gbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGVudHJ5LCB7XG4gICAgYXNzZXRIYXNoVHlwZTogb3B0aW9ucy5hc3NldEhhc2hUeXBlLFxuICAgIGFzc2V0SGFzaDogb3B0aW9ucy5hc3NldEhhc2gsXG4gICAgZXhjbHVkZTogREVQRU5ERU5DWV9FWENMVURFUyxcbiAgICBidW5kbGluZzoge1xuICAgICAgaW1hZ2UsXG4gICAgICBjb21tYW5kOiBbJ2Jhc2gnLCAnLWMnLCBkZXBzQ29tbWFuZF0sXG4gICAgfSxcbiAgfSk7XG59XG5cbi8qKlxuICogQ2hlY2tzIHRvIHNlZSBpZiB0aGUgYGVudHJ5YCBkaXJlY3RvcnkgY29udGFpbnMgYSB0eXBlIG9mIGRlcGVuZGVuY3kgdGhhdFxuICogd2Uga25vdyBob3cgdG8gaW5zdGFsbC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHN0YWdlRGVwZW5kZW5jaWVzKGVudHJ5OiBzdHJpbmcsIHN0YWdlZGlyOiBzdHJpbmcpOiBib29sZWFuIHtcbiAgY29uc3QgcHJlZml4ZXMgPSBbXG4gICAgJ1BpcGZpbGUnLFxuICAgICdweXByb2plY3QnLFxuICAgICdwb2V0cnknLFxuICAgICdyZXF1aXJlbWVudHMudHh0JyxcbiAgXTtcblxuICBsZXQgZm91bmQgPSBmYWxzZTtcbiAgZm9yIChjb25zdCBmaWxlIG9mIGZzLnJlYWRkaXJTeW5jKGVudHJ5KSkge1xuICAgIGZvciAoY29uc3QgcHJlZml4IG9mIHByZWZpeGVzKSB7XG4gICAgICBpZiAoZmlsZS5zdGFydHNXaXRoKHByZWZpeCkpIHtcbiAgICAgICAgZnMuY29weUZpbGVTeW5jKHBhdGguam9pbihlbnRyeSwgZmlsZSksIHBhdGguam9pbihzdGFnZWRpciwgZmlsZSkpO1xuICAgICAgICBmb3VuZCA9IHRydWU7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIGZvdW5kO1xufVxuXG5mdW5jdGlvbiBjaGFpbihjb21tYW5kczogc3RyaW5nW10pOiBzdHJpbmcge1xuICByZXR1cm4gY29tbWFuZHMuZmlsdGVyKGMgPT4gISFjKS5qb2luKCcgJiYgJyk7XG59XG4iXX0=