import argparse
import logging
import sys
from typing import Any

from ruamel.yaml import YAML
from ruamel.yaml.nodes import ScalarNode
from ruamel.yaml.representer import RoundTripRepresenter

from .dryrunner import TPVDryRunner
from .formatter import TPVConfigFormatter
from .linter import TPVConfigLinter, TPVLintError

log = logging.getLogger(__name__)


# https://stackoverflow.com/a/64933809
def repr_str(dumper: RoundTripRepresenter, data: str) -> ScalarNode:
    if "\n" in data:
        return dumper.represent_scalar("tag:yaml.org,2002:str", data, style="|")
    return dumper.represent_scalar("tag:yaml.org,2002:str", data)


# https://stackoverflow.com/a/37445121
def repr_none(dumper: RoundTripRepresenter, data: Any) -> ScalarNode:
    return dumper.represent_scalar("tag:yaml.org,2002:null", "")


def tpv_lint_config_file(args: Any) -> int:
    try:
        ignore = []
        if args.ignore is not None:
            ignore = [x.strip() for x in args.ignore.split(",")]
        tpv_linter = TPVConfigLinter.from_url_or_path(args.config, ignore, args.preserve_temp_code)
        tpv_linter.lint()
        log.info("lint successful.")
        return 0
    except TPVLintError:
        log.info("lint failed.")
        return 1


def tpv_format_config_file(args: Any) -> int:
    try:
        formatter = TPVConfigFormatter.from_url_or_path(args.config)
        yaml = YAML(typ="unsafe", pure=True)
        yaml.Representer = RoundTripRepresenter
        yaml.Representer.add_representer(str, repr_str)
        yaml.Representer.add_representer(type(None), repr_none)
        yaml.default_flow_style = False
        yaml.Emitter.MAX_SIMPLE_KEY_LENGTH = 1024
        yaml.dump(formatter.format(), sys.stdout)
        return 0
    except Exception:
        log.exception("format failed.")
        return 1


def tpv_dry_run_config_files(args: Any) -> None:
    dry_runner = TPVDryRunner.from_params(
        user_email=args.user,
        tool_id=args.tool,
        job_conf=args.job_conf,
        tpv_confs=args.config,
        roles=args.roles,
        history_tags=args.history_tags,
        input_size=args.input_size,
    )
    destination = dry_runner.run()
    yaml = YAML(typ="unsafe", pure=True)
    yaml.dump(destination, sys.stdout)


def create_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser()
    parser.set_defaults(func=lambda args: parser.print_help())

    # debugging and logging settings
    parser.add_argument(
        "-v",
        "--verbose",
        action="count",
        dest="verbosity_count",
        default=0,
        help="increases log verbosity for each occurrence.",
    )
    subparsers = parser.add_subparsers(metavar="<subcommand>")

    # File copy commands
    lint_parser = subparsers.add_parser(
        "lint",
        help="loads a TPV configuration file and checks it for syntax errors",
        description="The linter will check yaml syntax and compile python code blocks",
    )
    lint_parser.add_argument(
        "--ignore",
        type=str,
        help="Comma-separated list of lint error and warning codes to ignore",
    )
    lint_parser.add_argument(
        "--preserve-temp-code",
        default=False,
        action="store_true",
        help="Preserve the temporary code autogenerated during mypy type checks",
    )
    lint_parser.add_argument(
        "config",
        nargs="*",
        type=str,
        help="Path to the TPV config file to lint. Can be a local path or http url. If more than one file is provided,"
        " the files are merged in the order specified, with files specified later overriding earlier ones.",
    )
    lint_parser.set_defaults(func=tpv_lint_config_file)

    format_parser = subparsers.add_parser(
        "format",
        help="Reformats a TPV configuration file and prints it to stdout.",
        description="The formatter will reorder tools, users etc by name, moving defaults first",
    )
    format_parser.add_argument(
        "config",
        type=str,
        help="Path to the TPV config file to format. Can be a local path or http url.",
    )
    format_parser.set_defaults(func=tpv_format_config_file)

    dry_run_parser = subparsers.add_parser(
        "dry-run", help="Perform a dry run test of a TPV configuration.", description=""
    )
    dry_run_parser.add_argument("--job-conf", type=str, required=True, help="Galaxy job configuration file")
    dry_run_parser.add_argument("--input-size", type=int, help="Input dataset size (in GB)")
    dry_run_parser.add_argument(
        "--tool",
        type=str,
        default="_default_",
        help="Test mapping for Galaxy tool with given ID",
    )
    dry_run_parser.add_argument("--user", type=str, help="Test mapping for Galaxy user with username or email")
    dry_run_parser.add_argument("--roles", type=str, nargs="+", help="Add one or more Galaxy roles for user")
    dry_run_parser.add_argument(
        "--history-tags",
        type=str,
        nargs="+",
        help="Add one or more history tag names to user's history",
    )
    dry_run_parser.add_argument(
        "config",
        nargs="*",
        help="TPV configuration files, overrides tpv_config_files in Galaxy job configuration if provided",
    )
    dry_run_parser.set_defaults(func=tpv_dry_run_config_files)

    return parser


def configure_logging(verbosity_count: int) -> None:
    # Remove all handlers associated with the root logger object.
    # or basicConfig persists
    for handler in logging.root.handlers[:]:
        logging.root.removeHandler(handler)
    level = max(4 - verbosity_count, 1) * 10
    # set global logging level
    logging.basicConfig(stream=sys.stdout, level=level, format="%(levelname)-5s: %(name)s: %(message)s")
    # Set client log level
    if verbosity_count:
        log.setLevel(level)
    else:
        log.setLevel(logging.INFO)


def main() -> Any:
    parser = create_parser()
    args = parser.parse_args(sys.argv[1:])
    configure_logging(args.verbosity_count)
    # invoke subcommand
    return args.func(args)


if __name__ == "__main__":
    sys.exit(main())  # pragma: no cover
