import { IVpc, ISecurityGroup, SubnetSelection } from 'aws-cdk-lib/aws-ec2';
import { ContainerImage } from 'aws-cdk-lib/aws-ecs';
import { DockerImageCode } from 'aws-cdk-lib/aws-lambda';
import { Construct } from 'constructs';
/**
 * Properties for the `TokenInjectableDockerBuilder` construct.
 */
export interface TokenInjectableDockerBuilderProps {
    /**
     * The path to the directory containing the Dockerfile or source code.
     */
    readonly path: string;
    /**
     * Build arguments to pass to the Docker build process.
     * These are transformed into `--build-arg KEY=VALUE` flags.
     * @example
     * {
     *   TOKEN: 'my-secret-token',
     *   ENV: 'production'
     * }
     */
    readonly buildArgs?: {
        [key: string]: string;
    };
    /**
     * The ARN of the AWS Secrets Manager secret containing Docker login credentials.
     * This secret should store a JSON object with the following structure:
     * ```json
     * {
     *   "username": "my-docker-username",
     *   "password": "my-docker-password"
     * }
     * ```
     * If not provided (or not needed), the construct will skip Docker Hub login.
     *
     * **Note**: The secret must be in the same region as the stack.
     *
     * @example 'arn:aws:secretsmanager:us-east-1:123456789012:secret:DockerLoginSecret'
     */
    readonly dockerLoginSecretArn?: string;
    /**
     * The VPC in which the CodeBuild project will be deployed.
     * If provided, the CodeBuild project will be launched within the specified VPC.
     *
     * @default - No VPC is attached, and the CodeBuild project will use public internet.
     */
    readonly vpc?: IVpc;
    /**
     * The security groups to attach to the CodeBuild project.
     * These define the network access rules for the CodeBuild project.
     *
     * @default - No security groups are attached.
     */
    readonly securityGroups?: ISecurityGroup[];
    /**
     * The subnet selection to specify which subnets to use within the VPC.
     * Allows the user to select private, public, or isolated subnets.
     *
     * @default - All subnets in the VPC are used.
     */
    readonly subnetSelection?: SubnetSelection;
    /**
     * Custom commands to run during the install phase of CodeBuild.
     *
     * **Example**:
     * ```ts
     * installCommands: [
     *   'echo "Updating package lists..."',
     *   'apt-get update -y',
     *   'echo "Installing required packages..."',
     *   'apt-get install -y curl dnsutils',
     * ],
     * ```
     * @default - No additional install commands.
     */
    readonly installCommands?: string[];
    /**
     * Custom commands to run during the pre_build phase of CodeBuild.
     *
     * **Example**:
     * ```ts
     * preBuildCommands: [
     *   'echo "Fetching configuration from private API..."',
     *   'curl -o config.json https://api.example.com/config',
     * ],
     * ```
     * @default - No additional pre-build commands.
     */
    readonly preBuildCommands?: string[];
}
/**
 * A CDK construct to build and push Docker images to an ECR repository using
 * CodeBuild and Lambda custom resources, **then** retrieve the final image tag
 * so that ECS/Lambda references use the exact digest.
 */
export declare class TokenInjectableDockerBuilder extends Construct {
    /**
     * The ECR repository that stores the resulting Docker image.
     */
    private readonly ecrRepository;
    /**
     * An ECS-compatible container image referencing the tag
     * of the built Docker image.
     */
    readonly containerImage: ContainerImage;
    /**
     * A Lambda-compatible DockerImageCode referencing the the tag
     * of the built Docker image.
     */
    readonly dockerImageCode: DockerImageCode;
    /**
     * Creates a new `TokenInjectableDockerBuilder`.
     *
     * @param scope The scope in which to define this construct.
     * @param id The scoped construct ID.
     * @param props Configuration for building and pushing the Docker image.
     */
    constructor(scope: Construct, id: string, props: TokenInjectableDockerBuilderProps);
}
