"""
equipment.py
====================================
This is the equipment module of Copper. The module includes function that can be used by all types of equipment included in Copper.
"""

from copper.units import *
from copper.generator import *


class Equipment:
    def __init__(self):
        self.plotting_range = {}
        self.full_eff_alt = None
        self.full_eff_unit_alt = None
        self.full_eff = None
        self.full_eff_unit = None

    def convert_to_deg_c(value, unit="degF"):
        """Helper function to convert equipment data to degree F.

        :param float value: Value to convert to degree C
        :return: Vlue converted to degree C
        :rtype: float
        """
        if unit == "degF":
            curr_value = Units(value, unit)
            return curr_value.conversion("degC")
        else:
            return value

    def get_ref_values(self, out_var):
        """Get equipment reference/rated independent variables values (temperature and part load ratio) for an output variable (e.g., eir-f-t, eir-f-plr, cap-f-t).

        :param str out_var: Output variable
        :return: List of reference values
        :rtype: list

        """
        if "x2_norm" in list(self.plotting_range[out_var].keys()):
            return [
                self.plotting_range[out_var]["x1_norm"],
                self.plotting_range[out_var]["x2_norm"],
            ]
        else:
            return [self.plotting_range[out_var]["x1_norm"], 0.0]

    def get_eir_ref(self, alt):
        """Get the reference EIR (energy input ratio) of an equipment.

        :param bool alt: Specify if the alternative equipment efficiency should be used to calculate the EIR
        :return: Reference EIR
        :rtype: float

        """
        # Retrieve equipment efficiency and unit
        if alt:
            ref_eff = self.full_eff_alt
            ref_eff_unit = self.full_eff_unit_alt
        else:
            ref_eff = self.full_eff
            ref_eff_unit = self.full_eff_unit
        eff = Units(ref_eff, ref_eff_unit)
        return eff.conversion("eir")

    def generate_set_of_curves(
        self,
        method="best_match",
        pop_size=100,
        tol=0.0025,
        max_gen=300,
        vars="",
        sFac=0.5,
        retain=0.2,
        random_select=0.1,
        mutate=0.95,
        bounds=(6, 10),
        base_curves=[],
        random_seed=None,
        verbose=False,
        export_path="",
        export_format="json",
        export_name="",
        num_nearest_neighbors=10,
        max_restart=None,
    ):
        """Generate a set of curves for a particular equipment.

        :param str method: Method used to generate the set of curves: `best_match`, `nearest_neighbor`, or `weighted_average`
        :param int pop_size: Population size used by the genetic algorithm
        :param float tol: Tolerance used by the genetic algorithm to determine if the proposed solution is acceptable
                          The lower, the more stringent
        :param int max_gen: Maximum number of generation
        :param list() vars: List of variable to run the alorithm on
        :param float sFac: Linear fitness normalization factor, the higher the more aggressive the normalization will be
        :param float retain: Probability of retaining an individual in the next generation
        :param float random_select: Probability of randomly selecting an individual to be part of the next generation
        :param float mutate: Probability of an individual to be mutated in the next generation
        :param tuple() bounds: Random modification bounds
        :param int random_seed: Integer that is used to inialize the random number generator, this should be used when identical results are needed from one run to another
        :return: Set of curves object generated by the genetic algorithm that matches the  definition
        :rtype: SetofCurves

        """

        ga = Generator(
            self,
            method,
            pop_size,
            tol,
            max_gen,
            vars,
            sFac,
            retain,
            random_select,
            mutate,
            bounds,
            base_curves,
            random_seed,
            num_nearest_neighbors,
            max_restart,
        )
        set_of_curves = ga.generate_set_of_curves(verbose=verbose)

        if len(export_path):
            set_of_curves_obj = SetofCurves()
            set_of_curves_obj.curves = set_of_curves
            set_of_curves_obj.eqp = self
            set_of_curves_obj.export(
                path=export_path, fmt=export_format, name=export_name
            )
            return True
        else:
            return set_of_curves
