#!/usr/bin/env python3
"""
Comprehensive CLI Command Testing Script
========================================
Tests all documented commands to identify what actually works vs what's documented
"""

import subprocess
import sys
import json
from pathlib import Path
from typing import Dict, List, Any

class CommandTester:
    def __init__(self):
        self.results = {
            "working_commands": [],
            "broken_commands": [],
            "non_existent_commands": [],
            "command_count": 0,
            "working_count": 0,
            "broken_count": 0,
            "non_existent_count": 0
        }
        
    def run_command(self, command: str, expected_exit_code: int = 0) -> Dict[str, Any]:
        """Run a command and capture results"""
        try:
            result = subprocess.run(
                ["python3", "-m", "tusktsk.cli.main"] + command.split(),
                capture_output=True,
                text=True,
                timeout=30
            )
            
            return {
                "command": command,
                "exit_code": result.returncode,
                "stdout": result.stdout,
                "stderr": result.stderr,
                "success": result.returncode == expected_exit_code,
                "error": None
            }
        except subprocess.TimeoutExpired:
            return {
                "command": command,
                "exit_code": -1,
                "stdout": "",
                "stderr": "Command timed out",
                "success": False,
                "error": "timeout"
            }
        except Exception as e:
            return {
                "command": command,
                "exit_code": -1,
                "stdout": "",
                "stderr": str(e),
                "success": False,
                "error": str(e)
            }
    
    def test_command(self, command: str, description: str = "") -> None:
        """Test a single command"""
        self.results["command_count"] += 1
        print(f"Testing: {command}")
        
        result = self.run_command(command)
        
        if result["success"]:
            self.results["working_commands"].append({
                "command": command,
                "description": description,
                "result": result
            })
            self.results["working_count"] += 1
            print(f"  ✅ WORKING: {command}")
        else:
            if "not found" in result["stderr"] or "no such option" in result["stderr"]:
                self.results["non_existent_commands"].append({
                    "command": command,
                    "description": description,
                    "result": result
                })
                self.results["non_existent_count"] += 1
                print(f"  ❌ NON-EXISTENT: {command}")
            else:
                self.results["broken_commands"].append({
                    "command": command,
                    "description": description,
                    "result": result
                })
                self.results["broken_count"] += 1
                print(f"  🔧 BROKEN: {command}")
    
    def test_all_commands(self):
        """Test all documented commands"""
        print("🚀 Starting comprehensive CLI command testing...\n")
        
        # Test basic commands
        self.test_command("version", "Show version information")
        self.test_command("help", "Show help")
        
        # Test database commands
        self.test_command("db status", "Check database status")
        self.test_command("db init", "Initialize database")
        self.test_command("db console", "Open database console")
        self.test_command("db backup", "Backup database")
        self.test_command("db restore test.sql", "Restore database")
        self.test_command("db migrate test.sql", "Run migration")
        
        # Test development commands
        self.test_command("serve 3000", "Start development server")
        self.test_command("compile test.tsk", "Compile TSK file")
        self.test_command("optimize test.tsk", "Optimize TSK file")
        
        # Test test commands
        self.test_command("test all", "Run all tests")
        self.test_command("test parser", "Test parser")
        self.test_command("test fujsen", "Test FUJSEN")
        self.test_command("test sdk", "Test SDK")
        self.test_command("test performance", "Test performance")
        
        # Test service commands
        self.test_command("services start", "Start services")
        self.test_command("services stop", "Stop services")
        self.test_command("services restart", "Restart services")
        self.test_command("services status", "Service status")
        
        # Test cache commands
        self.test_command("cache clear", "Clear cache")
        self.test_command("cache status", "Cache status")
        self.test_command("cache warm", "Warm cache")
        self.test_command("cache memcached status", "Memcached status")
        self.test_command("cache memcached stats", "Memcached stats")
        self.test_command("cache memcached flush", "Memcached flush")
        self.test_command("cache memcached restart", "Memcached restart")
        self.test_command("cache memcached test", "Memcached test")
        self.test_command("cache distributed", "Distributed cache")
        
        # Test config commands
        self.test_command("config get app.name", "Get config value")
        self.test_command("config check", "Check config")
        self.test_command("config validate", "Validate config")
        self.test_command("config compile", "Compile config")
        self.test_command("config docs", "Config docs")
        self.test_command("config clear-cache", "Clear config cache")
        self.test_command("config stats", "Config stats")
        
        # Test binary commands
        self.test_command("binary compile test.tsk", "Compile to binary")
        self.test_command("binary execute test.tskb", "Execute binary")
        self.test_command("binary benchmark test.tsk", "Binary benchmark")
        self.test_command("binary optimize test.tskb", "Optimize binary")
        
        # Test peanuts commands
        self.test_command("peanuts compile test.peanuts", "Compile peanuts")
        self.test_command("peanuts auto-compile", "Auto-compile peanuts")
        self.test_command("peanuts load test.pnt", "Load peanuts")
        
        # Test AI commands
        self.test_command("ai claude 'test prompt'", "Claude AI")
        self.test_command("ai chatgpt 'test prompt'", "ChatGPT")
        self.test_command("ai custom 'https://api.example.com' 'test'", "Custom AI")
        self.test_command("ai config", "AI config")
        self.test_command("ai setup", "AI setup")
        self.test_command("ai test", "AI test")
        self.test_command("ai complete test.py", "AI complete")
        self.test_command("ai analyze test.py", "AI analyze")
        self.test_command("ai optimize test.py", "AI optimize")
        self.test_command("ai security test.py", "AI security")
        
        # Test CSS commands
        self.test_command("css expand test.css", "CSS expand")
        self.test_command("css map test.css", "CSS map")
        
        # Test license commands
        self.test_command("license check", "License check")
        self.test_command("license activate testkey", "License activate")
        
        # Test utility commands
        self.test_command("parse test.tsk", "Parse TSK file")
        self.test_command("validate test.tsk", "Validate TSK file")
        self.test_command("convert -i test.tsk -o test.json", "Convert format")
        self.test_command("get test.tsk app.name", "Get value")
        self.test_command("set test.tsk app.name 'test'", "Set value")
        
        # Test dependency commands
        self.test_command("deps install ai", "Install AI deps")
        self.test_command("deps list", "List deps")
        self.test_command("deps check", "Check deps")
        
        # Test non-existent commands from README
        print("\n🔍 Testing non-existent commands documented in README...")
        
        # AI non-existent commands
        self.test_command("ai models list", "List AI models")
        self.test_command("ai usage stats", "AI usage stats")
        self.test_command("ai keys rotate", "Rotate AI keys")
        self.test_command("ai cache clear", "Clear AI cache")
        self.test_command("ai benchmark", "AI benchmark")
        
        # Database non-existent commands
        self.test_command("db health", "Database health")
        self.test_command("db query 'SELECT 1'", "Database query")
        self.test_command("db rollback", "Database rollback")
        self.test_command("db optimize", "Database optimize")
        self.test_command("db vacuum", "Database vacuum")
        self.test_command("db reindex", "Database reindex")
        self.test_command("db analyze", "Database analyze")
        self.test_command("db connections", "Database connections")
        
        # Development non-existent commands
        self.test_command("serve --host 0.0.0.0", "Serve with host")
        self.test_command("serve --port 8080", "Serve with port")
        self.test_command("serve --ssl", "Serve with SSL")
        self.test_command("compile --watch test.tsk", "Compile with watch")
        self.test_command("optimize --profile test.tsk", "Optimize with profile")
        self.test_command("test unit", "Unit tests")
        self.test_command("test integration", "Integration tests")
        self.test_command("test coverage", "Test coverage")
        self.test_command("test watch", "Test watch")
        
        # Config non-existent commands
        self.test_command("config set app.name 'test'", "Set config")
        self.test_command("config list", "List config")
        self.test_command("config export config.json", "Export config")
        self.test_command("config import config.json", "Import config")
        self.test_command("config merge config2.tsk", "Merge config")
        
        # Security non-existent commands
        self.test_command("security auth login", "Security auth login")
        self.test_command("security auth logout", "Security auth logout")
        self.test_command("security auth status", "Security auth status")
        self.test_command("security auth refresh", "Security auth refresh")
        self.test_command("license validate", "License validate")
        self.test_command("license info", "License info")
        self.test_command("license transfer", "License transfer")
        self.test_command("license revoke", "License revoke")
        self.test_command("security scan ./", "Security scan")
        self.test_command("security encrypt test.txt", "Security encrypt")
        self.test_command("security decrypt test.enc", "Security decrypt")
        self.test_command("security audit", "Security audit")
        self.test_command("security hash test.txt", "Security hash")
        
        # Service non-existent commands
        self.test_command("services logs", "Service logs")
        self.test_command("services health", "Service health")
        
        # Binary non-existent commands
        self.test_command("binary info test.tskb", "Binary info")
        self.test_command("binary validate test.tskb", "Binary validate")
        self.test_command("binary extract test.tskb", "Binary extract")
        self.test_command("binary convert test.tsk", "Binary convert")
    
    def generate_report(self):
        """Generate comprehensive test report"""
        print(f"\n📊 TEST RESULTS SUMMARY")
        print(f"=" * 50)
        print(f"Total Commands Tested: {self.results['command_count']}")
        print(f"✅ Working Commands: {self.results['working_count']}")
        print(f"🔧 Broken Commands: {self.results['broken_count']}")
        print(f"❌ Non-existent Commands: {self.results['non_existent_count']}")
        print(f"Success Rate: {(self.results['working_count'] / self.results['command_count'] * 100):.1f}%")
        
        print(f"\n✅ WORKING COMMANDS ({self.results['working_count']}):")
        for cmd in self.results['working_commands']:
            print(f"  - {cmd['command']}")
        
        print(f"\n🔧 BROKEN COMMANDS ({self.results['broken_count']}):")
        for cmd in self.results['broken_commands']:
            print(f"  - {cmd['command']}: {cmd['result']['stderr'][:100]}...")
        
        print(f"\n❌ NON-EXISTENT COMMANDS ({self.results['non_existent_count']}):")
        for cmd in self.results['non_existent_commands']:
            print(f"  - {cmd['command']}")
        
        return self.results

def main():
    """Main test execution"""
    tester = CommandTester()
    tester.test_all_commands()
    results = tester.generate_report()
    
    # Save results to file
    with open("commandsneeded/validation_results.json", "w") as f:
        json.dump(results, f, indent=2)
    
    print(f"\n💾 Results saved to: commandsneeded/validation_results.json")
    
    return 0 if results['working_count'] > results['broken_count'] else 1

if __name__ == "__main__":
    sys.exit(main()) 