#!/usr/bin/env python3
"""
Test Working Commands Script
============================
Tests all available commands to identify which ones actually work
"""

import subprocess
import sys
import json
from pathlib import Path
from typing import Dict, List, Any

class WorkingCommandTester:
    def __init__(self):
        self.results = {
            "working_commands": [],
            "broken_commands": [],
            "command_count": 0,
            "working_count": 0,
            "broken_count": 0
        }
        
        # All available commands from CLI help
        self.available_commands = [
            # Basic commands
            ("version", "Version"),
            ("help", "Help"),
            
            # Database commands
            ("db status", "Database Status"),
            ("db init", "Database Init"),
            ("db backup test.sql", "Database Backup"),
            ("db restore test.sql", "Database Restore"),
            ("db migrate test.sql", "Database Migrate"),
            ("db rollback", "Database Rollback"),
            ("db optimize", "Database Optimize"),
            ("db vacuum", "Database Vacuum"),
            ("db reindex", "Database Reindex"),
            ("db analyze", "Database Analyze"),
            ("db connections", "Database Connections"),
            ("db console", "Database Console"),
            ("db health", "Database Health"),
            ("db query 'SELECT 1'", "Database Query"),
            
            # Development commands
            ("serve 3000", "Serve"),
            ("compile test.tsk", "Compile"),
            ("optimize test.tsk", "Optimize"),
            
            # Test commands
            ("test all", "Test All"),
            ("test parser", "Test Parser"),
            ("test fujsen", "Test FUJSEN"),
            ("test sdk", "Test SDK"),
            ("test performance", "Test Performance"),
            
            # Service commands
            ("services start", "Services Start"),
            ("services stop", "Services Stop"),
            ("services restart", "Services Restart"),
            ("services status", "Services Status"),
            
            # Cache commands
            ("cache clear", "Cache Clear"),
            ("cache status", "Cache Status"),
            ("cache warm", "Cache Warm"),
            ("cache memcached status", "Cache Memcached Status"),
            ("cache memcached stats", "Cache Memcached Stats"),
            ("cache memcached flush", "Cache Memcached Flush"),
            ("cache memcached restart", "Cache Memcached Restart"),
            ("cache memcached test", "Cache Memcached Test"),
            ("cache distributed", "Cache Distributed"),
            
            # Config commands
            ("config get app.name", "Config Get"),
            ("config check", "Config Check"),
            ("config validate", "Config Validate"),
            ("config compile", "Config Compile"),
            ("config docs app.tsk", "Config Docs"),
            ("config clear-cache", "Config Clear Cache"),
            ("config stats", "Config Stats"),
            
            # Binary commands
            ("binary compile test.tsk", "Binary Compile"),
            ("binary execute test.tskb", "Binary Execute"),
            ("binary benchmark test.tsk", "Binary Benchmark"),
            ("binary optimize test.tskb", "Binary Optimize"),
            
            # Peanuts commands
            ("peanuts compile test.peanuts", "Peanuts Compile"),
            ("peanuts auto-compile", "Peanuts Auto-compile"),
            ("peanuts load test.pnt", "Peanuts Load"),
            
            # AI commands
            ("ai claude 'test prompt'", "AI Claude"),
            ("ai chatgpt 'test prompt'", "AI ChatGPT"),
            ("ai custom 'https://api.example.com' 'test'", "AI Custom"),
            ("ai config", "AI Config"),
            ("ai setup", "AI Setup"),
            ("ai test", "AI Test"),
            ("ai complete test.py", "AI Complete"),
            ("ai analyze test.py", "AI Analyze"),
            ("ai optimize test.py", "AI Optimize"),
            ("ai security test.py", "AI Security"),
            
            # CSS commands
            ("css expand test.css", "CSS Expand"),
            ("css map test.css", "CSS Map"),
            
            # License commands
            ("license check", "License Check"),
            ("license activate testkey", "License Activate"),
            
            # Utility commands
            ("parse test.tsk", "Parse"),
            ("validate test.tsk", "Validate"),
            ("convert -i test.tsk -o test.json", "Convert"),
            ("get test.tsk app.name", "Get"),
            ("set test.tsk app.name 'test'", "Set"),
            
            # Dependency commands
            ("deps install ai", "Deps Install"),
            ("deps list", "Deps List"),
            ("deps check", "Deps Check"),
        ]
        
    def run_command(self, command: str, expected_exit_code: int = 0) -> Dict[str, Any]:
        """Run a command and capture results"""
        try:
            result = subprocess.run(
                ["python3", "-m", "tusktsk.cli.main"] + command.split(),
                capture_output=True,
                text=True,
                timeout=30
            )
            
            return {
                "command": command,
                "exit_code": result.returncode,
                "stdout": result.stdout,
                "stderr": result.stderr,
                "success": result.returncode == expected_exit_code,
                "error": None
            }
        except subprocess.TimeoutExpired:
            return {
                "command": command,
                "exit_code": -1,
                "stdout": "",
                "stderr": "Command timed out",
                "success": False,
                "error": "timeout"
            }
        except Exception as e:
            return {
                "command": command,
                "exit_code": -1,
                "stdout": "",
                "stderr": str(e),
                "success": False,
                "error": str(e)
            }
    
    def test_command(self, command: str, description: str = "") -> None:
        """Test a single command"""
        self.results["command_count"] += 1
        print(f"Testing: {command}")
        
        result = self.run_command(command)
        
        if result["success"]:
            self.results["working_commands"].append({
                "command": command,
                "description": description,
                "result": result
            })
            self.results["working_count"] += 1
            print(f"  ✅ WORKING: {command}")
        else:
            self.results["broken_commands"].append({
                "command": command,
                "description": description,
                "result": result
            })
            self.results["broken_count"] += 1
            print(f"  🔧 BROKEN: {command}")
    
    def test_all_commands(self):
        """Test all available commands"""
        print("🚀 Testing all available commands...\n")
        
        for command, description in self.available_commands:
            self.test_command(command, description)
    
    def generate_report(self):
        """Generate comprehensive test report"""
        print(f"\n📊 WORKING COMMANDS TEST RESULTS")
        print(f"=" * 60)
        print(f"Total Commands Tested: {self.results['command_count']}")
        print(f"✅ Working Commands: {self.results['working_count']}")
        print(f"🔧 Broken Commands: {self.results['broken_count']}")
        print(f"Success Rate: {(self.results['working_count'] / self.results['command_count'] * 100):.1f}%")
        
        print(f"\n✅ WORKING COMMANDS ({self.results['working_count']}):")
        for cmd in self.results['working_commands']:
            print(f"  - {cmd['command']}")
        
        print(f"\n🔧 BROKEN COMMANDS ({self.results['broken_count']}):")
        for cmd in self.results['broken_commands']:
            print(f"  - {cmd['command']}: {cmd['result']['stderr'][:100]}...")
        
        return self.results

def main():
    """Main test execution"""
    tester = WorkingCommandTester()
    tester.test_all_commands()
    results = tester.generate_report()
    
    # Save results to file
    with open("commandsneeded/working_commands_results.json", "w") as f:
        json.dump(results, f, indent=2)
    
    print(f"\n💾 Results saved to: commandsneeded/working_commands_results.json")
    
    return 0 if results['working_count'] > results['broken_count'] else 1

if __name__ == "__main__":
    sys.exit(main()) 