#!/usr/bin/env python3
"""
Comprehensive Test Suite for TuskLang Python SDK
================================================
Complete testing framework covering all operators, edge cases, performance, and integration

This test suite provides thorough coverage of the TuskLang Python SDK functionality,
ensuring reliability, performance, and correctness across all features.
"""

import unittest
import time
import tempfile
import os
import json
import threading
import asyncio
from typing import Dict, Any, List
from datetime import datetime, timedelta
import logging
import sys
import traceback

# Import the modules to test
from tsk_enhanced import TuskLangEnhanced
from real_operators import get_real_operator
from advanced_operators import (
    execute_advanced_operator, train_ml_model, predict_with_model,
    create_blockchain_block, create_quantum_qubit, process_dataset,
    create_enterprise_workflow, encrypt_data_secure
)


class TestSuiteConfig:
    """Configuration for the comprehensive test suite"""
    
    def __init__(self):
        self.verbose = True
        self.performance_threshold = 1.0  # seconds
        self.memory_threshold = 100  # MB
        self.test_timeout = 30  # seconds
        self.parallel_tests = 4
        self.generate_reports = True
        self.save_test_data = True


class PerformanceMonitor:
    """Monitor performance during tests"""
    
    def __init__(self):
        self.start_time = None
        self.end_time = None
        self.memory_usage = 0
        
    def start(self):
        """Start performance monitoring"""
        self.start_time = time.time()
        self.memory_usage = self._get_memory_usage()
        
    def stop(self):
        """Stop performance monitoring"""
        self.end_time = time.time()
        
    def get_execution_time(self) -> float:
        """Get execution time in seconds"""
        if self.start_time and self.end_time:
            return self.end_time - self.start_time
        return 0.0
        
    def get_memory_usage(self) -> float:
        """Get memory usage in MB"""
        return self._get_memory_usage() - self.memory_usage
        
    def _get_memory_usage(self) -> float:
        """Get current memory usage"""
        try:
            import psutil
            process = psutil.Process(os.getpid())
            return process.memory_info().rss / 1024 / 1024  # Convert to MB
        except ImportError:
            return 0.0


class TestResult:
    """Structured test result"""
    
    def __init__(self, test_name: str, success: bool, execution_time: float = 0.0, 
                 memory_usage: float = 0.0, error: str = None, metadata: Dict[str, Any] = None):
        self.test_name = test_name
        self.success = success
        self.execution_time = execution_time
        self.memory_usage = memory_usage
        self.error = error
        self.metadata = metadata or {}
        self.timestamp = datetime.now()


class ComprehensiveTestSuite(unittest.TestCase):
    """Comprehensive test suite for TuskLang Python SDK"""
    
    @classmethod
    def setUpClass(cls):
        """Set up test environment"""
        cls.config = TestSuiteConfig()
        cls.parser = TuskLangEnhanced()
        cls.performance_monitor = PerformanceMonitor()
        cls.test_results = []
        
        # Set up logging
        logging.basicConfig(
            level=logging.INFO,
            format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
            handlers=[
                logging.FileHandler('test_suite.log'),
                logging.StreamHandler(sys.stdout)
            ]
        )
        cls.logger = logging.getLogger('tusklang.test_suite')
        
        # Create test data directory
        cls.test_data_dir = tempfile.mkdtemp(prefix='tusklang_test_')
        cls.logger.info(f"Test data directory: {cls.test_data_dir}")
        
    @classmethod
    def tearDownClass(cls):
        """Clean up test environment"""
        import shutil
        if os.path.exists(cls.test_data_dir):
            shutil.rmtree(cls.test_data_dir)
        cls.logger.info("Test suite completed")
        
    def setUp(self):
        """Set up for each test"""
        self.performance_monitor.start()
        
    def tearDown(self):
        """Clean up after each test"""
        self.performance_monitor.stop()
        
        # Record test result
        result = TestResult(
            test_name=self._testMethodName,
            success=not hasattr(self, '_outcome') or not self._outcome.errors,
            execution_time=self.performance_monitor.get_execution_time(),
            memory_usage=self.performance_monitor.get_memory_usage(),
            error=str(self._outcome.errors[0][1]) if hasattr(self, '_outcome') and self._outcome.errors else None
        )
        self.test_results.append(result)
        
        # Log test result
        if result.success:
            self.logger.info(f"✓ {result.test_name} completed in {result.execution_time:.3f}s")
        else:
            self.logger.error(f"✗ {result.test_name} failed: {result.error}")


class BasicOperatorTests(ComprehensiveTestSuite):
    """Test basic TuskLang operators"""
    
    def test_001_variable_parsing(self):
        """Test variable parsing functionality"""
        # Test global variables
        self.parser.global_variables['test_var'] = 'test_value'
        result = self.parser.parse_value('$test_var')
        self.assertEqual(result, 'test_value')
        
        # Test section variables
        self.parser.current_section = 'test_section'
        self.parser.section_variables['test_section.local_var'] = 'local_value'
        result = self.parser.parse_value('local_var')
        self.assertEqual(result, 'local_value')
        
    def test_002_environment_variables(self):
        """Test environment variable parsing"""
        os.environ['TEST_ENV_VAR'] = 'env_value'
        result = self.parser.parse_value('@env("TEST_ENV_VAR")')
        self.assertEqual(result, 'env_value')
        
        # Test with default value
        result = self.parser.parse_value('@env("NONEXISTENT_VAR", "default_value")')
        self.assertEqual(result, 'default_value')
        
    def test_003_date_functions(self):
        """Test date function parsing"""
        result = self.parser.parse_value('@date("%Y")')
        self.assertIsInstance(result, str)
        self.assertEqual(len(result), 4)  # Year should be 4 digits
        
        result = self.parser.parse_value('@date("%Y-%m-%d")')
        self.assertIsInstance(result, str)
        self.assertIn('-', result)
        
    def test_004_array_parsing(self):
        """Test array parsing"""
        result = self.parser.parse_value('[1, 2, 3]')
        self.assertIsInstance(result, list)
        self.assertEqual(result, [1, 2, 3])
        
        result = self.parser.parse_value('["a", "b", "c"]')
        self.assertIsInstance(result, list)
        self.assertEqual(result, ['a', 'b', 'c'])
        
    def test_005_object_parsing(self):
        """Test object parsing"""
        result = self.parser.parse_value('{"key": "value", "number": 42}')
        self.assertIsInstance(result, dict)
        self.assertEqual(result['key'], 'value')
        self.assertEqual(result['number'], 42)
        
    def test_006_range_parsing(self):
        """Test range parsing"""
        result = self.parser.parse_value('8000-9000')
        self.assertIsInstance(result, dict)
        self.assertEqual(result['min'], 8000)
        self.assertEqual(result['max'], 9000)
        self.assertEqual(result['type'], 'range')
        
    def test_007_conditional_parsing(self):
        """Test conditional/ternary parsing"""
        result = self.parser.parse_value('true ? "yes" : "no"')
        self.assertEqual(result, 'yes')
        
        result = self.parser.parse_value('false ? "yes" : "no"')
        self.assertEqual(result, 'no')
        
    def test_008_string_concatenation(self):
        """Test string concatenation"""
        result = self.parser.parse_value('"Hello" + " " + "World"')
        self.assertEqual(result, 'Hello World')
        
    def test_009_file_operations(self):
        """Test file operations"""
        # Create temporary file
        test_file = os.path.join(self.test_data_dir, 'test.txt')
        with open(test_file, 'w') as f:
            f.write('test content')
            
        try:
            # Test file read
            result = self.parser.execute_operator('file', f'"read", "{test_file}"')
            self.assertEqual(result, 'test content')
            
            # Test file exists
            result = self.parser.execute_operator('file', f'"exists", "{test_file}"')
            self.assertTrue(result)
            
        finally:
            os.unlink(test_file)
            
    def test_010_json_operations(self):
        """Test JSON operations"""
        # Test JSON parse
        result = self.parser.execute_operator('json', '"parse", \'{"key": "value"}\'')
        self.assertIsInstance(result, dict)
        self.assertEqual(result['key'], 'value')
        
        # Test JSON stringify
        result = self.parser.execute_operator('json', '"stringify", \'{"key": "value"}\'')
        self.assertIsInstance(result, str)
        self.assertIn('"key"', result)
        
    def test_011_cache_operations(self):
        """Test cache operations"""
        result = self.parser.execute_operator('cache', '30, "cached_value"')
        self.assertEqual(result, 'cached_value')
        
        # Test cache retrieval
        result = self.parser.execute_operator('cache', '30, "cached_value"')
        self.assertEqual(result, 'cached_value')
        
    def test_012_string_operations(self):
        """Test string operations"""
        # Test uppercase
        result = self.parser.execute_operator('string', '"uppercase", "hello"')
        self.assertEqual(result, 'HELLO')
        
        # Test lowercase
        result = self.parser.execute_operator('string', '"lowercase", "WORLD"')
        self.assertEqual(result, 'world')
        
        # Test replace
        result = self.parser.execute_operator('string', '"replace", "hello world", "world", "python"')
        self.assertEqual(result, 'hello python')
        
    def test_013_regex_operations(self):
        """Test regex operations"""
        result = self.parser.execute_operator('regex', '"match", "hello world", "h.*o"')
        self.assertIsInstance(result, list)
        
        result = self.parser.execute_operator('regex', '"replace", "hello world", "o", "0"')
        self.assertEqual(result, 'hell0 w0rld')
        
    def test_014_hash_operations(self):
        """Test hash operations"""
        result = self.parser.execute_operator('hash', '"md5", "test"')
        self.assertIsInstance(result, str)
        self.assertEqual(len(result), 32)  # MD5 hash is 32 characters
        
        result = self.parser.execute_operator('hash', '"sha256", "test"')
        self.assertIsInstance(result, str)
        self.assertEqual(len(result), 64)  # SHA256 hash is 64 characters
        
    def test_015_base64_operations(self):
        """Test base64 operations"""
        # Test encode
        result = self.parser.execute_operator('base64', '"encode", "test"')
        self.assertIsInstance(result, str)
        
        # Test decode
        result = self.parser.execute_operator('base64', '"decode", "dGVzdA=="')
        self.assertEqual(result, 'test')


class AdvancedOperatorTests(ComprehensiveTestSuite):
    """Test advanced TuskLang operators"""
    
    def test_101_ai_ml_operators(self):
        """Test AI/ML operators"""
        # Test sentiment analysis
        result = execute_advanced_operator("ai_ml", "analyze_sentiment", text="I love this product!")
        self.assertTrue(result.success)
        self.assertIn("sentiment", result.data)
        
        # Test model training (if sklearn available)
        sample_data = [
            {"feature1": 1.0, "feature2": 2.0, "target": 1},
            {"feature1": 2.0, "feature2": 3.0, "target": 1},
            {"feature1": 3.0, "feature2": 1.0, "target": 0},
            {"feature1": 4.0, "feature2": 2.0, "target": 0}
        ]
        
        result = train_ml_model("random_forest", sample_data, "target", ["feature1", "feature2"])
        if result.success:
            model_id = result.data["model_id"]
            prediction = predict_with_model(model_id, [2.5, 2.5])
            self.assertTrue(prediction.success)
            self.assertIn("prediction", prediction.data)
        
    def test_102_blockchain_operators(self):
        """Test blockchain operators"""
        # Test transaction creation
        result = execute_advanced_operator("blockchain", "add_transaction", 
                                         sender="Alice", recipient="Bob", amount=10.0)
        self.assertTrue(result.success)
        
        # Test block creation
        transactions = [
            {"sender": "Alice", "recipient": "Bob", "amount": 10.0},
            {"sender": "Bob", "recipient": "Charlie", "amount": 5.0}
        ]
        
        result = create_blockchain_block(transactions)
        self.assertTrue(result.success)
        self.assertIn("block", result.data)
        
        # Test blockchain info
        result = execute_advanced_operator("blockchain", "get_info")
        self.assertTrue(result.success)
        self.assertIn("block_count", result.data)
        
    def test_103_quantum_operators(self):
        """Test quantum operators"""
        # Test qubit creation
        result = create_quantum_qubit("test_qubit")
        self.assertTrue(result.success)
        
        # Test Hadamard gate
        result = execute_advanced_operator("quantum", "apply_hadamard", qubit_id="test_qubit")
        self.assertTrue(result.success)
        
        # Test measurement
        result = execute_advanced_operator("quantum", "measure_qubit", qubit_id="test_qubit")
        self.assertTrue(result.success)
        self.assertIn("measurement", result.data)
        
    def test_104_data_processing_operators(self):
        """Test data processing operators"""
        sample_data = [
            {"name": "Alice", "age": 25, "city": "New York"},
            {"name": "Bob", "age": 30, "city": "Los Angeles"},
            {"name": "Charlie", "age": 35, "city": "New York"},
            {"name": "Diana", "age": 28, "city": "Chicago"}
        ]
        
        # Test dataset loading
        result = process_dataset("test_dataset", sample_data)
        self.assertTrue(result.success)
        
        # Test filtering
        result = execute_advanced_operator("data_processing", "filter_dataset", 
                                         dataset_id="test_dataset", condition="city == 'New York'")
        self.assertTrue(result.success)
        self.assertIn("filtered_data", result.data)
        
        # Test aggregation
        result = execute_advanced_operator("data_processing", "aggregate_data",
                                         dataset_id="test_dataset", group_by="city",
                                         aggregations={"age": "mean"})
        self.assertTrue(result.success)
        self.assertIn("aggregated_data", result.data)
        
    def test_105_enterprise_operators(self):
        """Test enterprise operators"""
        # Test workflow creation
        steps = [
            {"type": "data_transform", "input": "data", "transform": "filter", "condition": "age > 25"},
            {"type": "validation", "input": "data", "rules": [{"field": "name", "type": "required"}]},
            {"type": "notification", "message": "Workflow completed", "channel": "console"}
        ]
        
        result = create_enterprise_workflow("test_workflow", steps)
        self.assertTrue(result.success)
        
        # Test workflow execution
        context = {"data": [{"name": "Alice", "age": 30}, {"name": "Bob", "age": 20}]}
        result = execute_advanced_operator("enterprise", "execute_workflow", 
                                         workflow_id="test_workflow", context=context)
        self.assertTrue(result.success)
        
    def test_106_security_operators(self):
        """Test security operators"""
        # Test key generation
        result = execute_advanced_operator("security", "generate_key", key_id="test_key")
        self.assertTrue(result.success)
        
        # Test encryption
        result = encrypt_data_secure("test_key", "Hello, World!")
        self.assertTrue(result.success)
        self.assertIn("encrypted_data", result.data)
        
        # Test decryption
        encrypted_data = result.data["encrypted_data"]
        result = execute_advanced_operator("security", "decrypt", 
                                         key_id="test_key", encrypted_data=encrypted_data)
        self.assertTrue(result.success)
        self.assertEqual(result.data["decrypted_data"], "Hello, World!")
        
        # Test token creation
        result = execute_advanced_operator("security", "create_token",
                                         user_id="test_user", permissions=["read", "write"])
        self.assertTrue(result.success)
        self.assertIn("token", result.data)
        
        # Test token validation
        token = result.data["token"]
        result = execute_advanced_operator("security", "validate_token", token=token)
        self.assertTrue(result.success)


class PerformanceTests(ComprehensiveTestSuite):
    """Test performance characteristics"""
    
    def test_201_large_file_parsing(self):
        """Test parsing large files"""
        # Create large test file
        large_file = os.path.join(self.test_data_dir, 'large.tsk')
        with open(large_file, 'w') as f:
            for i in range(1000):
                f.write(f'section_{i} {{\n')
                f.write(f'  key_{i} = "value_{i}";\n')
                f.write(f'  number_{i} = {i};\n')
                f.write('}\n\n')
        
        try:
            start_time = time.time()
            result = self.parser.parse_file(large_file)
            end_time = time.time()
            
            self.assertIsInstance(result, dict)
            self.assertLess(end_time - start_time, self.config.performance_threshold)
            
        finally:
            os.unlink(large_file)
            
    def test_202_concurrent_operations(self):
        """Test concurrent operations"""
        def worker(worker_id):
            for i in range(10):
                self.parser.execute_operator('hash', f'"md5", "test_{worker_id}_{i}"')
        
        threads = []
        start_time = time.time()
        
        for i in range(self.config.parallel_tests):
            thread = threading.Thread(target=worker, args=(i,))
            threads.append(thread)
            thread.start()
        
        for thread in threads:
            thread.join()
        
        end_time = time.time()
        self.assertLess(end_time - start_time, self.config.performance_threshold * 2)
        
    def test_203_memory_usage(self):
        """Test memory usage"""
        initial_memory = self.performance_monitor._get_memory_usage()
        
        # Perform memory-intensive operations
        for i in range(1000):
            self.parser.execute_operator('cache', f'{i}, "value_{i}"')
        
        final_memory = self.performance_monitor._get_memory_usage()
        memory_increase = final_memory - initial_memory
        
        self.assertLess(memory_increase, self.config.memory_threshold)
        
    def test_204_operator_performance(self):
        """Test individual operator performance"""
        operators = [
            ('hash', '"md5", "test"'),
            ('base64', '"encode", "test"'),
            ('string', '"uppercase", "test"'),
            ('json', '"parse", \'{"key": "value"}\''),
            ('cache', '30, "test"')
        ]
        
        for operator, params in operators:
            start_time = time.time()
            for _ in range(100):
                self.parser.execute_operator(operator, params)
            end_time = time.time()
            
            avg_time = (end_time - start_time) / 100
            self.assertLess(avg_time, 0.01)  # Each operation should take less than 10ms


class IntegrationTests(ComprehensiveTestSuite):
    """Test integration scenarios"""
    
    def test_301_complete_workflow(self):
        """Test complete workflow integration"""
        # Create test configuration
        config_content = '''
        database {
            host = "localhost";
            port = 5432;
            name = "test_db";
        }
        
        api {
            endpoint = "https://api.example.com";
            timeout = 30;
            retries = 3;
        }
        
        security {
            encryption_key = @env("ENCRYPTION_KEY", "default_key");
            jwt_secret = @env("JWT_SECRET", "default_secret");
        }
        '''
        
        config_file = os.path.join(self.test_data_dir, 'config.tsk')
        with open(config_file, 'w') as f:
            f.write(config_content)
        
        try:
            # Parse configuration
            result = self.parser.parse_file(config_file)
            self.assertIsInstance(result, dict)
            self.assertIn('database', result)
            self.assertIn('api', result)
            self.assertIn('security', result)
            
            # Test configuration access
            db_host = self.parser.get('database.host')
            self.assertEqual(db_host, 'localhost')
            
            api_timeout = self.parser.get('api.timeout')
            self.assertEqual(api_timeout, 30)
            
        finally:
            os.unlink(config_file)
            
    def test_302_cross_file_communication(self):
        """Test cross-file communication"""
        # Create multiple configuration files
        file1_content = '''
        shared {
            version = "1.0.0";
            environment = "production";
        }
        '''
        
        file2_content = '''
        app {
            name = "test_app";
            version = @shared.tsk.get("version");
            env = @shared.tsk.get("environment");
        }
        '''
        
        file1 = os.path.join(self.test_data_dir, 'shared.tsk')
        file2 = os.path.join(self.test_data_dir, 'app.tsk')
        
        with open(file1, 'w') as f:
            f.write(file1_content)
        with open(file2, 'w') as f:
            f.write(file2_content)
        
        try:
            # Parse files
            self.parser.parse_file(file1)
            self.parser.parse_file(file2)
            
            # Test cross-file access
            app_name = self.parser.get('app.name')
            self.assertEqual(app_name, 'test_app')
            
            app_version = self.parser.get('app.version')
            self.assertEqual(app_version, '1.0.0')
            
            app_env = self.parser.get('app.env')
            self.assertEqual(app_env, 'production')
            
        finally:
            os.unlink(file1)
            os.unlink(file2)
            
    def test_303_advanced_operator_integration(self):
        """Test integration with advanced operators"""
        # Create a complex workflow
        workflow_steps = [
            {
                "type": "data_transform",
                "input": "raw_data",
                "transform": "filter",
                "condition": "age >= 18"
            },
            {
                "type": "validation",
                "input": "filtered_data",
                "rules": [
                    {"field": "name", "type": "required"},
                    {"field": "email", "type": "required"}
                ]
            },
            {
                "type": "notification",
                "message": "Data processing completed",
                "channel": "console"
            }
        ]
        
        # Create workflow
        result = create_enterprise_workflow("data_processing", workflow_steps)
        self.assertTrue(result.success)
        
        # Execute workflow with test data
        test_data = [
            {"name": "Alice", "age": 25, "email": "alice@example.com"},
            {"name": "Bob", "age": 17, "email": "bob@example.com"},
            {"name": "Charlie", "age": 30, "email": "charlie@example.com"}
        ]
        
        context = {"raw_data": test_data}
        result = execute_advanced_operator("enterprise", "execute_workflow",
                                         workflow_id="data_processing", context=context)
        self.assertTrue(result.success)
        
        # Verify results
        self.assertIn("results", result.data)
        self.assertEqual(len(result.data["results"]), 3)  # Three workflow steps
        
    def test_304_security_integration(self):
        """Test security integration"""
        # Generate encryption key
        result = execute_advanced_operator("security", "generate_key", key_id="integration_key")
        self.assertTrue(result.success)
        
        # Create access token
        result = execute_advanced_operator("security", "create_token",
                                         user_id="integration_user",
                                         permissions=["read", "write", "admin"])
        self.assertTrue(result.success)
        token = result.data["token"]
        
        # Encrypt sensitive data
        sensitive_data = "password123"
        result = encrypt_data_secure("integration_key", sensitive_data)
        self.assertTrue(result.success)
        encrypted_data = result.data["encrypted_data"]
        
        # Validate token and decrypt data
        result = execute_advanced_operator("security", "validate_token", token=token)
        self.assertTrue(result.success)
        
        result = execute_advanced_operator("security", "decrypt",
                                         key_id="integration_key",
                                         encrypted_data=encrypted_data)
        self.assertTrue(result.success)
        self.assertEqual(result.data["decrypted_data"], sensitive_data)


class ErrorHandlingTests(ComprehensiveTestSuite):
    """Test error handling and edge cases"""
    
    def test_401_invalid_syntax(self):
        """Test handling of invalid syntax"""
        # Test malformed JSON
        result = self.parser.parse_value('{"key": "value"')  # Missing closing brace
        self.assertIsInstance(result, str)  # Should return as string
        
        # Test malformed array
        result = self.parser.parse_value('[1, 2, 3')  # Missing closing bracket
        self.assertIsInstance(result, str)  # Should return as string
        
    def test_402_missing_variables(self):
        """Test handling of missing variables"""
        # Test missing global variable
        result = self.parser.parse_value('$nonexistent_var')
        self.assertEqual(result, '')  # Should return empty string
        
        # Test missing section variable
        self.parser.current_section = 'test_section'
        result = self.parser.parse_value('nonexistent_local_var')
        self.assertEqual(result, 'nonexistent_local_var')  # Should return as-is
        
    def test_403_invalid_operators(self):
        """Test handling of invalid operators"""
        # Test non-existent operator
        result = self.parser.execute_operator('nonexistent_operator', 'param')
        self.assertIsInstance(result, dict)  # Should return error dict
        
        # Test operator with invalid parameters
        result = self.parser.execute_operator('hash', 'invalid_params')
        self.assertIsInstance(result, dict)  # Should return error dict
        
    def test_404_file_errors(self):
        """Test handling of file errors"""
        # Test non-existent file
        result = self.parser.execute_operator('file', '"read", "/nonexistent/file.txt"')
        self.assertIsInstance(result, dict)  # Should return error dict
        
        # Test file without read permission
        no_read_file = os.path.join(self.test_data_dir, 'no_read.txt')
        with open(no_read_file, 'w') as f:
            f.write('test')
        os.chmod(no_read_file, 0o000)  # Remove all permissions
        
        try:
            result = self.parser.execute_operator('file', f'"read", "{no_read_file}"')
            self.assertIsInstance(result, dict)  # Should return error dict
        finally:
            os.chmod(no_read_file, 0o644)  # Restore permissions
            os.unlink(no_read_file)
            
    def test_405_advanced_operator_errors(self):
        """Test error handling in advanced operators"""
        # Test AI/ML with invalid data
        result = train_ml_model("invalid_model", [], "target", [])
        self.assertFalse(result.success)
        
        # Test blockchain with invalid transaction
        result = execute_advanced_operator("blockchain", "add_transaction",
                                         sender="", recipient="", amount=-1)
        self.assertTrue(result.success)  # Should handle gracefully
        
        # Test quantum with invalid qubit
        result = execute_advanced_operator("quantum", "measure_qubit", qubit_id="nonexistent")
        self.assertFalse(result.success)
        
        # Test security with invalid key
        result = encrypt_data_secure("nonexistent_key", "test")
        self.assertFalse(result.success)


class TestReportGenerator:
    """Generate comprehensive test reports"""
    
    def __init__(self, test_results: List[TestResult], config: TestSuiteConfig):
        self.test_results = test_results
        self.config = config
        
    def generate_report(self) -> Dict[str, Any]:
        """Generate comprehensive test report"""
        total_tests = len(self.test_results)
        passed_tests = sum(1 for result in self.test_results if result.success)
        failed_tests = total_tests - passed_tests
        
        total_execution_time = sum(result.execution_time for result in self.test_results)
        avg_execution_time = total_execution_time / total_tests if total_tests > 0 else 0
        
        total_memory_usage = sum(result.memory_usage for result in self.test_results)
        avg_memory_usage = total_memory_usage / total_tests if total_tests > 0 else 0
        
        # Group results by test category
        categories = {}
        for result in self.test_results:
            category = result.test_name.split('_')[0]
            if category not in categories:
                categories[category] = []
            categories[category].append(result)
        
        # Performance analysis
        performance_issues = []
        for result in self.test_results:
            if result.execution_time > self.config.performance_threshold:
                performance_issues.append({
                    "test": result.test_name,
                    "execution_time": result.execution_time,
                    "threshold": self.config.performance_threshold
                })
        
        # Error analysis
        errors = []
        for result in self.test_results:
            if not result.success and result.error:
                errors.append({
                    "test": result.test_name,
                    "error": result.error
                })
        
        report = {
            "summary": {
                "total_tests": total_tests,
                "passed_tests": passed_tests,
                "failed_tests": failed_tests,
                "success_rate": (passed_tests / total_tests * 100) if total_tests > 0 else 0,
                "total_execution_time": total_execution_time,
                "average_execution_time": avg_execution_time,
                "total_memory_usage": total_memory_usage,
                "average_memory_usage": avg_memory_usage
            },
            "categories": {
                category: {
                    "total": len(results),
                    "passed": sum(1 for r in results if r.success),
                    "failed": sum(1 for r in results if not r.success)
                }
                for category, results in categories.items()
            },
            "performance_issues": performance_issues,
            "errors": errors,
            "test_details": [
                {
                    "name": result.test_name,
                    "success": result.success,
                    "execution_time": result.execution_time,
                    "memory_usage": result.memory_usage,
                    "error": result.error,
                    "timestamp": result.timestamp.isoformat()
                }
                for result in self.test_results
            ],
            "generated_at": datetime.now().isoformat()
        }
        
        return report
    
    def save_report(self, report: Dict[str, Any], filename: str = None):
        """Save test report to file"""
        if filename is None:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            filename = f"test_report_{timestamp}.json"
        
        with open(filename, 'w') as f:
            json.dump(report, f, indent=2)
        
        print(f"Test report saved to: {filename}")


def run_comprehensive_test_suite():
    """Run the comprehensive test suite"""
    print("TuskLang Python SDK - Comprehensive Test Suite")
    print("=" * 60)
    
    # Create test suite
    loader = unittest.TestLoader()
    suite = unittest.TestSuite()
    
    # Add test classes
    test_classes = [
        BasicOperatorTests,
        AdvancedOperatorTests,
        PerformanceTests,
        IntegrationTests,
        ErrorHandlingTests
    ]
    
    for test_class in test_classes:
        tests = loader.loadTestsFromTestCase(test_class)
        suite.addTests(tests)
    
    # Run tests
    runner = unittest.TextTestRunner(verbosity=2)
    result = runner.run(suite)
    
    # Generate report
    if hasattr(result, 'test_results'):
        config = TestSuiteConfig()
        report_generator = TestReportGenerator(result.test_results, config)
        report = report_generator.generate_report()
        
        if config.generate_reports:
            report_generator.save_report(report)
        
        # Print summary
        print("\n" + "=" * 60)
        print("TEST SUITE SUMMARY")
        print("=" * 60)
        print(f"Total Tests: {report['summary']['total_tests']}")
        print(f"Passed: {report['summary']['passed_tests']}")
        print(f"Failed: {report['summary']['failed_tests']}")
        print(f"Success Rate: {report['summary']['success_rate']:.2f}%")
        print(f"Total Execution Time: {report['summary']['total_execution_time']:.3f}s")
        print(f"Average Execution Time: {report['summary']['average_execution_time']:.3f}s")
        
        if report['performance_issues']:
            print(f"\nPerformance Issues: {len(report['performance_issues'])}")
            for issue in report['performance_issues'][:5]:  # Show first 5
                print(f"  - {issue['test']}: {issue['execution_time']:.3f}s")
        
        if report['errors']:
            print(f"\nErrors: {len(report['errors'])}")
            for error in report['errors'][:5]:  # Show first 5
                print(f"  - {error['test']}: {error['error']}")
    
    return result.wasSuccessful()


if __name__ == "__main__":
    success = run_comprehensive_test_suite()
    sys.exit(0 if success else 1) 