#!/usr/bin/env python3
"""
Advanced AI-Powered Code Generation System for TuskLang Python SDK
Goal 8.1 Implementation - AI Code Generation System

Features:
- Natural language to TuskLang code generation
- Template-based code generation with AI enhancement
- Code validation and syntax checking
- Context-aware code suggestions
- Integration with existing TSK operators
- Safety and validation mechanisms
"""

import re
import json
import logging
import time
from typing import Dict, List, Optional, Any, Tuple
from dataclasses import dataclass, field
from enum import Enum
import random

# Import g7 components for integration
import sys
import os
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', 'g7'))
from ..g7.error_handler import handle_errors, error_handler
from ..g7.monitoring_framework import monitor_operation, monitoring_framework
from ..g7.performance_engine import optimize_operation, performance_engine

logger = logging.getLogger(__name__)

class CodeGenerationType(Enum):
    """Types of code generation"""
    OPERATOR = "operator"
    FUNCTION = "function"
    CLASS = "class"
    SCRIPT = "script"
    TEMPLATE = "template"
    CUSTOM = "custom"

@dataclass
class CodeGenerationRequest:
    """Request for code generation"""
    description: str
    generation_type: CodeGenerationType
    context: Dict[str, Any] = field(default_factory=dict)
    requirements: List[str] = field(default_factory=list)
    constraints: List[str] = field(default_factory=list)
    target_complexity: str = "medium"

@dataclass
class CodeGenerationResult:
    """Result of code generation"""
    code: str
    confidence: float
    generation_time: float
    suggestions: List[str] = field(default_factory=list)
    warnings: List[str] = field(default_factory=list)
    metadata: Dict[str, Any] = field(default_factory=dict)

class TuskLangTemplateEngine:
    """Template engine for TuskLang code generation"""
    
    def __init__(self):
        self.templates = self._load_templates()
        self.operator_patterns = self._load_operator_patterns()
    
    def _load_templates(self) -> Dict[str, str]:
        """Load code templates"""
        return {
            "operator": """
@operator("{operator_name}")
def {operator_name}_operator(context: Dict[str, Any]) -> Any:
    \"\"\"
    {description}
    \"\"\"
    try:
        {implementation}
        return result
    except Exception as e:
        logger.error(f"Error in {operator_name}_operator: {{e}}")
        raise
""",
            "function": """
def {function_name}({parameters}) -> {return_type}:
    \"\"\"
    {description}
    
    Args:
        {param_docs}
    
    Returns:
        {return_type}: {return_description}
    \"\"\"
    {implementation}
    return result
""",
            "class": """
class {class_name}:
    \"\"\"
    {description}
    \"\"\"
    
    def __init__(self, {init_params}):
        {init_implementation}
    
    {methods}
""",
            "script": """
#!/usr/bin/env python3
\"\"\"
{description}
\"\"\"

import logging
from typing import Dict, Any

logger = logging.getLogger(__name__)

{implementation}

if __name__ == "__main__":
    main()
"""
        }
    
    def _load_operator_patterns(self) -> Dict[str, str]:
        """Load common operator patterns"""
        return {
            "math": "result = context.get('value', 0) + context.get('addend', 0)",
            "string": "result = str(context.get('text', '')) + str(context.get('suffix', ''))",
            "list": "result = list(context.get('items', [])) + list(context.get('more_items', []))",
            "dict": "result = dict(context.get('base', {})) | dict(context.get('updates', {}))",
            "conditional": "result = context.get('condition', False) and context.get('true_value') or context.get('false_value')",
            "loop": "result = [item * context.get('multiplier', 1) for item in context.get('items', [])]",
            "validation": "result = all(isinstance(item, context.get('expected_type', str)) for item in context.get('items', []))",
            "transformation": "result = [context.get('transform_func', lambda x: x)(item) for item in context.get('items', [])]"
        }
    
    def generate_operator(self, name: str, description: str, pattern: str = "math") -> str:
        """Generate operator code"""
        implementation = self.operator_patterns.get(pattern, self.operator_patterns["math"])
        
        return self.templates["operator"].format(
            operator_name=name,
            description=description,
            implementation=implementation
        )
    
    def generate_function(self, name: str, description: str, parameters: str = "", 
                         return_type: str = "Any", implementation: str = "pass") -> str:
        """Generate function code"""
        param_docs = "\n        ".join([f"{param}: Parameter description" for param in parameters.split(", ") if param])
        
        return self.templates["function"].format(
            function_name=name,
            parameters=parameters,
            return_type=return_type,
            description=description,
            param_docs=param_docs,
            return_description="Result description",
            implementation=implementation
        )

class AICodeGenerator:
    """AI-powered code generation system"""
    
    def __init__(self):
        self.template_engine = TuskLangTemplateEngine()
        self.code_patterns = self._load_code_patterns()
        self.safety_validator = CodeSafetyValidator()
        self.context_analyzer = ContextAnalyzer()
        
        # Register with monitoring
        monitoring_framework.record_metric("ai_code_generator_initialized", 1.0)
    
    def _load_code_patterns(self) -> Dict[str, List[str]]:
        """Load common code patterns for AI generation"""
        return {
            "data_processing": [
                "filter_data", "transform_data", "aggregate_data", "validate_data"
            ],
            "api_integration": [
                "make_request", "parse_response", "handle_errors", "rate_limiting"
            ],
            "file_operations": [
                "read_file", "write_file", "process_file", "backup_file"
            ],
            "database_operations": [
                "query_database", "insert_data", "update_data", "delete_data"
            ],
            "security": [
                "encrypt_data", "decrypt_data", "validate_input", "sanitize_output"
            ],
            "performance": [
                "cache_result", "optimize_query", "batch_process", "parallel_execute"
            ]
        }
    
    @monitor_operation("generate_code")
    @handle_errors(retry=True)
    def generate_code(self, request: CodeGenerationRequest) -> CodeGenerationResult:
        """Generate code based on natural language description"""
        start_time = time.time()
        
        try:
            # Analyze context and requirements
            context_info = self.context_analyzer.analyze(request)
            
            # Generate code based on type
            if request.generation_type == CodeGenerationType.OPERATOR:
                code = self._generate_operator_code(request, context_info)
            elif request.generation_type == CodeGenerationType.FUNCTION:
                code = self._generate_function_code(request, context_info)
            elif request.generation_type == CodeGenerationType.CLASS:
                code = self._generate_class_code(request, context_info)
            elif request.generation_type == CodeGenerationType.SCRIPT:
                code = self._generate_script_code(request, context_info)
            else:
                code = self._generate_custom_code(request, context_info)
            
            # Validate generated code
            validation_result = self.safety_validator.validate_code(code, request)
            
            # Calculate confidence based on validation and context
            confidence = self._calculate_confidence(context_info, validation_result)
            
            # Generate suggestions and warnings
            suggestions = self._generate_suggestions(request, context_info)
            warnings = validation_result.warnings
            
            generation_time = time.time() - start_time
            
            result = CodeGenerationResult(
                code=code,
                confidence=confidence,
                generation_time=generation_time,
                suggestions=suggestions,
                warnings=warnings,
                metadata={
                    "context_info": context_info,
                    "validation_result": validation_result.__dict__
                }
            )
            
            # Record metrics
            monitoring_framework.record_metric("code_generation_success", 1.0)
            monitoring_framework.record_metric("code_generation_time", generation_time)
            monitoring_framework.record_metric("code_generation_confidence", confidence)
            
            return result
            
        except Exception as e:
            monitoring_framework.record_metric("code_generation_error", 1.0)
            error_handler.handle_error(e, {"request": request.__dict__})
            raise
    
    def _generate_operator_code(self, request: CodeGenerationRequest, context_info: Dict[str, Any]) -> str:
        """Generate operator code"""
        # Extract operator name from description
        operator_name = self._extract_operator_name(request.description)
        
        # Determine pattern based on context
        pattern = self._determine_operator_pattern(request.description, context_info)
        
        return self.template_engine.generate_operator(
            name=operator_name,
            description=request.description,
            pattern=pattern
        )
    
    def _generate_function_code(self, request: CodeGenerationRequest, context_info: Dict[str, Any]) -> str:
        """Generate function code"""
        # Extract function name and parameters
        function_name = self._extract_function_name(request.description)
        parameters = self._extract_parameters(request.description, context_info)
        return_type = self._determine_return_type(request.description, context_info)
        implementation = self._generate_implementation(request.description, context_info)
        
        return self.template_engine.generate_function(
            name=function_name,
            description=request.description,
            parameters=parameters,
            return_type=return_type,
            implementation=implementation
        )
    
    def _generate_class_code(self, request: CodeGenerationRequest, context_info: Dict[str, Any]) -> str:
        """Generate class code"""
        class_name = self._extract_class_name(request.description)
        init_params = self._extract_init_parameters(request.description)
        methods = self._generate_class_methods(request.description, context_info)
        
        return self.template_engine.templates["class"].format(
            class_name=class_name,
            description=request.description,
            init_params=init_params,
            init_implementation="self.initialized = True",
            methods=methods
        )
    
    def _generate_script_code(self, request: CodeGenerationRequest, context_info: Dict[str, Any]) -> str:
        """Generate script code"""
        implementation = self._generate_script_implementation(request.description, context_info)
        
        return self.template_engine.templates["script"].format(
            description=request.description,
            implementation=implementation
        )
    
    def _generate_custom_code(self, request: CodeGenerationRequest, context_info: Dict[str, Any]) -> str:
        """Generate custom code based on description"""
        # Use AI-like pattern matching to generate custom code
        patterns = self._identify_code_patterns(request.description)
        
        if "data_processing" in patterns:
            return self._generate_data_processing_code(request, context_info)
        elif "api_integration" in patterns:
            return self._generate_api_integration_code(request, context_info)
        elif "file_operations" in patterns:
            return self._generate_file_operations_code(request, context_info)
        else:
            return self._generate_generic_code(request, context_info)
    
    def _extract_operator_name(self, description: str) -> str:
        """Extract operator name from description"""
        # Simple extraction - in real AI system, this would use NLP
        words = description.lower().split()
        if "operator" in words:
            idx = words.index("operator")
            if idx > 0:
                return words[idx - 1]
        
        # Fallback to first meaningful word
        for word in words:
            if word not in ["a", "an", "the", "that", "this", "for", "to", "in", "on", "at"]:
                return word
        
        return "custom_operator"
    
    def _extract_function_name(self, description: str) -> str:
        """Extract function name from description"""
        # Convert description to function name
        words = description.lower().split()
        meaningful_words = [w for w in words if w not in ["a", "an", "the", "that", "this", "for", "to", "in", "on", "at"]]
        
        if meaningful_words:
            return "_".join(meaningful_words[:3])  # Limit to 3 words
        
        return "custom_function"
    
    def _extract_class_name(self, description: str) -> str:
        """Extract class name from description"""
        # Convert description to class name (PascalCase)
        words = description.lower().split()
        meaningful_words = [w for w in words if w not in ["a", "an", "the", "that", "this", "for", "to", "in", "on", "at"]]
        
        if meaningful_words:
            return "".join(word.capitalize() for word in meaningful_words[:3])
        
        return "CustomClass"
    
    def _determine_operator_pattern(self, description: str, context_info: Dict[str, Any]) -> str:
        """Determine operator pattern based on description"""
        description_lower = description.lower()
        
        if any(word in description_lower for word in ["add", "sum", "plus", "math"]):
            return "math"
        elif any(word in description_lower for word in ["string", "text", "concatenate"]):
            return "string"
        elif any(word in description_lower for word in ["list", "array", "collection"]):
            return "list"
        elif any(word in description_lower for word in ["dict", "dictionary", "map"]):
            return "dict"
        elif any(word in description_lower for word in ["if", "condition", "check"]):
            return "conditional"
        elif any(word in description_lower for word in ["loop", "iterate", "each"]):
            return "loop"
        elif any(word in description_lower for word in ["validate", "check", "verify"]):
            return "validation"
        elif any(word in description_lower for word in ["transform", "convert", "change"]):
            return "transformation"
        else:
            return "math"  # Default pattern
    
    def _extract_parameters(self, description: str, context_info: Dict[str, Any]) -> str:
        """Extract parameters from description"""
        # Simple parameter extraction - in real AI system, this would use NLP
        if "parameter" in description.lower() or "param" in description.lower():
            return "param1, param2"
        elif "input" in description.lower():
            return "input_data"
        else:
            return "data"
    
    def _determine_return_type(self, description: str, context_info: Dict[str, Any]) -> str:
        """Determine return type from description"""
        description_lower = description.lower()
        
        if any(word in description_lower for word in ["list", "array", "collection"]):
            return "List[Any]"
        elif any(word in description_lower for word in ["dict", "dictionary", "map"]):
            return "Dict[str, Any]"
        elif any(word in description_lower for word in ["bool", "boolean", "check", "validate"]):
            return "bool"
        elif any(word in description_lower for word in ["int", "number", "count"]):
            return "int"
        elif any(word in description_lower for word in ["str", "string", "text"]):
            return "str"
        else:
            return "Any"
    
    def _generate_implementation(self, description: str, context_info: Dict[str, Any]) -> str:
        """Generate implementation based on description"""
        # Simple implementation generation - in real AI system, this would be more sophisticated
        if "process" in description.lower():
            return "result = data.upper() if isinstance(data, str) else str(data)"
        elif "calculate" in description.lower():
            return "result = sum(data) if isinstance(data, (list, tuple)) else data"
        elif "validate" in description.lower():
            return "result = isinstance(data, (str, int, float, bool))"
        else:
            return "result = data"
    
    def _identify_code_patterns(self, description: str) -> List[str]:
        """Identify code patterns in description"""
        description_lower = description.lower()
        patterns = []
        
        for pattern, keywords in self.code_patterns.items():
            if any(keyword in description_lower for keyword in keywords):
                patterns.append(pattern)
        
        return patterns
    
    def _calculate_confidence(self, context_info: Dict[str, Any], validation_result: Any) -> float:
        """Calculate confidence score for generated code"""
        base_confidence = 0.7
        
        # Adjust based on context clarity
        if context_info.get("clarity_score", 0) > 0.8:
            base_confidence += 0.1
        
        # Adjust based on validation
        if hasattr(validation_result, 'warnings') and not validation_result.warnings:
            base_confidence += 0.1
        elif isinstance(validation_result, dict) and not validation_result.get('warnings', []):
            base_confidence += 0.1
        
        # Adjust based on pattern matching
        if context_info.get("pattern_match_score", 0) > 0.8:
            base_confidence += 0.1
        
        return min(base_confidence, 1.0)
    
    def _generate_suggestions(self, request: CodeGenerationRequest, context_info: Dict[str, Any]) -> List[str]:
        """Generate suggestions for improvement"""
        suggestions = []
        
        if context_info.get("clarity_score", 0) < 0.7:
            suggestions.append("Consider providing more specific requirements for better code generation")
        
        if not request.requirements:
            suggestions.append("Add specific requirements to improve code quality")
        
        if request.target_complexity == "high" and context_info.get("complexity_score", 0) < 0.6:
            suggestions.append("Consider adding more complex requirements for advanced functionality")
        
        return suggestions
    
    def _extract_init_parameters(self, description: str) -> str:
        """Extract initialization parameters for classes"""
        if "parameter" in description.lower() or "param" in description.lower():
            return "param1, param2"
        return "self"
    
    def _generate_script_implementation(self, description: str, context_info: Dict[str, Any]) -> str:
        """Generate script implementation"""
        return self._generate_implementation(description, context_info)
    
    def _generate_class_methods(self, description: str, context_info: Dict[str, Any]) -> str:
        """Generate class methods"""
        methods = []
        
        # Add __init__ method
        init_params = self._extract_init_parameters(description)
        methods.append(f"""
    def __init__(self, {init_params}):
        \"\"\"
        Initialize the class
        \"\"\"
        self.data = {init_params.split(',')[0] if init_params != 'self' else 'None'}
""")
        
        # Add a process method
        methods.append(f"""
    def process(self):
        \"\"\"
        Process the data
        \"\"\"
        return self.data
""")
        
        return "\n".join(methods)

class CodeSafetyValidator:
    """Validator for generated code safety"""
    
    def __init__(self):
        self.dangerous_patterns = [
            r"eval\s*\(",
            r"exec\s*\(",
            r"__import__\s*\(",
            r"open\s*\(",
            r"subprocess\.",
            r"os\.system\s*\(",
            r"input\s*\(",
        ]
        self.safety_checks = [
            "syntax_validation",
            "dangerous_pattern_detection",
            "basic_safety_analysis"
        ]
    
    def validate_code(self, code: str, request: CodeGenerationRequest) -> Any:
        """Validate generated code for safety"""
        warnings = []
        
        # Check for dangerous patterns
        for pattern in self.dangerous_patterns:
            if re.search(pattern, code):
                warnings.append(f"Potentially dangerous pattern detected: {pattern}")
        
        # Check for basic syntax
        try:
            compile(code, '<string>', 'exec')
        except SyntaxError as e:
            warnings.append(f"Syntax error: {e}")
        
        # Check for common issues
        if "pass" in code and len(code.strip()) < 50:
            warnings.append("Generated code may be too simple - consider adding implementation")
        
        return type('ValidationResult', (), {
            'warnings': warnings,
            'is_safe': len(warnings) == 0
        })()

class ContextAnalyzer:
    """Analyzer for understanding generation context"""
    
    def analyze(self, request: CodeGenerationRequest) -> Dict[str, Any]:
        """Analyze generation request context"""
        description = request.description.lower()
        
        # Calculate clarity score
        clarity_score = self._calculate_clarity_score(description)
        
        # Calculate complexity score
        complexity_score = self._calculate_complexity_score(description)
        
        # Calculate pattern match score
        pattern_match_score = self._calculate_pattern_match_score(description)
        
        return {
            "clarity_score": clarity_score,
            "complexity_score": complexity_score,
            "pattern_match_score": pattern_match_score,
            "word_count": len(description.split()),
            "has_requirements": len(request.requirements) > 0,
            "has_constraints": len(request.constraints) > 0
        }
    
    def _calculate_clarity_score(self, description: str) -> float:
        """Calculate clarity score of description"""
        words = description.split()
        if len(words) < 5:
            return 0.3
        elif len(words) < 10:
            return 0.6
        else:
            return 0.9
    
    def _calculate_complexity_score(self, description: str) -> float:
        """Calculate complexity score of description"""
        complexity_indicators = [
            "complex", "advanced", "sophisticated", "multiple", "various",
            "integration", "system", "framework", "architecture"
        ]
        
        score = 0.3  # Base score
        for indicator in complexity_indicators:
            if indicator in description:
                score += 0.1
        
        return min(score, 1.0)
    
    def _calculate_pattern_match_score(self, description: str) -> float:
        """Calculate pattern match score"""
        patterns = [
            "operator", "function", "class", "method", "script",
            "process", "calculate", "validate", "transform", "generate"
        ]
        
        matches = sum(1 for pattern in patterns if pattern in description)
        return min(matches / len(patterns), 1.0)

# Global AI code generator instance
ai_code_generator = AICodeGenerator()

def generate_tsk_code(description: str, code_type: str = "operator", **kwargs) -> CodeGenerationResult:
    """Convenience function for generating TuskLang code"""
    request = CodeGenerationRequest(
        description=description,
        generation_type=CodeGenerationType(code_type),
        **kwargs
    )
    
    return ai_code_generator.generate_code(request) 