#!/usr/bin/env python3
"""
Predictive Performance Optimization Engine for TuskLang Python SDK
Goal 8.2 Implementation - Predictive Performance Optimization

Features:
- ML-based performance prediction and optimization
- Real-time performance monitoring and analysis
- Predictive alerts and recommendations
- Automatic optimization suggestions
- Performance trend analysis
- Resource usage prediction
"""

import time
import json
import logging
import threading
from typing import Dict, List, Optional, Any, Tuple
from dataclasses import dataclass, field
from collections import deque, defaultdict
import statistics
import math

# Import g7 components for integration
import sys
import os
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', 'g7'))
from ..g7.error_handler import handle_errors, error_handler
from ..g7.monitoring_framework import monitor_operation, monitoring_framework
from ..g7.performance_engine import optimize_operation, performance_engine

logger = logging.getLogger(__name__)

@dataclass
class PerformanceMetric:
    """Performance metric data point"""
    timestamp: float
    operation_name: str
    duration: float
    memory_usage: float
    cpu_usage: float
    error_count: int
    success_count: int
    context: Dict[str, Any] = field(default_factory=dict)

@dataclass
class PerformancePrediction:
    """Performance prediction result"""
    operation_name: str
    predicted_duration: float
    predicted_memory: float
    confidence: float
    trend: str  # "improving", "stable", "degrading"
    recommendations: List[str] = field(default_factory=list)
    risk_level: str = "low"  # "low", "medium", "high", "critical"

@dataclass
class OptimizationRecommendation:
    """Optimization recommendation"""
    type: str  # "caching", "algorithm", "resource", "configuration"
    description: str
    expected_improvement: float
    implementation_effort: str  # "low", "medium", "high"
    priority: str  # "low", "medium", "high", "critical"

class PerformancePredictor:
    """ML-based performance prediction system"""
    
    def __init__(self, history_window: int = 1000):
        self.history_window = history_window
        self.performance_history: Dict[str, deque] = defaultdict(lambda: deque(maxlen=history_window))
        self.prediction_models: Dict[str, Any] = {}
        self.trend_analyzer = TrendAnalyzer()
        self.anomaly_detector = AnomalyDetector()
        
        # Register with monitoring
        monitoring_framework.record_metric("predictive_optimizer_initialized", 1.0)
    
    @monitor_operation("predict_performance")
    @handle_errors(retry=True)
    def predict_performance(self, operation_name: str, context: Dict[str, Any] = None) -> PerformancePrediction:
        """Predict performance for an operation"""
        try:
            # Get historical data
            history = list(self.performance_history[operation_name])
            
            if len(history) < 10:
                # Not enough data for prediction
                return PerformancePrediction(
                    operation_name=operation_name,
                    predicted_duration=0.0,
                    predicted_memory=0.0,
                    confidence=0.0,
                    trend="stable",
                    recommendations=["Collect more performance data for accurate predictions"]
                )
            
            # Analyze trends
            trend_analysis = self.trend_analyzer.analyze_trends(history)
            
            # Detect anomalies
            anomaly_score = self.anomaly_detector.detect_anomalies(history)
            
            # Make predictions
            predicted_duration = self._predict_duration(history, context)
            predicted_memory = self._predict_memory(history, context)
            
            # Calculate confidence
            confidence = self._calculate_prediction_confidence(history, trend_analysis)
            
            # Determine trend
            trend = self._determine_trend(trend_analysis)
            
            # Generate recommendations
            recommendations = self._generate_recommendations(history, trend_analysis, anomaly_score)
            
            # Determine risk level
            risk_level = self._determine_risk_level(predicted_duration, predicted_memory, anomaly_score)
            
            prediction = PerformancePrediction(
                operation_name=operation_name,
                predicted_duration=predicted_duration,
                predicted_memory=predicted_memory,
                confidence=confidence,
                trend=trend,
                recommendations=recommendations,
                risk_level=risk_level
            )
            
            # Record metrics
            monitoring_framework.record_metric("performance_prediction_success", 1.0)
            monitoring_framework.record_metric("prediction_confidence", confidence)
            
            return prediction
            
        except Exception as e:
            monitoring_framework.record_metric("performance_prediction_error", 1.0)
            error_handler.handle_error(e, {"operation_name": operation_name, "context": context})
            raise
    
    def record_performance(self, metric: PerformanceMetric):
        """Record performance metric for prediction"""
        self.performance_history[metric.operation_name].append(metric)
        
        # Record metrics for monitoring
        monitoring_framework.record_metric(f"{metric.operation_name}_duration", metric.duration)
        monitoring_framework.record_metric(f"{metric.operation_name}_memory", metric.memory_usage)
        monitoring_framework.record_metric(f"{metric.operation_name}_cpu", metric.cpu_usage)
    
    def _predict_duration(self, history: List[PerformanceMetric], context: Dict[str, Any] = None) -> float:
        """Predict operation duration"""
        if not history:
            return 0.0
        
        # Simple linear regression prediction
        recent_history = history[-20:]  # Use last 20 data points
        
        # Calculate trend
        durations = [m.duration for m in recent_history]
        timestamps = [m.timestamp for m in recent_history]
        
        if len(durations) < 2:
            return statistics.mean(durations) if durations else 0.0
        
        # Simple linear trend calculation
        x_mean = statistics.mean(timestamps)
        y_mean = statistics.mean(durations)
        
        numerator = sum((x - x_mean) * (y - y_mean) for x, y in zip(timestamps, durations))
        denominator = sum((x - x_mean) ** 2 for x in timestamps)
        
        if denominator == 0:
            return y_mean
        
        slope = numerator / denominator
        
        # Predict next value
        next_timestamp = timestamps[-1] + (timestamps[-1] - timestamps[-2]) if len(timestamps) > 1 else timestamps[-1] + 1
        predicted_duration = y_mean + slope * (next_timestamp - x_mean)
        
        # Ensure prediction is reasonable
        return max(0.0, predicted_duration)
    
    def _predict_memory(self, history: List[PerformanceMetric], context: Dict[str, Any] = None) -> float:
        """Predict memory usage"""
        if not history:
            return 0.0
        
        recent_history = history[-20:]
        memory_usage = [m.memory_usage for m in recent_history]
        
        # Use exponential moving average for memory prediction
        alpha = 0.3
        predicted_memory = memory_usage[0]
        
        for usage in memory_usage[1:]:
            predicted_memory = alpha * usage + (1 - alpha) * predicted_memory
        
        return predicted_memory
    
    def _calculate_prediction_confidence(self, history: List[PerformanceMetric], trend_analysis: Dict[str, Any]) -> float:
        """Calculate prediction confidence"""
        if len(history) < 10:
            return 0.3
        
        # Calculate variance in recent data
        recent_durations = [m.duration for m in history[-10:]]
        variance = statistics.variance(recent_durations) if len(recent_durations) > 1 else 0
        
        # Base confidence on data consistency
        base_confidence = 0.7
        
        # Adjust based on variance (lower variance = higher confidence)
        if variance > 0:
            confidence_adjustment = min(0.3, 1.0 / (1.0 + variance))
            base_confidence += confidence_adjustment
        
        # Adjust based on trend stability
        if trend_analysis.get("stability_score", 0) > 0.8:
            base_confidence += 0.1
        
        return min(base_confidence, 1.0)
    
    def _determine_trend(self, trend_analysis: Dict[str, Any]) -> str:
        """Determine performance trend"""
        slope = trend_analysis.get("slope", 0)
        stability = trend_analysis.get("stability_score", 0)
        
        if abs(slope) < 0.01 and stability > 0.8:
            return "stable"
        elif slope < -0.01:
            return "improving"
        else:
            return "degrading"
    
    def _generate_recommendations(self, history: List[PerformanceMetric], 
                                trend_analysis: Dict[str, Any], anomaly_score: float) -> List[str]:
        """Generate optimization recommendations"""
        recommendations = []
        
        # Analyze performance patterns
        avg_duration = statistics.mean([m.duration for m in history[-20:]])
        avg_memory = statistics.mean([m.memory_usage for m in history[-20:]])
        
        # Duration-based recommendations
        if avg_duration > 1.0:  # More than 1 second
            recommendations.append("Consider implementing caching for frequently accessed data")
            recommendations.append("Optimize algorithm complexity or use more efficient data structures")
        
        if avg_duration > 5.0:  # More than 5 seconds
            recommendations.append("Consider implementing parallel processing or async operations")
            recommendations.append("Profile the operation to identify bottlenecks")
        
        # Memory-based recommendations
        if avg_memory > 100.0:  # More than 100MB
            recommendations.append("Implement memory pooling for large objects")
            recommendations.append("Consider streaming processing for large datasets")
        
        # Trend-based recommendations
        if trend_analysis.get("trend") == "degrading":
            recommendations.append("Performance is degrading - investigate recent changes")
            recommendations.append("Consider implementing performance regression tests")
        
        # Anomaly-based recommendations
        if anomaly_score > 0.7:
            recommendations.append("Detected performance anomalies - investigate root cause")
            recommendations.append("Implement additional monitoring for this operation")
        
        return recommendations
    
    def _determine_risk_level(self, predicted_duration: float, predicted_memory: float, 
                            anomaly_score: float) -> str:
        """Determine performance risk level"""
        risk_score = 0
        
        # Duration risk
        if predicted_duration > 10.0:
            risk_score += 3
        elif predicted_duration > 5.0:
            risk_score += 2
        elif predicted_duration > 1.0:
            risk_score += 1
        
        # Memory risk
        if predicted_memory > 500.0:
            risk_score += 3
        elif predicted_memory > 200.0:
            risk_score += 2
        elif predicted_memory > 100.0:
            risk_score += 1
        
        # Anomaly risk
        if anomaly_score > 0.8:
            risk_score += 3
        elif anomaly_score > 0.5:
            risk_score += 2
        elif anomaly_score > 0.3:
            risk_score += 1
        
        # Determine risk level
        if risk_score >= 6:
            return "critical"
        elif risk_score >= 4:
            return "high"
        elif risk_score >= 2:
            return "medium"
        else:
            return "low"

class TrendAnalyzer:
    """Analyzer for performance trends"""
    
    def analyze_trends(self, history: List[PerformanceMetric]) -> Dict[str, Any]:
        """Analyze performance trends"""
        if len(history) < 5:
            return {"slope": 0, "stability_score": 0, "trend": "insufficient_data"}
        
        # Calculate trend slope
        durations = [m.duration for m in history]
        timestamps = [m.timestamp for m in history]
        
        slope = self._calculate_slope(timestamps, durations)
        
        # Calculate stability score
        stability_score = self._calculate_stability(durations)
        
        # Determine trend direction
        if abs(slope) < 0.001:
            trend = "stable"
        elif slope > 0:
            trend = "increasing"
        else:
            trend = "decreasing"
        
        return {
            "slope": slope,
            "stability_score": stability_score,
            "stability": stability_score,
            "trend": trend,
            "trend_direction": trend,
            "trend_strength": abs(slope),
            "data_points": len(history)
        }
    
    def _calculate_slope(self, x: List[float], y: List[float]) -> float:
        """Calculate linear regression slope"""
        if len(x) != len(y) or len(x) < 2:
            return 0.0
        
        x_mean = statistics.mean(x)
        y_mean = statistics.mean(y)
        
        numerator = sum((xi - x_mean) * (yi - y_mean) for xi, yi in zip(x, y))
        denominator = sum((xi - x_mean) ** 2 for xi in x)
        
        if denominator == 0:
            return 0.0
        
        return numerator / denominator
    
    def _calculate_stability(self, values: List[float]) -> float:
        """Calculate stability score (0-1, higher = more stable)"""
        if len(values) < 2:
            return 1.0
        
        # Calculate coefficient of variation
        mean_val = statistics.mean(values)
        if mean_val == 0:
            return 1.0
        
        std_dev = statistics.stdev(values) if len(values) > 1 else 0
        cv = std_dev / mean_val
        
        # Convert to stability score (lower CV = higher stability)
        stability = max(0.0, 1.0 - cv)
        return min(stability, 1.0)

class AnomalyDetector:
    """Detector for performance anomalies"""
    
    def __init__(self, threshold_multiplier: float = 2.0):
        self.threshold_multiplier = threshold_multiplier
    
    def detect_anomalies(self, history: List[PerformanceMetric]) -> float:
        """Detect anomalies in performance data"""
        if len(history) < 10:
            return 0.0
        
        durations = [m.duration for m in history]
        
        # Calculate baseline statistics
        mean_duration = statistics.mean(durations)
        std_duration = statistics.stdev(durations) if len(durations) > 1 else 0
        
        if std_duration == 0:
            return 0.0
        
        # Calculate anomaly scores for recent data points
        recent_durations = durations[-5:]  # Check last 5 data points
        anomaly_scores = []
        
        for duration in recent_durations:
            z_score = abs(duration - mean_duration) / std_duration
            anomaly_score = min(1.0, z_score / self.threshold_multiplier)
            anomaly_scores.append(anomaly_score)
        
        # Return average anomaly score
        return statistics.mean(anomaly_scores) if anomaly_scores else 0.0

class PredictiveOptimizer:
    """Main predictive optimization system"""
    
    def __init__(self):
        self.predictor = PerformancePredictor()
        self.optimization_engine = OptimizationEngine()
        self.alert_manager = PerformanceAlertManager()
        
        # Start background monitoring
        self._start_background_monitoring()
    
    def _start_background_monitoring(self):
        """Start background performance monitoring"""
        def monitor_performance():
            while True:
                try:
                    # Get current performance metrics from g7
                    metrics = performance_engine.get_performance_metrics()
                    
                    # Record metrics for prediction
                    for op_name, op_metrics in metrics.get("operations", {}).items():
                        if op_metrics.get("count", 0) > 0:
                            metric = PerformanceMetric(
                                timestamp=time.time(),
                                operation_name=op_name,
                                duration=op_metrics.get("average_ms", 0) / 1000.0,  # Convert to seconds
                                memory_usage=metrics.get("memory", {}).get("current_mb", 0),
                                cpu_usage=0.0,  # Would need system monitoring
                                error_count=0,  # Would need error tracking
                                success_count=op_metrics.get("count", 0)
                            )
                            self.predictor.record_performance(metric)
                    
                    time.sleep(60)  # Monitor every minute
                    
                except Exception as e:
                    logger.error(f"Background monitoring error: {e}")
                    time.sleep(60)
        
        monitor_thread = threading.Thread(target=monitor_performance, daemon=True)
        monitor_thread.start()
    
    @monitor_operation("optimize_performance")
    @handle_errors(retry=True)
    def optimize_performance(self, operation_name: str, context: Dict[str, Any] = None) -> List[OptimizationRecommendation]:
        """Generate optimization recommendations for an operation"""
        try:
            # Get performance prediction
            prediction = self.predictor.predict_performance(operation_name, context)
            
            # Generate optimization recommendations
            recommendations = self.optimization_engine.generate_recommendations(prediction, context)
            
            # Check if alerts should be triggered
            if prediction.risk_level in ["high", "critical"]:
                self.alert_manager.trigger_alert(prediction)
            
            return recommendations
            
        except Exception as e:
            error_handler.handle_error(e, {"operation_name": operation_name, "context": context})
            raise
    
    def get_performance_insights(self, operation_name: str = None) -> Dict[str, Any]:
        """Get comprehensive performance insights"""
        insights = {
            "predictions": {},
            "trends": {},
            "recommendations": [],
            "alerts": [],
            "anomalies": {},
            "alert_history": self.alert_manager.get_alert_history(),
            "performance_summary": {
                "total_operations": len(self.predictor.performance_history),
                "total_predictions": 0,
                "total_alerts": len(self.alert_manager.alert_history)
            }
        }
        
        if operation_name:
            # Get insights for specific operation
            insights["operation_name"] = operation_name
            prediction = self.predictor.predict_performance(operation_name)
            insights["predictions"][operation_name] = prediction.__dict__
            insights["performance_summary"]["total_predictions"] += 1
            
            history = list(self.predictor.performance_history[operation_name])
            insights["data_points"] = len(history)
            if history:
                trend_analysis = self.predictor.trend_analyzer.analyze_trends(history)
                insights["trends"][operation_name] = trend_analysis
                anomaly_score = self.predictor.anomaly_detector.detect_anomalies(history)
                insights["anomalies"][operation_name] = anomaly_score
        else:
            # Get insights for all operations
            for op_name in self.predictor.performance_history.keys():
                try:
                    prediction = self.predictor.predict_performance(op_name)
                    insights["predictions"][op_name] = prediction.__dict__
                    insights["performance_summary"]["total_predictions"] += 1
                    
                    history = list(self.predictor.performance_history[op_name])
                    if history:
                        trend_analysis = self.predictor.trend_analyzer.analyze_trends(history)
                        insights["trends"][op_name] = trend_analysis
                except Exception as e:
                    logger.warning(f"Failed to get insights for {op_name}: {e}")
        
        return insights

class OptimizationEngine:
    """Engine for generating optimization recommendations"""
    
    def generate_recommendations(self, prediction: PerformancePrediction, 
                               context: Dict[str, Any] = None) -> List[OptimizationRecommendation]:
        """Generate optimization recommendations"""
        recommendations = []
        
        # Duration-based recommendations
        if prediction.predicted_duration > 5.0:
            recommendations.append(OptimizationRecommendation(
                type="algorithm",
                description="Consider implementing caching for expensive operations",
                expected_improvement=0.5,
                implementation_effort="medium",
                priority="high"
            ))
        
        if prediction.predicted_duration > 10.0:
            recommendations.append(OptimizationRecommendation(
                type="algorithm",
                description="Implement parallel processing or async operations",
                expected_improvement=0.7,
                implementation_effort="high",
                priority="critical"
            ))
        
        # Memory-based recommendations
        if prediction.predicted_memory > 200.0:
            recommendations.append(OptimizationRecommendation(
                type="resource",
                description="Implement memory pooling and cleanup",
                expected_improvement=0.4,
                implementation_effort="medium",
                priority="high"
            ))
        
        # Trend-based recommendations
        if prediction.trend == "degrading":
            recommendations.append(OptimizationRecommendation(
                type="configuration",
                description="Investigate recent changes that may be causing performance degradation",
                expected_improvement=0.3,
                implementation_effort="low",
                priority="high"
            ))
        
        # Risk-based recommendations
        if prediction.risk_level == "critical":
            recommendations.append(OptimizationRecommendation(
                type="monitoring",
                description="Implement additional performance monitoring and alerting",
                expected_improvement=0.2,
                implementation_effort="low",
                priority="critical"
            ))
        
        return recommendations

class PerformanceAlertManager:
    """Manager for performance alerts"""
    
    def __init__(self):
        self.alert_thresholds = {
            "duration": 5.0,  # seconds
            "memory": 200.0,  # MB
            "anomaly_score": 0.7
        }
        self.alert_history = []
    
    def trigger_alert(self, prediction: PerformancePrediction):
        """Trigger performance alert"""
        alert_message = f"Performance alert for {prediction.operation_name}: "
        alert_message += f"Risk level: {prediction.risk_level}, "
        alert_message += f"Predicted duration: {prediction.predicted_duration:.2f}s, "
        alert_message += f"Predicted memory: {prediction.predicted_memory:.2f}MB"
        
        logger.warning(alert_message)
        
        # Record alert in history
        alert = {
            "operation_name": prediction.operation_name,
            "risk_level": prediction.risk_level,
            "predicted_duration": prediction.predicted_duration,
            "predicted_memory": prediction.predicted_memory,
            "timestamp": time.time()
        }
        self.alert_history.append(alert)
        
        # Record alert metric
        monitoring_framework.record_metric("performance_alert_triggered", 1.0)
        
        # Could integrate with external alerting systems here
    
    def get_alert_history(self) -> List[Dict[str, Any]]:
        """Get alert history"""
        return self.alert_history.copy()

# Global predictive optimizer instance
predictive_optimizer = PredictiveOptimizer()

def predict_operation_performance(operation_name: str, context: Dict[str, Any] = None) -> PerformancePrediction:
    """Convenience function for performance prediction"""
    return predictive_optimizer.predictor.predict_performance(operation_name, context)

def get_optimization_recommendations(operation_name: str, context: Dict[str, Any] = None) -> List[OptimizationRecommendation]:
    """Convenience function for optimization recommendations"""
    return predictive_optimizer.optimize_performance(operation_name, context) 