#!/usr/bin/env python3
"""
Advanced Syntax Intelligence System for TuskLang Python SDK
Goal 8.3 Implementation - Advanced Syntax Intelligence

Features:
- Real-time syntax analysis and validation
- AI-powered code suggestions and auto-completion
- Context-aware code intelligence
- Syntax error detection and correction
- Code pattern recognition and suggestions
- Integration with existing TSK operators
- Intelligent code formatting and optimization
"""

import re
import json
import logging
import time
from typing import Dict, List, Optional, Any, Tuple, Set
from dataclasses import dataclass, field
from enum import Enum
from collections import defaultdict, deque
import difflib

# Import g7 components for integration
import sys
import os
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', 'g7'))
from ..g7.error_handler import handle_errors, error_handler
from ..g7.monitoring_framework import monitor_operation, monitoring_framework
from ..g7.performance_engine import optimize_operation, performance_engine

logger = logging.getLogger(__name__)

class SuggestionType(Enum):
    """Types of syntax suggestions"""
    AUTOCOMPLETE = "autocomplete"
    ERROR_CORRECTION = "error_correction"
    OPTIMIZATION = "optimization"
    PATTERN_SUGGESTION = "pattern_suggestion"
    BEST_PRACTICE = "best_practice"
    SECURITY = "security"

class SyntaxErrorType(Enum):
    """Types of syntax errors"""
    MISSING_OPERATOR = "missing_operator"
    INVALID_SYNTAX = "invalid_syntax"
    MISSING_PARAMETER = "missing_parameter"
    TYPE_MISMATCH = "type_mismatch"
    UNDEFINED_VARIABLE = "undefined_variable"
    DUPLICATE_DEFINITION = "duplicate_definition"

@dataclass
class SyntaxSuggestion:
    """Syntax suggestion result"""
    suggestion_type: SuggestionType
    suggestion: str
    confidence: float
    description: str
    replacement_range: Tuple[int, int] = (0, 0)
    context: Dict[str, Any] = field(default_factory=dict)

@dataclass
class SyntaxError:
    """Syntax error information"""
    error_type: SyntaxErrorType
    message: str
    line_number: int
    column: int
    severity: str  # "error", "warning", "info"
    suggestions: List[str] = field(default_factory=list)
    context: Dict[str, Any] = field(default_factory=dict)

@dataclass
class CodeContext:
    """Code context information"""
    current_line: str
    cursor_position: int
    previous_lines: List[str] = field(default_factory=list)
    following_lines: List[str] = field(default_factory=list)
    variables: Dict[str, Any] = field(default_factory=dict)
    functions: List[str] = field(default_factory=list)
    operators: List[str] = field(default_factory=list)
    imports: List[str] = field(default_factory=list)

class TuskLangSyntaxAnalyzer:
    """TuskLang syntax analyzer"""
    
    def __init__(self):
        self.operator_patterns = self._load_operator_patterns()
        self.syntax_rules = self._load_syntax_rules()
        self.common_patterns = self._load_common_patterns()
        self.error_patterns = self._load_error_patterns()
        
        # Register with monitoring
        monitoring_framework.record_metric("syntax_analyzer_initialized", 1.0)
    
    def _load_operator_patterns(self) -> Dict[str, str]:
        """Load TuskLang operator patterns"""
        return {
            "math": r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)",
            "string": r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)",
            "list": r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)",
            "dict": r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)",
            "conditional": r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)",
            "loop": r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)",
            "validation": r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)",
            "transformation": r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)"
        }
    
    def _load_syntax_rules(self) -> Dict[str, List[str]]:
        """Load syntax validation rules"""
        return {
            "operator_declaration": [
                r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)",
                r"def\s+\w+\s*\([^)]*\)\s*:",
                r"return\s+.*"
            ],
            "function_declaration": [
                r"def\s+\w+\s*\([^)]*\)\s*:",
                r"return\s+.*"
            ],
            "variable_assignment": [
                r"\w+\s*=\s*.*",
                r"\w+\s*:\s*.*"
            ],
            "import_statement": [
                r"import\s+.*",
                r"from\s+.*\s+import\s+.*"
            ]
        }
    
    def _load_common_patterns(self) -> Dict[str, List[str]]:
        """Load common code patterns"""
        return {
            "data_processing": [
                "filter_data", "transform_data", "aggregate_data", "validate_data"
            ],
            "api_integration": [
                "make_request", "parse_response", "handle_errors", "rate_limiting"
            ],
            "file_operations": [
                "read_file", "write_file", "process_file", "backup_file"
            ],
            "database_operations": [
                "query_database", "insert_data", "update_data", "delete_data"
            ],
            "security": [
                "encrypt_data", "decrypt_data", "validate_input", "sanitize_output"
            ],
            "performance": [
                "cache_result", "optimize_query", "batch_process", "parallel_execute"
            ]
        }
    
    def _load_error_patterns(self) -> Dict[str, str]:
        """Load error detection patterns"""
        return {
            "missing_operator": r"@operator\s*\(\s*[\"'][^\"']*[\"']\s*\)\s*$",
            "invalid_syntax": r"def\s+\w+\s*\([^)]*\)\s*$",
            "missing_parameter": r"def\s+\w+\s*\(\s*\)\s*:",
            "type_mismatch": r"return\s+[^#]*$",
            "undefined_variable": r"\b\w+\b(?![^(]*\()",
            "duplicate_definition": r"def\s+(\w+)\s*\([^)]*\)\s*:"
        }

class AISyntaxIntelligence:
    """AI-powered syntax intelligence system"""
    
    def __init__(self):
        self.syntax_analyzer = TuskLangSyntaxAnalyzer()
        self.suggestion_engine = SuggestionEngine()
        self.error_detector = ErrorDetector()
        self.context_analyzer = ContextAnalyzer()
        self.pattern_recognizer = PatternRecognizer()
        
        # Learning components
        self.usage_patterns = defaultdict(int)
        self.suggestion_history = deque(maxlen=1000)
        self.error_history = deque(maxlen=1000)
        
        # Register with monitoring
        monitoring_framework.record_metric("syntax_intelligence_initialized", 1.0)
    
    @monitor_operation("analyze_syntax")
    @handle_errors(retry=True)
    def analyze_syntax(self, code: str, context: CodeContext = None) -> List[SyntaxSuggestion]:
        """Analyze code syntax and provide suggestions"""
        try:
            suggestions = []
            
            # Analyze context
            if context:
                context_info = self.context_analyzer.analyze_context(context)
            else:
                context_info = self._extract_context_from_code(code)
            
            # Detect syntax errors
            errors = self.error_detector.detect_errors(code, context_info)
            
            # Generate error correction suggestions
            for error in errors:
                error_suggestions = self.suggestion_engine.generate_error_suggestions(error, context_info)
                suggestions.extend(error_suggestions)
            
            # Generate auto-completion suggestions
            if context and context.cursor_position > 0:
                autocomplete_suggestions = self.suggestion_engine.generate_autocomplete_suggestions(context, context_info)
                suggestions.extend(autocomplete_suggestions)
            
            # Generate optimization suggestions
            optimization_suggestions = self.suggestion_engine.generate_optimization_suggestions(code, context_info)
            suggestions.extend(optimization_suggestions)
            
            # Generate pattern suggestions
            pattern_suggestions = self.suggestion_engine.generate_pattern_suggestions(code, context_info)
            suggestions.extend(pattern_suggestions)
            
            # Sort suggestions by confidence
            suggestions.sort(key=lambda x: x.confidence, reverse=True)
            
            # Record metrics
            monitoring_framework.record_metric("syntax_analysis_success", 1.0)
            monitoring_framework.record_metric("suggestions_generated", len(suggestions))
            
            return suggestions
            
        except Exception as e:
            monitoring_framework.record_metric("syntax_analysis_error", 1.0)
            error_handler.handle_error(e, {"code_length": len(code), "context": context})
            raise
    
    @monitor_operation("validate_syntax")
    @handle_errors(retry=True)
    def validate_syntax(self, code: str) -> List[SyntaxError]:
        """Validate code syntax and return errors"""
        try:
            errors = self.error_detector.detect_errors(code)
            
            # Record metrics
            monitoring_framework.record_metric("syntax_validation_success", 1.0)
            monitoring_framework.record_metric("errors_found", len(errors))
            
            return errors
            
        except Exception as e:
            monitoring_framework.record_metric("syntax_validation_error", 1.0)
            error_handler.handle_error(e, {"code_length": len(code)})
            raise
    
    @monitor_operation("get_autocomplete")
    @handle_errors(retry=True)
    def get_autocomplete(self, context: CodeContext) -> List[SyntaxSuggestion]:
        """Get auto-completion suggestions for current context"""
        try:
            context_info = self.context_analyzer.analyze_context(context)
            suggestions = self.suggestion_engine.generate_autocomplete_suggestions(context, context_info)
            
            # Record usage patterns
            self._record_usage_pattern(context, suggestions)
            
            # Record metrics
            monitoring_framework.record_metric("autocomplete_success", 1.0)
            monitoring_framework.record_metric("autocomplete_suggestions", len(suggestions))
            
            return suggestions
            
        except Exception as e:
            monitoring_framework.record_metric("autocomplete_error", 1.0)
            error_handler.handle_error(e, {"context": context})
            raise
    
    def _extract_context_from_code(self, code: str) -> Dict[str, Any]:
        """Extract context information from code"""
        lines = code.split('\n')
        context = {
            'variables': set(),
            'functions': set(),
            'operators': set(),
            'imports': set(),
            'patterns': []
        }
        
        for line in lines:
            # Extract variables
            var_matches = re.findall(r'\b(\w+)\s*=', line)
            context['variables'].update(var_matches)
            
            # Extract functions
            func_matches = re.findall(r'def\s+(\w+)', line)
            context['functions'].update(func_matches)
            
            # Extract operators
            op_matches = re.findall(r'@operator\s*\(\s*[\"\']([^\"\']*)[\"\']', line)
            context['operators'].update(op_matches)
            
            # Extract imports
            import_matches = re.findall(r'(?:import|from)\s+(\w+)', line)
            context['imports'].update(import_matches)
        
        return context
    
    def _record_usage_pattern(self, context: CodeContext, suggestions: List[SyntaxSuggestion]):
        """Record usage patterns for learning"""
        for suggestion in suggestions:
            pattern_key = f"{context.current_line[:20]}_{suggestion.suggestion_type.value}"
            self.usage_patterns[pattern_key] += 1
            self.suggestion_history.append({
                'context': context.current_line,
                'suggestion': suggestion.suggestion,
                'type': suggestion.suggestion_type.value,
                'timestamp': time.time()
            })

class SuggestionEngine:
    """Engine for generating syntax suggestions"""
    
    def __init__(self):
        self.tusk_operators = self._load_tusk_operators()
        self.common_suggestions = self._load_common_suggestions()
        self.pattern_suggestions = self._load_pattern_suggestions()
    
    def _load_tusk_operators(self) -> List[str]:
        """Load available TuskLang operators"""
        return [
            "add", "subtract", "multiply", "divide", "modulo",
            "concat", "split", "join", "replace", "trim",
            "append", "prepend", "remove", "filter", "map",
            "reduce", "sort", "reverse", "unique", "count",
            "sum", "average", "min", "max", "range",
            "if", "else", "while", "for", "foreach",
            "try", "catch", "finally", "throw", "return"
        ]
    
    def _load_common_suggestions(self) -> Dict[str, List[str]]:
        """Load common code suggestions"""
        return {
            "imports": [
                "import logging",
                "from typing import Dict, List, Optional, Any",
                "from dataclasses import dataclass",
                "import json",
                "import time"
            ],
            "decorators": [
                "@monitor_operation",
                "@handle_errors",
                "@optimize_operation",
                "@dataclass",
                "@property"
            ],
            "patterns": [
                "try:\n    pass\nexcept Exception as e:\n    logger.error(f\"Error: {e}\")\n    raise",
                "if __name__ == \"__main__\":\n    main()",
                "def main():\n    pass"
            ]
        }
    
    def _load_pattern_suggestions(self) -> Dict[str, str]:
        """Load pattern-based suggestions"""
        return {
            "data_processing": "Consider using list comprehension or generator expressions for better performance",
            "error_handling": "Add proper error handling with specific exception types",
            "logging": "Add logging statements for better debugging and monitoring",
            "type_hints": "Add type hints for better code documentation and IDE support",
            "documentation": "Add docstring for function documentation",
            "optimization": "Consider caching results for expensive operations"
        }
    
    def generate_autocomplete_suggestions(self, context: CodeContext, context_info: Dict[str, Any]) -> List[SyntaxSuggestion]:
        """Generate auto-completion suggestions"""
        suggestions = []
        current_word = self._get_current_word(context)
        
        # Operator suggestions
        if "@operator" in context.current_line or "def" in context.current_line:
            for operator in self.tusk_operators:
                if operator.startswith(current_word):
                    suggestions.append(SyntaxSuggestion(
                        suggestion_type=SuggestionType.AUTOCOMPLETE,
                        suggestion=operator,
                        confidence=0.9,
                        description=f"Complete operator name: {operator}",
                        replacement_range=self._get_replacement_range(context, current_word)
                    ))
        
        # Variable suggestions
        for var in context_info.get('variables', []):
            if var.startswith(current_word):
                suggestions.append(SyntaxSuggestion(
                    suggestion_type=SuggestionType.AUTOCOMPLETE,
                    suggestion=var,
                    confidence=0.8,
                    description=f"Complete variable name: {var}",
                    replacement_range=self._get_replacement_range(context, current_word)
                ))
        
        # Function suggestions
        for func in context_info.get('functions', []):
            if func.startswith(current_word):
                suggestions.append(SyntaxSuggestion(
                    suggestion_type=SuggestionType.AUTOCOMPLETE,
                    suggestion=func,
                    confidence=0.8,
                    description=f"Complete function name: {func}",
                    replacement_range=self._get_replacement_range(context, current_word)
                ))
        
        # Import suggestions
        if "import" in context.current_line or "from" in context.current_line:
            for imp in self.common_suggestions["imports"]:
                if imp.lower().startswith(current_word.lower()):
                    suggestions.append(SyntaxSuggestion(
                        suggestion_type=SuggestionType.AUTOCOMPLETE,
                        suggestion=imp,
                        confidence=0.7,
                        description=f"Import statement: {imp}",
                        replacement_range=self._get_replacement_range(context, current_word)
                    ))
        
        return suggestions
    
    def generate_error_suggestions(self, error: SyntaxError, context_info: Dict[str, Any]) -> List[SyntaxSuggestion]:
        """Generate suggestions for syntax errors"""
        suggestions = []
        
        if error.error_type == SyntaxErrorType.MISSING_OPERATOR:
            suggestions.append(SyntaxSuggestion(
                suggestion_type=SuggestionType.ERROR_CORRECTION,
                suggestion="@operator(\"operator_name\")",
                confidence=0.9,
                description="Add operator decorator",
                context={"error": error}
            ))
        
        elif error.error_type == SyntaxErrorType.INVALID_SYNTAX:
            suggestions.append(SyntaxSuggestion(
                suggestion_type=SuggestionType.ERROR_CORRECTION,
                suggestion=":",
                confidence=0.8,
                description="Add missing colon",
                context={"error": error}
            ))
        
        elif error.error_type == SyntaxErrorType.MISSING_PARAMETER:
            suggestions.append(SyntaxSuggestion(
                suggestion_type=SuggestionType.ERROR_CORRECTION,
                suggestion="context: Dict[str, Any]",
                confidence=0.8,
                description="Add context parameter",
                context={"error": error}
            ))
        
        return suggestions
    
    def generate_optimization_suggestions(self, code: str, context_info: Dict[str, Any]) -> List[SyntaxSuggestion]:
        """Generate optimization suggestions"""
        suggestions = []
        
        # Check for performance patterns
        if "for" in code and "in" in code and "range" not in code:
            suggestions.append(SyntaxSuggestion(
                suggestion_type=SuggestionType.OPTIMIZATION,
                suggestion="Consider using list comprehension or generator expressions",
                confidence=0.7,
                description="Performance optimization suggestion",
                context={"pattern": "loop_optimization"}
            ))
        
        # Check for error handling
        if "try:" not in code and "except" not in code:
            suggestions.append(SyntaxSuggestion(
                suggestion_type=SuggestionType.BEST_PRACTICE,
                suggestion="Add proper error handling with try-except blocks",
                confidence=0.6,
                description="Error handling best practice",
                context={"pattern": "error_handling"}
            ))
        
        return suggestions
    
    def generate_pattern_suggestions(self, code: str, context_info: Dict[str, Any]) -> List[SyntaxSuggestion]:
        """Generate pattern-based suggestions"""
        suggestions = []
        
        for pattern, suggestion in self.pattern_suggestions.items():
            if pattern in code.lower():
                suggestions.append(SyntaxSuggestion(
                    suggestion_type=SuggestionType.PATTERN_SUGGESTION,
                    suggestion=suggestion,
                    confidence=0.6,
                    description=f"Pattern suggestion for {pattern}",
                    context={"pattern": pattern}
                ))
        
        return suggestions
    
    def _get_current_word(self, context: CodeContext) -> str:
        """Get the current word being typed"""
        if context.cursor_position == 0:
            return ""
        
        # Find the start of the current word
        start = context.cursor_position
        while start > 0 and context.current_line[start - 1].isalnum():
            start -= 1
        
        return context.current_line[start:context.cursor_position]
    
    def _get_replacement_range(self, context: CodeContext, word: str) -> Tuple[int, int]:
        """Get the range for word replacement"""
        if not word:
            return (context.cursor_position, context.cursor_position)
        
        start = context.cursor_position - len(word)
        end = context.cursor_position
        return (start, end)

class ErrorDetector:
    """Detector for syntax errors"""
    
    def __init__(self):
        self.error_patterns = self._load_error_patterns()
    
    def _load_error_patterns(self) -> Dict[str, str]:
        """Load error detection patterns"""
        return {
            "missing_operator": r"def\s+\w+\s*\([^)]*\)\s*:(?!\s*@operator)",
            "invalid_syntax": r"def\s+\w+\s*\([^)]*\)\s*$",
            "missing_parameter": r"def\s+\w+\s*\(\s*\)\s*:",
            "type_mismatch": r"return\s+[^#]*$",
            "undefined_variable": r"\b\w+\b(?![^(]*\()",
            "duplicate_definition": r"def\s+(\w+)\s*\([^)]*\)\s*:"
        }
    
    def detect_errors(self, code: str, context_info: Dict[str, Any] = None) -> List[SyntaxError]:
        """Detect syntax errors in code"""
        errors = []
        lines = code.split('\n')
        
        for line_num, line in enumerate(lines, 1):
            # Check for missing operator decorator
            if re.search(self.error_patterns["missing_operator"], line):
                errors.append(SyntaxError(
                    error_type=SyntaxErrorType.MISSING_OPERATOR,
                    message="Function should have @operator decorator",
                    line_number=line_num,
                    column=0,
                    severity="error"
                ))
            
            # Check for incomplete function definition
            if re.search(self.error_patterns["invalid_syntax"], line):
                errors.append(SyntaxError(
                    error_type=SyntaxErrorType.INVALID_SYNTAX,
                    message="Incomplete function definition",
                    line_number=line_num,
                    column=0,
                    severity="error"
                ))
            
            # Check for missing parameters
            if re.search(self.error_patterns["missing_parameter"], line):
                errors.append(SyntaxError(
                    error_type=SyntaxErrorType.MISSING_PARAMETER,
                    message="Function should have context parameter",
                    line_number=line_num,
                    column=0,
                    severity="warning"
                ))
        
        return errors

class ContextAnalyzer:
    """Analyzer for code context"""
    
    def analyze_context(self, context: CodeContext) -> Dict[str, Any]:
        """Analyze code context"""
        return {
            'variables': list(context.variables.keys()),
            'functions': list(context.functions),
            'operators': list(context.operators),
            'imports': list(context.imports),
            'current_line_type': self._analyze_line_type(context.current_line),
            'suggestions': self._generate_context_suggestions(context)
        }
    
    def _analyze_line_type(self, line: str) -> str:
        """Analyze the type of current line"""
        if line.strip().startswith('@'):
            return 'decorator'
        elif line.strip().startswith('def '):
            return 'function_definition'
        elif line.strip().startswith('import ') or line.strip().startswith('from '):
            return 'import'
        elif '=' in line and not line.strip().startswith('#'):
            return 'assignment'
        elif line.strip().startswith('return '):
            return 'return'
        elif line.strip().startswith('if ') or line.strip().startswith('elif ') or line.strip().startswith('else:'):
            return 'conditional'
        elif line.strip().startswith('for ') or line.strip().startswith('while '):
            return 'loop'
        else:
            return 'statement'
    
    def _generate_context_suggestions(self, context: CodeContext) -> List[str]:
        """Generate context-based suggestions"""
        suggestions = []
        
        if not context.imports:
            suggestions.append("Consider adding common imports like logging and typing")
        
        if not context.functions:
            suggestions.append("Consider adding a main function")
        
        return suggestions

class PatternRecognizer:
    """Recognizer for code patterns"""
    
    def __init__(self):
        self.patterns = self._load_patterns()
    
    def _load_patterns(self) -> Dict[str, str]:
        """Load code patterns"""
        return {
            "data_processing": r"(filter|map|reduce|sort|group)",
            "error_handling": r"(try|except|finally|raise)",
            "logging": r"(log|debug|info|warning|error)",
            "validation": r"(validate|check|verify|assert)",
            "caching": r"(cache|memoize|store)",
            "async": r"(async|await|asyncio)"
        }
    
    def recognize_patterns(self, code: str) -> List[str]:
        """Recognize patterns in code"""
        patterns = []
        
        for pattern_name, pattern_regex in self.patterns.items():
            if re.search(pattern_regex, code, re.IGNORECASE):
                patterns.append(pattern_name)
        
        return patterns

# Convenience functions
def analyze_code_syntax(code: str, context: CodeContext = None) -> List[SyntaxSuggestion]:
    """Analyze code syntax and provide suggestions"""
    intelligence = AISyntaxIntelligence()
    return intelligence.analyze_syntax(code, context)

def validate_code_syntax(code: str) -> List[SyntaxError]:
    """Validate code syntax and return errors"""
    intelligence = AISyntaxIntelligence()
    return intelligence.validate_syntax(code)

def get_autocomplete_suggestions(context: CodeContext) -> List[SyntaxSuggestion]:
    """Get auto-completion suggestions for current context"""
    intelligence = AISyntaxIntelligence()
    return intelligence.get_autocomplete(context) 