"""
Elasticsearch Operator - Full-Text Search Engine Integration
Production-ready Elasticsearch integration with indexing, searching, aggregations, and cluster management.
"""

import asyncio
import json
import logging
import ssl
import time
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional, Union
from urllib.parse import urlparse

# Elasticsearch Support
try:
    from elasticsearch import Elasticsearch, AsyncElasticsearch
    from elasticsearch.exceptions import (
        ConnectionError, RequestError, NotFoundError, 
        ConflictError, TransportError, AuthenticationException
    )
    from elasticsearch.helpers import bulk, streaming_bulk, parallel_bulk
    ELASTICSEARCH_AVAILABLE = True
except ImportError:
    ELASTICSEARCH_AVAILABLE = False
    print("elasticsearch library not available. @elasticsearch operator will be limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class ElasticsearchConfig:
    """Elasticsearch connection configuration."""
    hosts: List[str] = field(default_factory=lambda: ["localhost:9200"])
    username: Optional[str] = None
    password: Optional[str] = None
    api_key: Optional[str] = None
    cloud_id: Optional[str] = None
    timeout: int = 30
    max_retries: int = 3
    retry_on_timeout: bool = True
    ca_certs: Optional[str] = None
    client_cert: Optional[str] = None
    client_key: Optional[str] = None
    verify_certs: bool = True
    use_ssl: bool = False
    sniff_on_start: bool = False
    sniff_on_connection_fail: bool = True
    sniff_timeout: float = 0.1
    sniffer_timeout: float = 10

@dataclass
class ElasticsearchDocument:
    """Elasticsearch document structure."""
    index: str
    id: Optional[str] = None
    source: Dict[str, Any] = field(default_factory=dict)
    routing: Optional[str] = None
    version: Optional[int] = None
    version_type: Optional[str] = None

@dataclass
class ElasticsearchQuery:
    """Elasticsearch query structure."""
    query: Dict[str, Any] = field(default_factory=dict)
    filter: Optional[Dict[str, Any]] = None
    sort: Optional[List[Dict[str, Any]]] = None
    size: int = 10
    from_: int = 0
    source: Optional[Union[bool, List[str]]] = None
    highlight: Optional[Dict[str, Any]] = None
    aggregations: Optional[Dict[str, Any]] = None

@dataclass
class ElasticsearchIndexSettings:
    """Elasticsearch index settings."""
    number_of_shards: int = 1
    number_of_replicas: int = 1
    refresh_interval: str = "1s"
    max_result_window: int = 10000
    analysis: Optional[Dict[str, Any]] = None
    mappings: Optional[Dict[str, Any]] = None

@dataclass
class ElasticsearchBulkOperation:
    """Elasticsearch bulk operation."""
    operation: str  # index, create, update, delete
    document: ElasticsearchDocument
    doc: Optional[Dict[str, Any]] = None  # for update operations
    doc_as_upsert: bool = False

class ElasticsearchConnectionManager:
    """Manages Elasticsearch connections with health monitoring."""
    
    def __init__(self, config: ElasticsearchConfig):
        self.config = config
        self.client: Optional[Elasticsearch] = None
        self.async_client: Optional[AsyncElasticsearch] = None
        self.cluster_health = {}
        self.node_info = {}
        self._health_check_interval = 60
    
    async def initialize_connection(self) -> bool:
        """Initialize Elasticsearch connection."""
        if not ELASTICSEARCH_AVAILABLE:
            logger.error("Elasticsearch client library not available")
            return False
        
        try:
            # Build client configuration
            client_config = {
                'hosts': self.config.hosts,
                'timeout': self.config.timeout,
                'max_retries': self.config.max_retries,
                'retry_on_timeout': self.config.retry_on_timeout,
                'sniff_on_start': self.config.sniff_on_start,
                'sniff_on_connection_fail': self.config.sniff_on_connection_fail,
                'sniff_timeout': self.config.sniff_timeout,
                'sniffer_timeout': self.config.sniffer_timeout
            }
            
            # SSL Configuration
            if self.config.use_ssl:
                client_config.update({
                    'use_ssl': True,
                    'verify_certs': self.config.verify_certs,
                    'ca_certs': self.config.ca_certs,
                    'client_cert': self.config.client_cert,
                    'client_key': self.config.client_key
                })
            
            # Authentication
            if self.config.username and self.config.password:
                client_config['http_auth'] = (self.config.username, self.config.password)
            elif self.config.api_key:
                client_config['api_key'] = self.config.api_key
            elif self.config.cloud_id:
                client_config['cloud_id'] = self.config.cloud_id
            
            # Create clients
            self.client = Elasticsearch(**client_config)
            self.async_client = AsyncElasticsearch(**client_config)
            
            # Test connection
            info = self.client.info()
            logger.info(f"Connected to Elasticsearch cluster: {info['cluster_name']}")
            
            # Start health monitoring
            asyncio.create_task(self._health_monitor_task())
            
            return True
            
        except Exception as e:
            logger.error(f"Error connecting to Elasticsearch: {str(e)}")
            return False
    
    async def _health_monitor_task(self):
        """Background task to monitor cluster health."""
        while True:
            try:
                if self.client:
                    health = self.client.cluster.health()
                    self.cluster_health = {
                        'cluster_name': health['cluster_name'],
                        'status': health['status'],
                        'timed_out': health['timed_out'],
                        'number_of_nodes': health['number_of_nodes'],
                        'number_of_data_nodes': health['number_of_data_nodes'],
                        'active_primary_shards': health['active_primary_shards'],
                        'active_shards': health['active_shards'],
                        'relocating_shards': health['relocating_shards'],
                        'initializing_shards': health['initializing_shards'],
                        'unassigned_shards': health['unassigned_shards']
                    }
                    
                    if health['status'] in ['yellow', 'red']:
                        logger.warning(f"Elasticsearch cluster health: {health['status']}")
                    
                await asyncio.sleep(self._health_check_interval)
                
            except Exception as e:
                logger.error(f"Health monitor error: {str(e)}")
                await asyncio.sleep(self._health_check_interval * 2)

class ElasticsearchOperator:
    """@elasticsearch operator implementation with full production features."""
    
    def __init__(self):
        self.connection_manager: Optional[ElasticsearchConnectionManager] = None
        self.operation_stats = {
            'index_operations': 0,
            'search_operations': 0,
            'delete_operations': 0,
            'update_operations': 0,
            'bulk_operations': 0,
            'aggregation_operations': 0
        }
        self._executor = ThreadPoolExecutor(max_workers=10)
    
    async def connect(self, config: Optional[ElasticsearchConfig] = None) -> bool:
        """Connect to Elasticsearch cluster."""
        if config is None:
            config = ElasticsearchConfig()
        
        self.connection_manager = ElasticsearchConnectionManager(config)
        return await self.connection_manager.initialize_connection()
    
    @property
    def client(self) -> Optional[Any]:
        """Get synchronous client."""
        return self.connection_manager.client if self.connection_manager else None
    
    @property
    def async_client(self) -> Optional[Any]:
        """Get asynchronous client."""
        return self.connection_manager.async_client if self.connection_manager else None
    
    async def index_document(self, document: ElasticsearchDocument) -> Dict[str, Any]:
        """Index a document."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            params = {
                'index': document.index,
                'body': document.source
            }
            
            if document.id:
                params['id'] = document.id
            if document.routing:
                params['routing'] = document.routing
            if document.version:
                params['version'] = document.version
                params['version_type'] = document.version_type or 'internal'
            
            result = await self.async_client.index(**params)
            self.operation_stats['index_operations'] += 1
            
            return {
                'index': result['_index'],
                'id': result['_id'],
                'version': result['_version'],
                'result': result['result'],
                'shards': result['_shards']
            }
            
        except Exception as e:
            logger.error(f"Error indexing document: {str(e)}")
            raise
    
    async def get_document(self, index: str, doc_id: str, source: Optional[List[str]] = None) -> Optional[Dict[str, Any]]:
        """Get document by ID."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            params = {'index': index, 'id': doc_id}
            if source:
                params['_source'] = source
            
            result = await self.async_client.get(**params)
            
            return {
                'index': result['_index'],
                'id': result['_id'],
                'version': result['_version'],
                'found': result['found'],
                'source': result.get('_source', {})
            }
            
        except NotFoundError:
            return None
        except Exception as e:
            logger.error(f"Error getting document {doc_id}: {str(e)}")
            raise
    
    async def update_document(self, index: str, doc_id: str, doc: Dict[str, Any], upsert: bool = False) -> Dict[str, Any]:
        """Update document."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            params = {
                'index': index,
                'id': doc_id,
                'body': {'doc': doc}
            }
            
            if upsert:
                params['body']['doc_as_upsert'] = True
            
            result = await self.async_client.update(**params)
            self.operation_stats['update_operations'] += 1
            
            return {
                'index': result['_index'],
                'id': result['_id'],
                'version': result['_version'],
                'result': result['result'],
                'shards': result['_shards']
            }
            
        except Exception as e:
            logger.error(f"Error updating document {doc_id}: {str(e)}")
            raise
    
    async def delete_document(self, index: str, doc_id: str) -> Dict[str, Any]:
        """Delete document by ID."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            result = await self.async_client.delete(index=index, id=doc_id)
            self.operation_stats['delete_operations'] += 1
            
            return {
                'index': result['_index'],
                'id': result['_id'],
                'version': result['_version'],
                'result': result['result'],
                'shards': result['_shards']
            }
            
        except Exception as e:
            logger.error(f"Error deleting document {doc_id}: {str(e)}")
            raise
    
    async def search(self, query: ElasticsearchQuery, indices: Optional[List[str]] = None) -> Dict[str, Any]:
        """Execute search query."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            # Build search body
            search_body = {}
            
            if query.query:
                search_body['query'] = query.query
            
            if query.filter:
                if 'query' not in search_body:
                    search_body['query'] = {'bool': {'filter': query.filter}}
                else:
                    if 'bool' not in search_body['query']:
                        search_body['query'] = {'bool': {'must': search_body['query']}}
                    search_body['query']['bool']['filter'] = query.filter
            
            if query.sort:
                search_body['sort'] = query.sort
            
            if query.source is not None:
                search_body['_source'] = query.source
            
            if query.highlight:
                search_body['highlight'] = query.highlight
            
            if query.aggregations:
                search_body['aggs'] = query.aggregations
            
            params = {
                'index': indices or ['_all'],
                'body': search_body,
                'size': query.size,
                'from_': query.from_
            }
            
            result = await self.async_client.search(**params)
            self.operation_stats['search_operations'] += 1
            
            if query.aggregations:
                self.operation_stats['aggregation_operations'] += 1
            
            return {
                'took': result['took'],
                'timed_out': result['timed_out'],
                'total_hits': result['hits']['total']['value'],
                'max_score': result['hits']['max_score'],
                'hits': [
                    {
                        'index': hit['_index'],
                        'id': hit['_id'],
                        'score': hit['_score'],
                        'source': hit['_source'],
                        'highlight': hit.get('highlight', {})
                    }
                    for hit in result['hits']['hits']
                ],
                'aggregations': result.get('aggregations', {})
            }
            
        except Exception as e:
            logger.error(f"Error executing search: {str(e)}")
            raise
    
    async def multi_search(self, queries: List[ElasticsearchQuery], indices: Optional[List[str]] = None) -> List[Dict[str, Any]]:
        """Execute multiple searches."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            body = []
            for query in queries:
                # Search header
                header = {'index': indices or ['_all']}
                body.append(header)
                
                # Search body
                search_body = {}
                if query.query:
                    search_body['query'] = query.query
                if query.sort:
                    search_body['sort'] = query.sort
                if query.source is not None:
                    search_body['_source'] = query.source
                if query.aggregations:
                    search_body['aggs'] = query.aggregations
                
                search_body.update({
                    'size': query.size,
                    'from': query.from_
                })
                
                body.append(search_body)
            
            result = await self.async_client.msearch(body=body)
            self.operation_stats['search_operations'] += len(queries)
            
            responses = []
            for response in result['responses']:
                if 'error' in response:
                    responses.append({'error': response['error']})
                else:
                    responses.append({
                        'took': response['took'],
                        'total_hits': response['hits']['total']['value'],
                        'hits': [
                            {
                                'index': hit['_index'],
                                'id': hit['_id'],
                                'score': hit['_score'],
                                'source': hit['_source']
                            }
                            for hit in response['hits']['hits']
                        ]
                    })
            
            return responses
            
        except Exception as e:
            logger.error(f"Error executing multi-search: {str(e)}")
            raise
    
    async def bulk_operations(self, operations: List[ElasticsearchBulkOperation]) -> Dict[str, Any]:
        """Execute bulk operations."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            actions = []
            for operation in operations:
                action = {
                    '_op_type': operation.operation,
                    '_index': operation.document.index,
                    '_source': operation.document.source
                }
                
                if operation.document.id:
                    action['_id'] = operation.document.id
                if operation.document.routing:
                    action['_routing'] = operation.document.routing
                
                # Handle update operations
                if operation.operation == 'update':
                    if operation.doc:
                        action['_source'] = {'doc': operation.doc}
                        if operation.doc_as_upsert:
                            action['_source']['doc_as_upsert'] = True
                
                actions.append(action)
            
            result = await self.async_client.bulk(body=actions)
            self.operation_stats['bulk_operations'] += 1
            
            # Process results
            processed = {
                'took': result['took'],
                'errors': result['errors'],
                'items': []
            }
            
            for item in result['items']:
                for op_type, details in item.items():
                    processed['items'].append({
                        'operation': op_type,
                        'index': details['_index'],
                        'id': details['_id'],
                        'version': details.get('_version'),
                        'result': details.get('result'),
                        'status': details['status'],
                        'error': details.get('error')
                    })
            
            return processed
            
        except Exception as e:
            logger.error(f"Error executing bulk operations: {str(e)}")
            raise
    
    async def create_index(self, index_name: str, settings: ElasticsearchIndexSettings) -> bool:
        """Create index with settings."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            index_body = {
                'settings': {
                    'number_of_shards': settings.number_of_shards,
                    'number_of_replicas': settings.number_of_replicas,
                    'refresh_interval': settings.refresh_interval,
                    'max_result_window': settings.max_result_window
                }
            }
            
            if settings.analysis:
                index_body['settings']['analysis'] = settings.analysis
            
            if settings.mappings:
                index_body['mappings'] = settings.mappings
            
            await self.async_client.indices.create(index=index_name, body=index_body)
            logger.info(f"Created index: {index_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error creating index {index_name}: {str(e)}")
            raise
    
    async def delete_index(self, index_name: str) -> bool:
        """Delete index."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            await self.async_client.indices.delete(index=index_name)
            logger.info(f"Deleted index: {index_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error deleting index {index_name}: {str(e)}")
            raise
    
    async def index_exists(self, index_name: str) -> bool:
        """Check if index exists."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            return await self.async_client.indices.exists(index=index_name)
        except Exception as e:
            logger.error(f"Error checking index existence {index_name}: {str(e)}")
            return False
    
    async def put_mapping(self, index_name: str, mapping: Dict[str, Any]) -> bool:
        """Update index mapping."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            await self.async_client.indices.put_mapping(
                index=index_name,
                body=mapping
            )
            logger.info(f"Updated mapping for index: {index_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error updating mapping for {index_name}: {str(e)}")
            raise
    
    async def refresh_index(self, index_name: str) -> bool:
        """Refresh index."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            await self.async_client.indices.refresh(index=index_name)
            return True
        except Exception as e:
            logger.error(f"Error refreshing index {index_name}: {str(e)}")
            return False
    
    async def get_cluster_health(self) -> Dict[str, Any]:
        """Get cluster health."""
        if not self.connection_manager:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        return self.connection_manager.cluster_health
    
    async def get_index_stats(self, index_name: str) -> Dict[str, Any]:
        """Get index statistics."""
        if not self.async_client:
            raise RuntimeError("Not connected to Elasticsearch cluster")
        
        try:
            stats = await self.async_client.indices.stats(index=index_name)
            index_stats = stats['indices'][index_name]
            
            return {
                'total_docs': index_stats['total']['docs']['count'],
                'total_size': index_stats['total']['store']['size_in_bytes'],
                'primary_docs': index_stats['primaries']['docs']['count'],
                'primary_size': index_stats['primaries']['store']['size_in_bytes'],
                'indexing_total': index_stats['total']['indexing']['index_total'],
                'search_total': index_stats['total']['search']['query_total'],
                'segments_count': index_stats['total']['segments']['count'],
                'segments_memory': index_stats['total']['segments']['memory_in_bytes']
            }
            
        except Exception as e:
            logger.error(f"Error getting index stats {index_name}: {str(e)}")
            raise
    
    def get_statistics(self) -> Dict[str, Any]:
        """Get operation statistics."""
        stats = {
            'operations': self.operation_stats.copy(),
            'connected': self.connection_manager is not None
        }
        
        if self.connection_manager and self.connection_manager.cluster_health:
            stats['cluster_health'] = self.connection_manager.cluster_health
        
        return stats
    
    async def close(self):
        """Close connections and cleanup."""
        if self.async_client:
            await self.async_client.close()
        
        if self.client:
            self.client.close()
        
        self._executor.shutdown(wait=True)
        logger.info("Elasticsearch operator closed")

# Export the operator
__all__ = [
    'ElasticsearchOperator', 'ElasticsearchConfig', 'ElasticsearchDocument', 
    'ElasticsearchQuery', 'ElasticsearchIndexSettings', 'ElasticsearchBulkOperation'
] 