"""
Grafana Operator - Visualization and Dashboards Integration
Production-ready Grafana integration with dashboard management, data sources, and alerting.
"""

import asyncio
import base64
import json
import logging
import time
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional, Union
from urllib.parse import urljoin

# HTTP Support
try:
    import aiohttp
    import requests
    HTTP_AVAILABLE = True
except ImportError:
    HTTP_AVAILABLE = False
    print("HTTP libraries not available. @grafana operator will be limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class GrafanaConfig:
    """Grafana configuration."""
    base_url: str = "http://localhost:3000"
    username: str = "admin"
    password: str = "admin"
    api_key: Optional[str] = None
    verify_ssl: bool = True
    timeout: int = 30
    org_id: int = 1

@dataclass
class GrafanaDashboard:
    """Grafana dashboard structure."""
    title: str
    tags: List[str] = field(default_factory=list)
    timezone: str = "browser"
    panels: List[Dict[str, Any]] = field(default_factory=list)
    time_range: Dict[str, str] = field(default_factory=lambda: {"from": "now-6h", "to": "now"})
    refresh: str = "5s"
    annotations: List[Dict[str, Any]] = field(default_factory=list)
    templating: Dict[str, Any] = field(default_factory=dict)
    links: List[Dict[str, Any]] = field(default_factory=list)
    editable: bool = True
    graph_tooltip: str = "default"
    uid: Optional[str] = None
    version: int = 1

@dataclass
class GrafanaDataSource:
    """Grafana data source configuration."""
    name: str
    type: str  # prometheus, influxdb, elasticsearch, etc.
    url: str
    access: str = "proxy"  # direct or proxy
    basic_auth: bool = False
    basic_auth_user: Optional[str] = None
    basic_auth_password: Optional[str] = None
    with_credentials: bool = False
    is_default: bool = False
    json_data: Dict[str, Any] = field(default_factory=dict)
    secure_json_data: Dict[str, Any] = field(default_factory=dict)
    uid: Optional[str] = None

@dataclass
class GrafanaPanel:
    """Grafana panel configuration."""
    title: str
    type: str  # graph, stat, table, etc.
    targets: List[Dict[str, Any]] = field(default_factory=list)
    grid_pos: Dict[str, int] = field(default_factory=dict)
    options: Dict[str, Any] = field(default_factory=dict)
    field_config: Dict[str, Any] = field(default_factory=dict)
    transformations: List[Dict[str, Any]] = field(default_factory=list)
    alert: Optional[Dict[str, Any]] = None

@dataclass
class GrafanaAlert:
    """Grafana alert rule."""
    name: str
    message: str
    frequency: str = "10s"
    conditions: List[Dict[str, Any]] = field(default_factory=list)
    execution_error_state: str = "alerting"
    no_data_state: str = "no_data"
    for_duration: str = "5m"
    annotations: Dict[str, str] = field(default_factory=dict)
    labels: Dict[str, str] = field(default_factory=dict)

@dataclass
class GrafanaNotificationChannel:
    """Grafana notification channel."""
    name: str
    type: str  # email, slack, webhook, etc.
    settings: Dict[str, Any] = field(default_factory=dict)
    is_default: bool = False
    send_reminder: bool = False
    disable_resolve_message: bool = False
    uid: Optional[str] = None

class GrafanaAPIClient:
    """Grafana API client with authentication and error handling."""
    
    def __init__(self, config: GrafanaConfig):
        self.config = config
        self.session: Optional[requests.Session] = None
        self.headers = {}
        
    async def initialize(self) -> bool:
        """Initialize API client with authentication."""
        if not HTTP_AVAILABLE:
            logger.error("HTTP libraries not available")
            return False
        
        try:
            self.session = requests.Session()
            
            # Setup authentication
            if self.config.api_key:
                self.headers['Authorization'] = f'Bearer {self.config.api_key}'
            else:
                # Basic auth
                auth_string = base64.b64encode(
                    f'{self.config.username}:{self.config.password}'.encode()
                ).decode()
                self.headers['Authorization'] = f'Basic {auth_string}'
            
            self.headers['Content-Type'] = 'application/json'
            self.headers['Accept'] = 'application/json'
            
            # Test connection
            response = await self._make_request('GET', '/api/health')
            if response and response.get('database') == 'ok':
                logger.info(f"Connected to Grafana at {self.config.base_url}")
                return True
            else:
                logger.error("Grafana health check failed")
                return False
                
        except Exception as e:
            logger.error(f"Error initializing Grafana client: {str(e)}")
            return False
    
    async def _make_request(self, method: str, endpoint: str, data: Optional[Dict[str, Any]] = None,
                          params: Optional[Dict[str, Any]] = None) -> Optional[Dict[str, Any]]:
        """Make HTTP request to Grafana API."""
        if not self.session:
            raise RuntimeError("API client not initialized")
        
        url = urljoin(self.config.base_url, endpoint)
        
        try:
            kwargs = {
                'headers': self.headers,
                'timeout': self.config.timeout,
                'verify': self.config.verify_ssl
            }
            
            if params:
                kwargs['params'] = params
            
            if data:
                kwargs['json'] = data
            
            response = self.session.request(method, url, **kwargs)
            
            # Handle different response codes
            if response.status_code in [200, 201]:
                return response.json() if response.content else {}
            elif response.status_code == 204:
                return {}
            elif response.status_code == 404:
                return None
            else:
                logger.error(f"Grafana API error {response.status_code}: {response.text}")
                response.raise_for_status()
                
        except requests.exceptions.RequestException as e:
            logger.error(f"Request error: {str(e)}")
            raise
        except json.JSONDecodeError as e:
            logger.error(f"JSON decode error: {str(e)}")
            raise
    
    async def get(self, endpoint: str, params: Optional[Dict[str, Any]] = None) -> Optional[Dict[str, Any]]:
        """GET request."""
        return await self._make_request('GET', endpoint, params=params)
    
    async def post(self, endpoint: str, data: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """POST request."""
        return await self._make_request('POST', endpoint, data=data)
    
    async def put(self, endpoint: str, data: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """PUT request."""
        return await self._make_request('PUT', endpoint, data=data)
    
    async def patch(self, endpoint: str, data: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """PATCH request."""
        return await self._make_request('PATCH', endpoint, data=data)
    
    async def delete(self, endpoint: str) -> bool:
        """DELETE request."""
        result = await self._make_request('DELETE', endpoint)
        return result is not None

class GrafanaOperator:
    """@grafana operator implementation with full production features."""
    
    def __init__(self):
        self.api_client: Optional[GrafanaAPIClient] = None
        self.config: Optional[GrafanaConfig] = None
        self.operation_stats = {
            'dashboard_operations': 0,
            'datasource_operations': 0,
            'alert_operations': 0,
            'panel_operations': 0,
            'query_operations': 0
        }
        self._executor = ThreadPoolExecutor(max_workers=5)
    
    async def connect(self, config: Optional[GrafanaConfig] = None) -> bool:
        """Connect to Grafana instance."""
        if config is None:
            config = GrafanaConfig()
        
        self.config = config
        self.api_client = GrafanaAPIClient(config)
        
        return await self.api_client.initialize()
    
    # Dashboard Operations
    async def create_dashboard(self, dashboard: GrafanaDashboard, folder_id: Optional[int] = None,
                             overwrite: bool = False) -> Optional[Dict[str, Any]]:
        """Create dashboard."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        dashboard_data = {
            "dashboard": {
                "uid": dashboard.uid,
                "title": dashboard.title,
                "tags": dashboard.tags,
                "timezone": dashboard.timezone,
                "panels": dashboard.panels,
                "time": dashboard.time_range,
                "refresh": dashboard.refresh,
                "annotations": {"list": dashboard.annotations},
                "templating": {"list": dashboard.templating.get("list", [])},
                "links": dashboard.links,
                "editable": dashboard.editable,
                "graphTooltip": 0,  # default tooltip
                "version": dashboard.version
            },
            "overwrite": overwrite
        }
        
        if folder_id is not None:
            dashboard_data["folderId"] = folder_id
        
        try:
            result = await self.api_client.post('/api/dashboards/db', dashboard_data)
            if result:
                self.operation_stats['dashboard_operations'] += 1
                logger.info(f"Created dashboard: {dashboard.title}")
            return result
            
        except Exception as e:
            logger.error(f"Error creating dashboard: {str(e)}")
            raise
    
    async def get_dashboard_by_uid(self, uid: str) -> Optional[Dict[str, Any]]:
        """Get dashboard by UID."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            result = await self.api_client.get(f'/api/dashboards/uid/{uid}')
            return result
            
        except Exception as e:
            logger.error(f"Error getting dashboard {uid}: {str(e)}")
            raise
    
    async def update_dashboard(self, dashboard: GrafanaDashboard, folder_id: Optional[int] = None) -> Optional[Dict[str, Any]]:
        """Update dashboard."""
        return await self.create_dashboard(dashboard, folder_id, overwrite=True)
    
    async def delete_dashboard_by_uid(self, uid: str) -> bool:
        """Delete dashboard by UID."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            success = await self.api_client.delete(f'/api/dashboards/uid/{uid}')
            if success:
                self.operation_stats['dashboard_operations'] += 1
                logger.info(f"Deleted dashboard: {uid}")
            return success
            
        except Exception as e:
            logger.error(f"Error deleting dashboard {uid}: {str(e)}")
            raise
    
    async def search_dashboards(self, query: str = "", tags: List[str] = None,
                               folder_ids: List[int] = None) -> List[Dict[str, Any]]:
        """Search dashboards."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        params = {}
        if query:
            params['query'] = query
        if tags:
            params['tag'] = tags
        if folder_ids:
            params['folderIds'] = folder_ids
        
        try:
            result = await self.api_client.get('/api/search', params)
            return result if result else []
            
        except Exception as e:
            logger.error(f"Error searching dashboards: {str(e)}")
            raise
    
    # Data Source Operations
    async def create_datasource(self, datasource: GrafanaDataSource) -> Optional[Dict[str, Any]]:
        """Create data source."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        datasource_data = {
            "name": datasource.name,
            "type": datasource.type,
            "url": datasource.url,
            "access": datasource.access,
            "basicAuth": datasource.basic_auth,
            "withCredentials": datasource.with_credentials,
            "isDefault": datasource.is_default,
            "jsonData": datasource.json_data,
            "uid": datasource.uid
        }
        
        if datasource.basic_auth_user:
            datasource_data["basicAuthUser"] = datasource.basic_auth_user
        
        if datasource.secure_json_data:
            datasource_data["secureJsonData"] = datasource.secure_json_data
        
        try:
            result = await self.api_client.post('/api/datasources', datasource_data)
            if result:
                self.operation_stats['datasource_operations'] += 1
                logger.info(f"Created datasource: {datasource.name}")
            return result
            
        except Exception as e:
            logger.error(f"Error creating datasource: {str(e)}")
            raise
    
    async def get_datasource_by_name(self, name: str) -> Optional[Dict[str, Any]]:
        """Get data source by name."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            result = await self.api_client.get(f'/api/datasources/name/{name}')
            return result
            
        except Exception as e:
            logger.error(f"Error getting datasource {name}: {str(e)}")
            raise
    
    async def update_datasource(self, datasource_id: int, datasource: GrafanaDataSource) -> Optional[Dict[str, Any]]:
        """Update data source."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        datasource_data = {
            "id": datasource_id,
            "name": datasource.name,
            "type": datasource.type,
            "url": datasource.url,
            "access": datasource.access,
            "basicAuth": datasource.basic_auth,
            "withCredentials": datasource.with_credentials,
            "isDefault": datasource.is_default,
            "jsonData": datasource.json_data
        }
        
        try:
            result = await self.api_client.put(f'/api/datasources/{datasource_id}', datasource_data)
            if result:
                self.operation_stats['datasource_operations'] += 1
                logger.info(f"Updated datasource: {datasource.name}")
            return result
            
        except Exception as e:
            logger.error(f"Error updating datasource: {str(e)}")
            raise
    
    async def delete_datasource_by_name(self, name: str) -> bool:
        """Delete data source by name."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            success = await self.api_client.delete(f'/api/datasources/name/{name}')
            if success:
                self.operation_stats['datasource_operations'] += 1
                logger.info(f"Deleted datasource: {name}")
            return success
            
        except Exception as e:
            logger.error(f"Error deleting datasource {name}: {str(e)}")
            raise
    
    async def list_datasources(self) -> List[Dict[str, Any]]:
        """List all data sources."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            result = await self.api_client.get('/api/datasources')
            return result if result else []
            
        except Exception as e:
            logger.error(f"Error listing datasources: {str(e)}")
            raise
    
    # Alert Operations
    async def create_notification_channel(self, channel: GrafanaNotificationChannel) -> Optional[Dict[str, Any]]:
        """Create notification channel."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        channel_data = {
            "uid": channel.uid,
            "name": channel.name,
            "type": channel.type,
            "settings": channel.settings,
            "isDefault": channel.is_default,
            "sendReminder": channel.send_reminder,
            "disableResolveMessage": channel.disable_resolve_message
        }
        
        try:
            result = await self.api_client.post('/api/alert-notifications', channel_data)
            if result:
                self.operation_stats['alert_operations'] += 1
                logger.info(f"Created notification channel: {channel.name}")
            return result
            
        except Exception as e:
            logger.error(f"Error creating notification channel: {str(e)}")
            raise
    
    async def get_notification_channels(self) -> List[Dict[str, Any]]:
        """Get all notification channels."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            result = await self.api_client.get('/api/alert-notifications')
            return result if result else []
            
        except Exception as e:
            logger.error(f"Error getting notification channels: {str(e)}")
            raise
    
    async def test_notification_channel(self, channel_id: int) -> bool:
        """Test notification channel."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            result = await self.api_client.post(f'/api/alert-notifications/{channel_id}/test', {})
            return result is not None
            
        except Exception as e:
            logger.error(f"Error testing notification channel: {str(e)}")
            raise
    
    # Panel and Query Operations
    async def query_datasource(self, datasource_name: str, query: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """Query data source."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        # Get datasource ID
        datasource = await self.get_datasource_by_name(datasource_name)
        if not datasource:
            raise ValueError(f"Datasource {datasource_name} not found")
        
        query_data = {
            "queries": [query],
            "from": query.get("from", "now-1h"),
            "to": query.get("to", "now")
        }
        
        try:
            result = await self.api_client.post(f'/api/datasources/{datasource["id"]}/query', query_data)
            if result:
                self.operation_stats['query_operations'] += 1
            return result
            
        except Exception as e:
            logger.error(f"Error querying datasource: {str(e)}")
            raise
    
    async def create_panel(self, panel: GrafanaPanel) -> Dict[str, Any]:
        """Create panel configuration."""
        panel_data = {
            "id": int(time.time()),  # Simple ID generation
            "title": panel.title,
            "type": panel.type,
            "targets": panel.targets,
            "gridPos": panel.grid_pos,
            "options": panel.options,
            "fieldConfig": panel.field_config,
            "transformations": panel.transformations
        }
        
        if panel.alert:
            panel_data["alert"] = panel.alert
        
        self.operation_stats['panel_operations'] += 1
        return panel_data
    
    # Folder Operations
    async def create_folder(self, title: str, uid: Optional[str] = None) -> Optional[Dict[str, Any]]:
        """Create folder."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        folder_data = {
            "uid": uid,
            "title": title
        }
        
        try:
            result = await self.api_client.post('/api/folders', folder_data)
            if result:
                logger.info(f"Created folder: {title}")
            return result
            
        except Exception as e:
            logger.error(f"Error creating folder: {str(e)}")
            raise
    
    async def get_folders(self) -> List[Dict[str, Any]]:
        """Get all folders."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            result = await self.api_client.get('/api/folders')
            return result if result else []
            
        except Exception as e:
            logger.error(f"Error getting folders: {str(e)}")
            raise
    
    # Organization Operations
    async def get_current_org(self) -> Optional[Dict[str, Any]]:
        """Get current organization."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            result = await self.api_client.get('/api/org')
            return result
            
        except Exception as e:
            logger.error(f"Error getting current org: {str(e)}")
            raise
    
    async def get_org_users(self) -> List[Dict[str, Any]]:
        """Get organization users."""
        if not self.api_client:
            raise RuntimeError("Not connected to Grafana")
        
        try:
            result = await self.api_client.get('/api/org/users')
            return result if result else []
            
        except Exception as e:
            logger.error(f"Error getting org users: {str(e)}")
            raise
    
    # Helper Methods
    async def create_prometheus_dashboard(self, title: str, metrics: List[str],
                                        prometheus_datasource: str = "Prometheus") -> Optional[Dict[str, Any]]:
        """Create a dashboard for Prometheus metrics."""
        panels = []
        
        for i, metric in enumerate(metrics):
            panel = GrafanaPanel(
                title=f"{metric} Graph",
                type="graph",
                targets=[{
                    "expr": metric,
                    "datasource": prometheus_datasource,
                    "refId": "A"
                }],
                grid_pos={
                    "h": 8,
                    "w": 12,
                    "x": (i % 2) * 12,
                    "y": (i // 2) * 8
                }
            )
            panels.append(await self.create_panel(panel))
        
        dashboard = GrafanaDashboard(
            title=title,
            panels=panels,
            tags=["prometheus", "auto-generated"]
        )
        
        return await self.create_dashboard(dashboard)
    
    def get_statistics(self) -> Dict[str, Any]:
        """Get operation statistics."""
        return {
            'operations': self.operation_stats.copy(),
            'connected': self.api_client is not None,
            'base_url': self.config.base_url if self.config else None
        }
    
    async def close(self):
        """Close connections and cleanup."""
        if self.api_client and self.api_client.session:
            self.api_client.session.close()
        
        self._executor.shutdown(wait=True)
        logger.info("Grafana operator closed")

# Export the operator
__all__ = [
    'GrafanaOperator', 'GrafanaConfig', 'GrafanaDashboard', 'GrafanaDataSource',
    'GrafanaPanel', 'GrafanaAlert', 'GrafanaNotificationChannel'
] 