#!/usr/bin/env python3
"""
Tests for Advanced Performance Optimization Engine
Goal 7.1 Implementation Tests
"""

import unittest
import time
import threading
import asyncio
from unittest.mock import patch, MagicMock
import sys
import os

# Add parent directory to path for imports
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))

from performance_engine import PerformanceEngine, CacheEntry, optimize_operation, optimize_async_operation

class TestPerformanceEngine(unittest.TestCase):
    """Test cases for PerformanceEngine"""
    
    def setUp(self):
        """Set up test fixtures"""
        self.engine = PerformanceEngine(max_cache_size=100, max_memory_mb=100)
    
    def tearDown(self):
        """Clean up after tests"""
        self.engine.clear_all_caches()
    
    def test_cache_set_and_get(self):
        """Test basic cache set and get operations"""
        # Test basic set/get
        self.engine.cache_set("test_key", "test_value")
        result = self.engine.cache_get("test_key")
        self.assertEqual(result, "test_value")
        
        # Test with TTL
        self.engine.cache_set("ttl_key", "ttl_value", ttl=1)
        result = self.engine.cache_get("ttl_key")
        self.assertEqual(result, "ttl_value")
        
        # Wait for TTL to expire
        time.sleep(1.1)
        result = self.engine.cache_get("ttl_key")
        self.assertIsNone(result)
    
    def test_cache_eviction(self):
        """Test cache eviction when full"""
        # Fill cache beyond limit
        for i in range(150):
            self.engine.cache_set(f"key_{i}", f"value_{i}")
        
        # Check that cache size is within limits
        self.assertLessEqual(len(self.engine.cache), self.engine.max_cache_size)
        
        # Verify LRU eviction (oldest keys should be gone)
        self.assertIsNone(self.engine.cache_get("key_0"))
        self.assertIsNotNone(self.engine.cache_get("key_149"))
    
    def test_cache_access_tracking(self):
        """Test cache access tracking and LRU behavior"""
        # Add multiple entries
        self.engine.cache_set("key1", "value1")
        self.engine.cache_set("key2", "value2")
        self.engine.cache_set("key3", "value3")
        
        # Access key1 to make it most recently used
        self.engine.cache_get("key1")
        
        # Fill cache to trigger eviction
        for i in range(100):
            self.engine.cache_set(f"evict_key_{i}", f"evict_value_{i}")
        
        # key1 should still be there (most recently used)
        self.assertIsNotNone(self.engine.cache_get("key1"))
        # key2 and key3 should be evicted (least recently used)
        self.assertIsNone(self.engine.cache_get("key2"))
        self.assertIsNone(self.engine.cache_get("key3"))
    
    def test_profiling_operation(self):
        """Test operation profiling"""
        with self.engine.profile_operation("test_op"):
            time.sleep(0.1)  # Simulate work
        
        metrics = self.engine.get_performance_metrics()
        self.assertIn("test_op", metrics["operations"])
        
        op_metrics = metrics["operations"]["test_op"]
        self.assertEqual(op_metrics["count"], 1)
        self.assertGreater(op_metrics["average_ms"], 0)
    
    def test_memory_monitoring(self):
        """Test memory monitoring functionality"""
        # Trigger some memory usage
        for i in range(1000):
            self.engine.cache_set(f"mem_key_{i}", "x" * 1000)
        
        metrics = self.engine.get_performance_metrics()
        self.assertIn("memory", metrics)
        self.assertIn("current_mb", metrics["memory"])
        self.assertGreater(metrics["memory"]["current_mb"], 0)
    
    def test_resource_pooling(self):
        """Test resource pooling functionality"""
        def create_resource():
            return {"id": f"resource_{time.time()}", "created": time.time()}
        
        # Get resources from pool
        resource1 = self.engine.get_resource_pool("test_pool", create_resource)
        resource2 = self.engine.get_resource_pool("test_pool", create_resource)
        
        self.assertIsNotNone(resource1)
        self.assertIsNotNone(resource2)
        self.assertNotEqual(resource1["id"], resource2["id"])
        
        # Return resources to pool
        self.engine.return_resource_to_pool("test_pool", resource1)
        self.engine.return_resource_to_pool("test_pool", resource2)
        
        # Get resources again - should reuse from pool
        resource3 = self.engine.get_resource_pool("test_pool", create_resource)
        self.assertIn(resource3["id"], [resource1["id"], resource2["id"]])
    
    def test_optimize_operation_decorator(self):
        """Test optimize_operation decorator"""
        @optimize_operation("decorated_op")
        def test_function():
            time.sleep(0.1)
            return "success"
        
        result = test_function()
        self.assertEqual(result, "success")
        
        metrics = self.engine.get_performance_metrics()
        self.assertIn("decorated_op", metrics["operations"])
    
    def test_optimize_async_operation_decorator(self):
        """Test optimize_async_operation decorator"""
        @optimize_async_operation("async_decorated_op")
        async def async_test_function():
            await asyncio.sleep(0.1)
            return "async_success"
        
        async def run_test():
            result = await async_test_function()
            self.assertEqual(result, "async_success")
            
            metrics = self.engine.get_performance_metrics()
            self.assertIn("async_decorated_op", metrics["operations"])
        
        asyncio.run(run_test())
    
    def test_concurrent_access(self):
        """Test concurrent access to cache"""
        def cache_worker(worker_id):
            for i in range(10):
                key = f"worker_{worker_id}_key_{i}"
                value = f"worker_{worker_id}_value_{i}"
                self.engine.cache_set(key, value)
                time.sleep(0.01)
                retrieved = self.engine.cache_get(key)
                self.assertEqual(retrieved, value)
        
        # Create multiple threads
        threads = []
        for i in range(5):
            thread = threading.Thread(target=cache_worker, args=(i,))
            threads.append(thread)
            thread.start()
        
        # Wait for all threads to complete
        for thread in threads:
            thread.join()
        
        # Verify cache integrity
        metrics = self.engine.get_performance_metrics()
        self.assertGreater(metrics["cache"]["size"], 0)
    
    def test_cache_metrics(self):
        """Test cache metrics calculation"""
        # Add some cache entries
        for i in range(10):
            self.engine.cache_set(f"metric_key_{i}", f"metric_value_{i}")
        
        # Access some entries multiple times
        for _ in range(5):
            self.engine.cache_get("metric_key_0")
            self.engine.cache_get("metric_key_1")
        
        metrics = self.engine.get_performance_metrics()
        cache_metrics = metrics["cache"]
        
        self.assertEqual(cache_metrics["size"], 10)
        self.assertEqual(cache_metrics["max_size"], 100)
        self.assertGreater(cache_metrics["hit_rate"], 0)
        self.assertGreater(cache_metrics["memory_usage_mb"], 0)
    
    def test_memory_cleanup(self):
        """Test memory cleanup functionality"""
        # Fill cache to trigger cleanup
        for i in range(1000):
            self.engine.cache_set(f"cleanup_key_{i}", "x" * 1000)
        
        # Force cleanup
        self.engine._cleanup_memory()
        
        # Verify cache is cleared
        metrics = self.engine.get_performance_metrics()
        self.assertEqual(metrics["cache"]["size"], 0)
    
    def test_cache_entry_creation(self):
        """Test CacheEntry dataclass"""
        entry = CacheEntry(
            value="test_value",
            created_at=time.time(),
            last_accessed=time.time(),
            access_count=5,
            size_bytes=1024,
            ttl=60.0
        )
        
        self.assertEqual(entry.value, "test_value")
        self.assertEqual(entry.access_count, 5)
        self.assertEqual(entry.size_bytes, 1024)
        self.assertEqual(entry.ttl, 60.0)

if __name__ == "__main__":
    unittest.main() 