"""
Advanced Blockchain Engine with Smart Contract Support
Ethereum/Solana integration with wallet management and DeFi capabilities.
"""

import asyncio
import hashlib
import json
import logging
import time
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime
from typing import Any, Dict, List, Optional, Union
from decimal import Decimal

# Web3 and blockchain libraries
try:
    from web3 import Web3
    from eth_account import Account
    from eth_utils import to_checksum_address
    WEB3_AVAILABLE = True
except ImportError:
    WEB3_AVAILABLE = False
    print("Web3.py not available. Ethereum features will be limited.")

try:
    import solana
    from solana.rpc.api import Client as SolanaClient
    from solana.keypair import Keypair
    from solana.publickey import PublicKey
    SOLANA_AVAILABLE = True
except ImportError:
    SOLANA_AVAILABLE = False
    print("Solana libraries not available. Solana features will be limited.")

# Cryptography
try:
    from cryptography.hazmat.primitives import hashes, serialization
    from cryptography.hazmat.primitives.asymmetric import rsa
    CRYPTO_AVAILABLE = True
except ImportError:
    CRYPTO_AVAILABLE = False
    print("Cryptography library not available. Wallet features will be limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class Block:
    """Basic blockchain block structure."""
    index: int
    timestamp: float
    data: Dict[str, Any]
    previous_hash: str
    nonce: int = 0
    hash: str = ""

@dataclass
class Transaction:
    """Blockchain transaction."""
    from_address: str
    to_address: str
    amount: Decimal
    fee: Decimal
    timestamp: float
    signature: str = ""
    hash: str = ""
    status: str = "pending"  # pending, confirmed, failed

@dataclass
class Wallet:
    """Cryptocurrency wallet."""
    address: str
    private_key: str
    public_key: str
    mnemonic: Optional[str] = None
    balance: Decimal = Decimal('0')
    network: str = "ethereum"

@dataclass
class SmartContractResult:
    """Smart contract execution result."""
    transaction_hash: str
    gas_used: int
    status: str
    logs: List[Dict[str, Any]]
    return_value: Any = None

class SimpleBlockchain:
    """Simple blockchain implementation for testing."""
    
    def __init__(self, difficulty: int = 4):
        self.chain = [self.create_genesis_block()]
        self.difficulty = difficulty
        self.pending_transactions = []
        self.mining_reward = Decimal('10')
    
    def create_genesis_block(self) -> Block:
        """Create the first block in the chain."""
        genesis_block = Block(
            index=0,
            timestamp=time.time(),
            data={"message": "Genesis Block"},
            previous_hash="0"
        )
        genesis_block.hash = self.calculate_hash(genesis_block)
        return genesis_block
    
    def calculate_hash(self, block: Block) -> str:
        """Calculate block hash."""
        block_string = json.dumps({
            "index": block.index,
            "timestamp": block.timestamp,
            "data": block.data,
            "previous_hash": block.previous_hash,
            "nonce": block.nonce
        }, sort_keys=True)
        
        return hashlib.sha256(block_string.encode()).hexdigest()
    
    def mine_block(self, block: Block) -> Block:
        """Mine a block using proof of work."""
        target = "0" * self.difficulty
        
        while block.hash[:self.difficulty] != target:
            block.nonce += 1
            block.hash = self.calculate_hash(block)
        
        logger.info(f"Block mined: {block.hash}")
        return block
    
    def add_block(self, data: Dict[str, Any]) -> Block:
        """Add a new block to the chain."""
        previous_block = self.chain[-1]
        
        new_block = Block(
            index=len(self.chain),
            timestamp=time.time(),
            data=data,
            previous_hash=previous_block.hash
        )
        
        mined_block = self.mine_block(new_block)
        self.chain.append(mined_block)
        
        return mined_block
    
    def is_chain_valid(self) -> bool:
        """Validate the blockchain."""
        for i in range(1, len(self.chain)):
            current_block = self.chain[i]
            previous_block = self.chain[i - 1]
            
            # Check if current block hash is valid
            if current_block.hash != self.calculate_hash(current_block):
                return False
            
            # Check if current block points to previous block
            if current_block.previous_hash != previous_block.hash:
                return False
        
        return True
    
    def get_balance(self, address: str) -> Decimal:
        """Get balance for an address."""
        balance = Decimal('0')
        
        for block in self.chain:
            if 'transactions' in block.data:
                for tx in block.data['transactions']:
                    if tx.get('to_address') == address:
                        balance += Decimal(str(tx.get('amount', 0)))
                    if tx.get('from_address') == address:
                        balance -= Decimal(str(tx.get('amount', 0)))
                        balance -= Decimal(str(tx.get('fee', 0)))
        
        return balance

class WalletManager:
    """Cryptocurrency wallet management."""
    
    def __init__(self):
        self.wallets = {}
    
    def create_ethereum_wallet(self, mnemonic: Optional[str] = None) -> Wallet:
        """Create a new Ethereum wallet."""
        if not WEB3_AVAILABLE:
            raise Exception("Web3.py not available for Ethereum wallet creation")
        
        # Generate or use provided mnemonic
        if mnemonic:
            # In a real implementation, you'd derive keys from mnemonic
            account = Account.create()
        else:
            account = Account.create()
        
        wallet = Wallet(
            address=account.address,
            private_key=account.key.hex(),
            public_key=account.key.hex(),  # Simplified
            network="ethereum"
        )
        
        self.wallets[wallet.address] = wallet
        logger.info(f"Created Ethereum wallet: {wallet.address}")
        
        return wallet
    
    def create_solana_wallet(self) -> Wallet:
        """Create a new Solana wallet."""
        if not SOLANA_AVAILABLE:
            raise Exception("Solana libraries not available for wallet creation")
        
        keypair = Keypair()
        
        wallet = Wallet(
            address=str(keypair.public_key),
            private_key=str(keypair.secret_key),
            public_key=str(keypair.public_key),
            network="solana"
        )
        
        self.wallets[wallet.address] = wallet
        logger.info(f"Created Solana wallet: {wallet.address}")
        
        return wallet
    
    def import_wallet(self, private_key: str, network: str = "ethereum") -> Wallet:
        """Import wallet from private key."""
        if network == "ethereum" and WEB3_AVAILABLE:
            account = Account.from_key(private_key)
            
            wallet = Wallet(
                address=account.address,
                private_key=private_key,
                public_key=account.key.hex(),
                network=network
            )
        else:
            # Generic wallet import
            wallet = Wallet(
                address=f"imported_{network}_wallet",
                private_key=private_key,
                public_key="",
                network=network
            )
        
        self.wallets[wallet.address] = wallet
        logger.info(f"Imported {network} wallet: {wallet.address}")
        
        return wallet
    
    def get_wallet(self, address: str) -> Optional[Wallet]:
        """Get wallet by address."""
        return self.wallets.get(address)
    
    def list_wallets(self) -> List[Wallet]:
        """List all wallets."""
        return list(self.wallets.values())
    
    def export_wallet(self, address: str) -> Dict[str, str]:
        """Export wallet data."""
        wallet = self.wallets.get(address)
        if not wallet:
            raise ValueError(f"Wallet not found: {address}")
        
        return {
            "address": wallet.address,
            "private_key": wallet.private_key,
            "public_key": wallet.public_key,
            "network": wallet.network
        }

class EthereumConnector:
    """Ethereum blockchain connector."""
    
    def __init__(self, rpc_url: str = "https://mainnet.infura.io/v3/YOUR_PROJECT_ID"):
        self.rpc_url = rpc_url
        self.web3 = None
        
        if WEB3_AVAILABLE:
            try:
                self.web3 = Web3(Web3.HTTPProvider(rpc_url))
                logger.info(f"Connected to Ethereum: {self.web3.isConnected()}")
            except Exception as e:
                logger.warning(f"Could not connect to Ethereum: {str(e)}")
    
    def get_balance(self, address: str) -> Decimal:
        """Get ETH balance for address."""
        if not self.web3:
            return Decimal('0')
        
        try:
            balance_wei = self.web3.eth.get_balance(to_checksum_address(address))
            balance_eth = Web3.fromWei(balance_wei, 'ether')
            return Decimal(str(balance_eth))
        except Exception as e:
            logger.error(f"Error getting balance: {str(e)}")
            return Decimal('0')
    
    def send_transaction(self, from_wallet: Wallet, to_address: str, 
                        amount: Decimal, gas_price: Optional[int] = None) -> str:
        """Send Ethereum transaction."""
        if not self.web3:
            raise Exception("Web3 not available")
        
        try:
            # Get transaction parameters
            nonce = self.web3.eth.get_transaction_count(from_wallet.address)
            gas_price = gas_price or self.web3.eth.gas_price
            
            # Build transaction
            transaction = {
                'nonce': nonce,
                'to': to_checksum_address(to_address),
                'value': Web3.toWei(amount, 'ether'),
                'gas': 21000,  # Standard gas limit for ETH transfer
                'gasPrice': gas_price,
            }
            
            # Sign transaction
            signed_txn = self.web3.eth.account.sign_transaction(
                transaction, from_wallet.private_key
            )
            
            # Send transaction
            tx_hash = self.web3.eth.send_raw_transaction(signed_txn.rawTransaction)
            
            logger.info(f"Transaction sent: {tx_hash.hex()}")
            return tx_hash.hex()
            
        except Exception as e:
            logger.error(f"Error sending transaction: {str(e)}")
            raise
    
    def deploy_contract(self, wallet: Wallet, contract_bytecode: str, 
                       abi: List[Dict], constructor_args: List[Any] = None) -> str:
        """Deploy smart contract."""
        if not self.web3:
            raise Exception("Web3 not available")
        
        try:
            # Create contract instance
            contract = self.web3.eth.contract(
                abi=abi,
                bytecode=contract_bytecode
            )
            
            # Build constructor transaction
            constructor = contract.constructor(*constructor_args) if constructor_args else contract.constructor()
            
            # Get transaction parameters
            nonce = self.web3.eth.get_transaction_count(wallet.address)
            
            transaction = constructor.buildTransaction({
                'from': wallet.address,
                'nonce': nonce,
                'gas': 2000000,  # High gas limit for deployment
                'gasPrice': self.web3.eth.gas_price,
            })
            
            # Sign and send transaction
            signed_txn = self.web3.eth.account.sign_transaction(
                transaction, wallet.private_key
            )
            
            tx_hash = self.web3.eth.send_raw_transaction(signed_txn.rawTransaction)
            
            logger.info(f"Contract deployed: {tx_hash.hex()}")
            return tx_hash.hex()
            
        except Exception as e:
            logger.error(f"Error deploying contract: {str(e)}")
            raise
    
    def call_contract_function(self, contract_address: str, abi: List[Dict],
                             function_name: str, args: List[Any] = None,
                             from_wallet: Optional[Wallet] = None) -> Any:
        """Call smart contract function."""
        if not self.web3:
            raise Exception("Web3 not available")
        
        try:
            # Create contract instance
            contract = self.web3.eth.contract(
                address=to_checksum_address(contract_address),
                abi=abi
            )
            
            # Get function
            contract_function = getattr(contract.functions, function_name)
            
            if from_wallet:
                # This is a transaction (state-changing function)
                nonce = self.web3.eth.get_transaction_count(from_wallet.address)
                
                transaction = contract_function(*args).buildTransaction({
                    'from': from_wallet.address,
                    'nonce': nonce,
                    'gas': 200000,
                    'gasPrice': self.web3.eth.gas_price,
                })
                
                signed_txn = self.web3.eth.account.sign_transaction(
                    transaction, from_wallet.private_key
                )
                
                tx_hash = self.web3.eth.send_raw_transaction(signed_txn.rawTransaction)
                return tx_hash.hex()
            else:
                # This is a call (read-only function)
                if args:
                    result = contract_function(*args).call()
                else:
                    result = contract_function().call()
                
                return result
                
        except Exception as e:
            logger.error(f"Error calling contract function: {str(e)}")
            raise

class BlockchainEngine:
    """
    Comprehensive Blockchain Engine with Smart Contract Support.
    Supports Ethereum, Solana integration, and wallet management.
    """
    
    def __init__(self, ethereum_rpc: str = None):
        self.wallet_manager = WalletManager()
        self.simple_blockchain = SimpleBlockchain()
        
        # Initialize connectors
        self.ethereum_connector = None
        if ethereum_rpc and WEB3_AVAILABLE:
            self.ethereum_connector = EthereumConnector(ethereum_rpc)
        
        self.transaction_history = []
        self.contract_registry = {}
        
        self.stats = {
            'wallets_created': 0,
            'transactions_sent': 0,
            'contracts_deployed': 0,
            'total_processing_time': 0.0
        }
    
    async def create_wallet(self, network: str = "ethereum", 
                          mnemonic: Optional[str] = None) -> Wallet:
        """Create a new wallet."""
        start_time = time.time()
        
        logger.info(f"Creating {network} wallet")
        
        try:
            if network.lower() == "ethereum":
                wallet = self.wallet_manager.create_ethereum_wallet(mnemonic)
            elif network.lower() == "solana":
                wallet = self.wallet_manager.create_solana_wallet()
            else:
                raise ValueError(f"Unsupported network: {network}")
            
            self.stats['wallets_created'] += 1
            self.stats['total_processing_time'] += time.time() - start_time
            
            return wallet
            
        except Exception as e:
            logger.error(f"Error creating wallet: {str(e)}")
            raise
    
    async def import_wallet(self, private_key: str, network: str = "ethereum") -> Wallet:
        """Import existing wallet."""
        logger.info(f"Importing {network} wallet")
        
        wallet = self.wallet_manager.import_wallet(private_key, network)
        self.stats['wallets_created'] += 1
        
        return wallet
    
    async def get_balance(self, address: str, network: str = "ethereum") -> Decimal:
        """Get wallet balance."""
        logger.info(f"Getting balance for {address[:10]}...")
        
        if network == "ethereum" and self.ethereum_connector:
            return self.ethereum_connector.get_balance(address)
        elif network == "simple":
            return self.simple_blockchain.get_balance(address)
        else:
            # Mock balance for unsupported networks
            return Decimal('1.5')
    
    async def send_transaction(self, from_address: str, to_address: str, 
                             amount: Decimal, network: str = "ethereum",
                             fee: Decimal = Decimal('0.001')) -> str:
        """Send cryptocurrency transaction."""
        start_time = time.time()
        
        logger.info(f"Sending {amount} from {from_address[:10]}... to {to_address[:10]}...")
        
        # Get wallet
        wallet = self.wallet_manager.get_wallet(from_address)
        if not wallet:
            raise ValueError(f"Wallet not found: {from_address}")
        
        try:
            if network == "ethereum" and self.ethereum_connector:
                tx_hash = self.ethereum_connector.send_transaction(wallet, to_address, amount)
            else:
                # Create transaction for simple blockchain
                transaction = Transaction(
                    from_address=from_address,
                    to_address=to_address,
                    amount=amount,
                    fee=fee,
                    timestamp=time.time(),
                    hash=hashlib.sha256(f"{from_address}{to_address}{amount}{time.time()}".encode()).hexdigest()
                )
                
                # Add to simple blockchain
                block_data = {
                    "transactions": [
                        {
                            "from_address": transaction.from_address,
                            "to_address": transaction.to_address,
                            "amount": str(transaction.amount),
                            "fee": str(transaction.fee),
                            "hash": transaction.hash
                        }
                    ]
                }
                
                self.simple_blockchain.add_block(block_data)
                tx_hash = transaction.hash
            
            # Record transaction
            self.transaction_history.append({
                "hash": tx_hash,
                "from": from_address,
                "to": to_address,
                "amount": str(amount),
                "network": network,
                "timestamp": time.time()
            })
            
            self.stats['transactions_sent'] += 1
            self.stats['total_processing_time'] += time.time() - start_time
            
            logger.info(f"Transaction sent: {tx_hash}")
            return tx_hash
            
        except Exception as e:
            logger.error(f"Error sending transaction: {str(e)}")
            raise
    
    async def deploy_smart_contract(self, wallet_address: str, 
                                  contract_code: str, abi: List[Dict],
                                  constructor_args: List[Any] = None,
                                  network: str = "ethereum") -> str:
        """Deploy smart contract."""
        start_time = time.time()
        
        logger.info(f"Deploying smart contract on {network}")
        
        wallet = self.wallet_manager.get_wallet(wallet_address)
        if not wallet:
            raise ValueError(f"Wallet not found: {wallet_address}")
        
        try:
            if network == "ethereum" and self.ethereum_connector:
                tx_hash = self.ethereum_connector.deploy_contract(
                    wallet, contract_code, abi, constructor_args
                )
            else:
                # Mock deployment for other networks
                tx_hash = hashlib.sha256(f"contract_{time.time()}".encode()).hexdigest()
                
                # Store contract info
                contract_address = f"0x{tx_hash[:40]}"
                self.contract_registry[contract_address] = {
                    "abi": abi,
                    "deployer": wallet_address,
                    "network": network,
                    "deployment_hash": tx_hash,
                    "deployed_at": time.time()
                }
            
            self.stats['contracts_deployed'] += 1
            self.stats['total_processing_time'] += time.time() - start_time
            
            logger.info(f"Contract deployed: {tx_hash}")
            return tx_hash
            
        except Exception as e:
            logger.error(f"Error deploying contract: {str(e)}")
            raise
    
    async def call_smart_contract(self, contract_address: str, 
                                function_name: str, args: List[Any] = None,
                                from_wallet: Optional[str] = None,
                                network: str = "ethereum") -> Any:
        """Call smart contract function."""
        logger.info(f"Calling contract function {function_name} on {contract_address[:10]}...")
        
        # Get contract info
        contract_info = self.contract_registry.get(contract_address)
        if not contract_info and network != "ethereum":
            raise ValueError(f"Contract not found: {contract_address}")
        
        try:
            if network == "ethereum" and self.ethereum_connector:
                wallet = None
                if from_wallet:
                    wallet = self.wallet_manager.get_wallet(from_wallet)
                
                abi = contract_info['abi'] if contract_info else []
                
                result = self.ethereum_connector.call_contract_function(
                    contract_address, abi, function_name, args, wallet
                )
                
                return result
            else:
                # Mock contract call
                return f"Mock result for {function_name} with args {args}"
                
        except Exception as e:
            logger.error(f"Error calling contract: {str(e)}")
            raise
    
    def validate_blockchain(self, network: str = "simple") -> bool:
        """Validate blockchain integrity."""
        if network == "simple":
            return self.simple_blockchain.is_chain_valid()
        else:
            # For external networks, assume valid
            return True
    
    def get_transaction_history(self, address: Optional[str] = None) -> List[Dict]:
        """Get transaction history."""
        if address:
            return [
                tx for tx in self.transaction_history
                if tx['from'] == address or tx['to'] == address
            ]
        return self.transaction_history
    
    def get_network_info(self, network: str = "ethereum") -> Dict[str, Any]:
        """Get network information."""
        info = {
            "network": network,
            "connected": False,
            "block_height": 0,
            "gas_price": "0"
        }
        
        if network == "ethereum" and self.ethereum_connector and self.ethereum_connector.web3:
            try:
                info.update({
                    "connected": self.ethereum_connector.web3.isConnected(),
                    "block_height": self.ethereum_connector.web3.eth.block_number,
                    "gas_price": str(self.ethereum_connector.web3.eth.gas_price)
                })
            except:
                pass
        elif network == "simple":
            info.update({
                "connected": True,
                "block_height": len(self.simple_blockchain.chain),
                "gas_price": "0"
            })
        
        return info
    
    def get_processing_stats(self) -> Dict[str, Any]:
        """Get processing statistics."""
        return self.stats.copy()

# Example usage and testing
async def main():
    """Example usage of the Blockchain Engine."""
    print("=== Blockchain Engine Demo ===")
    
    # Initialize blockchain engine
    blockchain_engine = BlockchainEngine()
    
    # Test 1: Create wallets
    print("\n1. Creating wallets...")
    
    try:
        if WEB3_AVAILABLE:
            eth_wallet = await blockchain_engine.create_wallet("ethereum")
            print(f"Ethereum wallet created: {eth_wallet.address}")
        else:
            print("Ethereum wallet creation skipped (Web3.py not available)")
    except Exception as e:
        print(f"Ethereum wallet creation failed: {str(e)}")
    
    try:
        if SOLANA_AVAILABLE:
            sol_wallet = await blockchain_engine.create_wallet("solana")
            print(f"Solana wallet created: {sol_wallet.address}")
        else:
            print("Solana wallet creation skipped (Solana libraries not available)")
    except Exception as e:
        print(f"Solana wallet creation failed: {str(e)}")
    
    # Test 2: Simple blockchain operations
    print("\n2. Testing simple blockchain...")
    
    # Create test wallets for simple blockchain
    wallet1 = Wallet(
        address="wallet1_address",
        private_key="private_key_1",
        public_key="public_key_1",
        network="simple"
    )
    
    wallet2 = Wallet(
        address="wallet2_address", 
        private_key="private_key_2",
        public_key="public_key_2",
        network="simple"
    )
    
    blockchain_engine.wallet_manager.wallets[wallet1.address] = wallet1
    blockchain_engine.wallet_manager.wallets[wallet2.address] = wallet2
    
    # Send test transaction on simple blockchain
    try:
        tx_hash = await blockchain_engine.send_transaction(
            wallet1.address,
            wallet2.address,
            Decimal('5.0'),
            network="simple"
        )
        print(f"Simple blockchain transaction: {tx_hash}")
        
        # Check balances
        balance1 = await blockchain_engine.get_balance(wallet1.address, "simple")
        balance2 = await blockchain_engine.get_balance(wallet2.address, "simple")
        print(f"Wallet 1 balance: {balance1}")
        print(f"Wallet 2 balance: {balance2}")
        
    except Exception as e:
        print(f"Simple blockchain transaction failed: {str(e)}")
    
    # Test 3: Blockchain validation
    print("\n3. Testing blockchain validation...")
    is_valid = blockchain_engine.validate_blockchain("simple")
    print(f"Simple blockchain is valid: {is_valid}")
    
    # Test 4: Network information
    print("\n4. Network information...")
    networks = ["simple", "ethereum"]
    for network in networks:
        info = blockchain_engine.get_network_info(network)
        print(f"{network.title()} network: {info}")
    
    # Test 5: Transaction history
    print("\n5. Transaction history...")
    history = blockchain_engine.get_transaction_history()
    for tx in history:
        print(f"TX: {tx['hash'][:10]}... {tx['amount']} from {tx['from'][:10]}... to {tx['to'][:10]}...")
    
    # Test 6: Processing statistics
    print("\n6. Processing statistics:")
    stats = blockchain_engine.get_processing_stats()
    for key, value in stats.items():
        print(f"{key}: {value}")
    
    # Test 7: Available features
    print("\n7. Available features:")
    print(f"Web3.py (Ethereum): {WEB3_AVAILABLE}")
    print(f"Solana: {SOLANA_AVAILABLE}")
    print(f"Cryptography: {CRYPTO_AVAILABLE}")
    
    print("\n=== Blockchain Engine Demo Complete ===")

if __name__ == "__main__":
    asyncio.run(main()) 