"""
Advanced IoT Device Manager with Protocol Support
MQTT, CoAP, LoRaWAN device connectivity and management.
"""

import asyncio
import json
import logging
import random
import time
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from enum import Enum
from typing import Any, Callable, Dict, List, Optional, Union

# MQTT support
try:
    import paho.mqtt.client as mqtt
    MQTT_AVAILABLE = True
except ImportError:
    MQTT_AVAILABLE = False
    print("MQTT library not available. MQTT features will be limited.")

# CoAP support
try:
    from aiocoap import Context, Message, Code
    COAP_AVAILABLE = True
except ImportError:
    COAP_AVAILABLE = False
    print("CoAP library not available. CoAP features will be limited.")

# Serial communication for LoRaWAN
try:
    import serial
    SERIAL_AVAILABLE = True
except ImportError:
    SERIAL_AVAILABLE = False
    print("pySerial not available. Serial communication will be limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class DeviceStatus(Enum):
    """Device status enumeration."""
    ONLINE = "online"
    OFFLINE = "offline"
    CONNECTING = "connecting"
    ERROR = "error"
    MAINTENANCE = "maintenance"

class Protocol(Enum):
    """Communication protocol enumeration."""
    MQTT = "mqtt"
    COAP = "coap"
    LORAWAN = "lorawan"
    HTTP = "http"
    WEBSOCKET = "websocket"

@dataclass
class DeviceInfo:
    """IoT device information."""
    device_id: str
    name: str
    device_type: str
    protocol: Protocol
    status: DeviceStatus = DeviceStatus.OFFLINE
    ip_address: Optional[str] = None
    mac_address: Optional[str] = None
    firmware_version: Optional[str] = None
    last_seen: Optional[datetime] = None
    metadata: Dict[str, Any] = field(default_factory=dict)

@dataclass
class SensorReading:
    """Sensor data reading."""
    device_id: str
    sensor_type: str
    value: float
    unit: str
    timestamp: datetime
    quality: float = 1.0  # 0-1 quality score

@dataclass
class DeviceCommand:
    """Device command structure."""
    device_id: str
    command: str
    parameters: Dict[str, Any]
    timestamp: datetime
    response_timeout: int = 30
    status: str = "pending"  # pending, sent, acknowledged, failed

class MQTTManager:
    """MQTT protocol manager."""
    
    def __init__(self, broker_host: str = "localhost", broker_port: int = 1883):
        self.broker_host = broker_host
        self.broker_port = broker_port
        self.client = None
        self.connected = False
        self.subscriptions = {}
        self.message_callbacks = {}
        
        if MQTT_AVAILABLE:
            self.client = mqtt.Client()
            self.client.on_connect = self._on_connect
            self.client.on_message = self._on_message
            self.client.on_disconnect = self._on_disconnect
    
    def _on_connect(self, client, userdata, flags, rc):
        """MQTT connection callback."""
        if rc == 0:
            self.connected = True
            logger.info("Connected to MQTT broker")
            
            # Resubscribe to all topics
            for topic in self.subscriptions:
                client.subscribe(topic)
        else:
            logger.error(f"Failed to connect to MQTT broker: {rc}")
    
    def _on_message(self, client, userdata, msg):
        """MQTT message callback."""
        topic = msg.topic
        payload = msg.payload.decode('utf-8')
        
        logger.info(f"Received MQTT message on {topic}: {payload}")
        
        # Call registered callbacks
        for pattern, callback in self.message_callbacks.items():
            if pattern in topic:
                try:
                    callback(topic, payload)
                except Exception as e:
                    logger.error(f"Error in MQTT callback: {str(e)}")
    
    def _on_disconnect(self, client, userdata, rc):
        """MQTT disconnection callback."""
        self.connected = False
        logger.warning("Disconnected from MQTT broker")
    
    def connect(self) -> bool:
        """Connect to MQTT broker."""
        if not MQTT_AVAILABLE:
            logger.error("MQTT library not available")
            return False
        
        try:
            self.client.connect(self.broker_host, self.broker_port, 60)
            self.client.loop_start()
            return True
        except Exception as e:
            logger.error(f"Error connecting to MQTT broker: {str(e)}")
            return False
    
    def disconnect(self):
        """Disconnect from MQTT broker."""
        if self.client:
            self.client.loop_stop()
            self.client.disconnect()
    
    def publish(self, topic: str, payload: str, qos: int = 0) -> bool:
        """Publish MQTT message."""
        if not self.connected:
            logger.warning("Not connected to MQTT broker")
            return False
        
        try:
            result = self.client.publish(topic, payload, qos)
            return result.rc == mqtt.MQTT_ERR_SUCCESS
        except Exception as e:
            logger.error(f"Error publishing MQTT message: {str(e)}")
            return False
    
    def subscribe(self, topic: str, callback: Callable[[str, str], None]) -> bool:
        """Subscribe to MQTT topic."""
        if not MQTT_AVAILABLE:
            return False
        
        try:
            self.subscriptions[topic] = callback
            self.message_callbacks[topic] = callback
            
            if self.connected:
                result = self.client.subscribe(topic)
                return result[0] == mqtt.MQTT_ERR_SUCCESS
            return True
        except Exception as e:
            logger.error(f"Error subscribing to MQTT topic: {str(e)}")
            return False

class CoAPManager:
    """CoAP protocol manager."""
    
    def __init__(self):
        self.context = None
        self.servers = {}
    
    async def initialize(self):
        """Initialize CoAP context."""
        if COAP_AVAILABLE:
            try:
                self.context = await Context.create_client_context()
                logger.info("CoAP context initialized")
            except Exception as e:
                logger.error(f"Error initializing CoAP: {str(e)}")
    
    async def send_get_request(self, uri: str) -> Optional[str]:
        """Send CoAP GET request."""
        if not COAP_AVAILABLE or not self.context:
            logger.warning("CoAP not available")
            return None
        
        try:
            request = Message(code=Code.GET, uri=uri)
            response = await self.context.request(request).response
            
            return response.payload.decode('utf-8')
        except Exception as e:
            logger.error(f"Error sending CoAP GET request: {str(e)}")
            return None
    
    async def send_post_request(self, uri: str, payload: str) -> Optional[str]:
        """Send CoAP POST request."""
        if not COAP_AVAILABLE or not self.context:
            logger.warning("CoAP not available")
            return None
        
        try:
            request = Message(code=Code.POST, uri=uri, payload=payload.encode('utf-8'))
            response = await self.context.request(request).response
            
            return response.payload.decode('utf-8')
        except Exception as e:
            logger.error(f"Error sending CoAP POST request: {str(e)}")
            return None

class LoRaWANManager:
    """LoRaWAN protocol manager."""
    
    def __init__(self, port: str = '/dev/ttyUSB0', baudrate: int = 9600):
        self.port = port
        self.baudrate = baudrate
        self.serial_connection = None
        self.connected = False
    
    def connect(self) -> bool:
        """Connect to LoRaWAN gateway."""
        if not SERIAL_AVAILABLE:
            logger.error("pySerial not available for LoRaWAN")
            return False
        
        try:
            self.serial_connection = serial.Serial(
                self.port, self.baudrate, timeout=1
            )
            self.connected = True
            logger.info(f"Connected to LoRaWAN gateway on {self.port}")
            return True
        except Exception as e:
            logger.error(f"Error connecting to LoRaWAN gateway: {str(e)}")
            return False
    
    def disconnect(self):
        """Disconnect from LoRaWAN gateway."""
        if self.serial_connection:
            self.serial_connection.close()
            self.connected = False
    
    def send_command(self, command: str) -> Optional[str]:
        """Send AT command to LoRaWAN module."""
        if not self.connected:
            logger.warning("Not connected to LoRaWAN gateway")
            return None
        
        try:
            # Send command
            self.serial_connection.write(f"{command}\r\n".encode())
            
            # Read response
            response = self.serial_connection.readline().decode().strip()
            logger.info(f"LoRaWAN response: {response}")
            
            return response
        except Exception as e:
            logger.error(f"Error sending LoRaWAN command: {str(e)}")
            return None
    
    def send_data(self, device_id: str, data: str) -> bool:
        """Send data via LoRaWAN."""
        if not self.connected:
            return False
        
        # Convert data to hex
        hex_data = data.encode().hex()
        
        # Send AT command to transmit data
        response = self.send_command(f"AT+SEND={device_id},{hex_data}")
        
        return response and "OK" in response

class DeviceRegistry:
    """Device registry and management."""
    
    def __init__(self):
        self.devices: Dict[str, DeviceInfo] = {}
        self.sensor_readings: Dict[str, List[SensorReading]] = {}
        self.device_commands: Dict[str, List[DeviceCommand]] = {}
    
    def register_device(self, device: DeviceInfo) -> bool:
        """Register a new IoT device."""
        try:
            self.devices[device.device_id] = device
            self.sensor_readings[device.device_id] = []
            self.device_commands[device.device_id] = []
            
            logger.info(f"Device registered: {device.device_id} ({device.name})")
            return True
        except Exception as e:
            logger.error(f"Error registering device: {str(e)}")
            return False
    
    def unregister_device(self, device_id: str) -> bool:
        """Unregister an IoT device."""
        if device_id in self.devices:
            del self.devices[device_id]
            del self.sensor_readings[device_id]
            del self.device_commands[device_id]
            
            logger.info(f"Device unregistered: {device_id}")
            return True
        return False
    
    def get_device(self, device_id: str) -> Optional[DeviceInfo]:
        """Get device information."""
        return self.devices.get(device_id)
    
    def list_devices(self, status: Optional[DeviceStatus] = None) -> List[DeviceInfo]:
        """List devices, optionally filtered by status."""
        devices = list(self.devices.values())
        
        if status:
            devices = [d for d in devices if d.status == status]
        
        return devices
    
    def update_device_status(self, device_id: str, status: DeviceStatus) -> bool:
        """Update device status."""
        if device_id in self.devices:
            self.devices[device_id].status = status
            self.devices[device_id].last_seen = datetime.now()
            return True
        return False
    
    def add_sensor_reading(self, reading: SensorReading) -> bool:
        """Add sensor reading."""
        if reading.device_id in self.sensor_readings:
            self.sensor_readings[reading.device_id].append(reading)
            
            # Keep only last 1000 readings per device
            if len(self.sensor_readings[reading.device_id]) > 1000:
                self.sensor_readings[reading.device_id] = self.sensor_readings[reading.device_id][-1000:]
            
            return True
        return False
    
    def get_sensor_readings(self, device_id: str, limit: int = 100) -> List[SensorReading]:
        """Get recent sensor readings for device."""
        readings = self.sensor_readings.get(device_id, [])
        return readings[-limit:]
    
    def add_device_command(self, command: DeviceCommand) -> bool:
        """Add device command."""
        if command.device_id in self.device_commands:
            self.device_commands[command.device_id].append(command)
            return True
        return False
    
    def get_device_commands(self, device_id: str, status: Optional[str] = None) -> List[DeviceCommand]:
        """Get device commands."""
        commands = self.device_commands.get(device_id, [])
        
        if status:
            commands = [c for c in commands if c.status == status]
        
        return commands

class IoTDeviceManager:
    """
    Comprehensive IoT Device Manager.
    Supports MQTT, CoAP, LoRaWAN device connectivity and management.
    """
    
    def __init__(self):
        self.device_registry = DeviceRegistry()
        self.mqtt_manager = MQTTManager()
        self.coap_manager = CoAPManager()
        self.lorawan_manager = LoRaWANManager()
        
        self.protocol_managers = {
            Protocol.MQTT: self.mqtt_manager,
            Protocol.COAP: self.coap_manager,
            Protocol.LORAWAN: self.lorawan_manager
        }
        
        self.stats = {
            'devices_registered': 0,
            'messages_sent': 0,
            'messages_received': 0,
            'commands_executed': 0,
            'sensor_readings_collected': 0
        }
        
        self.running = False
        self._setup_mqtt_callbacks()
    
    def _setup_mqtt_callbacks(self):
        """Setup MQTT message callbacks."""
        def on_sensor_data(topic: str, payload: str):
            try:
                # Parse sensor data from MQTT message
                # Expected format: devices/{device_id}/sensors/{sensor_type}
                parts = topic.split('/')
                if len(parts) >= 4:
                    device_id = parts[1]
                    sensor_type = parts[3]
                    
                    data = json.loads(payload)
                    reading = SensorReading(
                        device_id=device_id,
                        sensor_type=sensor_type,
                        value=float(data.get('value', 0)),
                        unit=data.get('unit', ''),
                        timestamp=datetime.now(),
                        quality=data.get('quality', 1.0)
                    )
                    
                    self.device_registry.add_sensor_reading(reading)
                    self.device_registry.update_device_status(device_id, DeviceStatus.ONLINE)
                    self.stats['sensor_readings_collected'] += 1
                    
            except Exception as e:
                logger.error(f"Error processing sensor data: {str(e)}")
        
        def on_device_status(topic: str, payload: str):
            try:
                # Parse device status from MQTT message
                # Expected format: devices/{device_id}/status
                parts = topic.split('/')
                if len(parts) >= 3:
                    device_id = parts[1]
                    
                    data = json.loads(payload)
                    status = DeviceStatus(data.get('status', 'offline'))
                    
                    self.device_registry.update_device_status(device_id, status)
                    
            except Exception as e:
                logger.error(f"Error processing device status: {str(e)}")
        
        # Register callbacks
        self.mqtt_manager.subscribe("devices/+/sensors/+", on_sensor_data)
        self.mqtt_manager.subscribe("devices/+/status", on_device_status)
    
    async def initialize(self):
        """Initialize IoT device manager."""
        logger.info("Initializing IoT Device Manager...")
        
        # Initialize protocol managers
        self.mqtt_manager.connect()
        await self.coap_manager.initialize()
        self.lorawan_manager.connect()
        
        self.running = True
        logger.info("IoT Device Manager initialized")
    
    async def shutdown(self):
        """Shutdown IoT device manager."""
        logger.info("Shutting down IoT Device Manager...")
        
        self.running = False
        
        # Disconnect protocol managers
        self.mqtt_manager.disconnect()
        self.lorawan_manager.disconnect()
        
        logger.info("IoT Device Manager shutdown complete")
    
    async def register_device(self, device: DeviceInfo) -> bool:
        """Register a new IoT device."""
        success = self.device_registry.register_device(device)
        
        if success:
            self.stats['devices_registered'] += 1
            
            # Subscribe to device-specific topics for MQTT devices
            if device.protocol == Protocol.MQTT:
                self.mqtt_manager.subscribe(f"devices/{device.device_id}/+", self._on_device_message)
        
        return success
    
    async def discover_devices(self, protocol: Protocol, timeout: int = 10) -> List[DeviceInfo]:
        """Discover devices on the network."""
        logger.info(f"Discovering {protocol.value} devices...")
        
        discovered_devices = []
        
        if protocol == Protocol.MQTT:
            # Mock MQTT device discovery
            for i in range(3):
                device = DeviceInfo(
                    device_id=f"mqtt_device_{i+1}",
                    name=f"MQTT Sensor {i+1}",
                    device_type="sensor",
                    protocol=Protocol.MQTT,
                    status=DeviceStatus.ONLINE,
                    ip_address=f"192.168.1.{100+i}",
                    firmware_version="1.0.0"
                )
                discovered_devices.append(device)
        
        elif protocol == Protocol.COAP:
            # Mock CoAP device discovery
            for i in range(2):
                device = DeviceInfo(
                    device_id=f"coap_device_{i+1}",
                    name=f"CoAP Actuator {i+1}",
                    device_type="actuator",
                    protocol=Protocol.COAP,
                    status=DeviceStatus.ONLINE,
                    ip_address=f"192.168.1.{110+i}",
                    firmware_version="2.0.0"
                )
                discovered_devices.append(device)
        
        elif protocol == Protocol.LORAWAN:
            # Mock LoRaWAN device discovery
            device = DeviceInfo(
                device_id="lorawan_device_1",
                name="LoRaWAN Environmental Sensor",
                device_type="environmental_sensor",
                protocol=Protocol.LORAWAN,
                status=DeviceStatus.ONLINE,
                firmware_version="1.5.0",
                metadata={"rssi": -85, "snr": 7.5}
            )
            discovered_devices.append(device)
        
        logger.info(f"Discovered {len(discovered_devices)} {protocol.value} devices")
        return discovered_devices
    
    async def send_command(self, device_id: str, command: str, 
                          parameters: Dict[str, Any] = None) -> bool:
        """Send command to IoT device."""
        device = self.device_registry.get_device(device_id)
        if not device:
            logger.error(f"Device not found: {device_id}")
            return False
        
        command_obj = DeviceCommand(
            device_id=device_id,
            command=command,
            parameters=parameters or {},
            timestamp=datetime.now()
        )
        
        self.device_registry.add_device_command(command_obj)
        
        try:
            if device.protocol == Protocol.MQTT:
                topic = f"devices/{device_id}/commands"
                payload = json.dumps({
                    "command": command,
                    "parameters": parameters
                })
                
                success = self.mqtt_manager.publish(topic, payload)
            
            elif device.protocol == Protocol.COAP:
                uri = f"coap://{device.ip_address}/commands"
                payload = json.dumps({
                    "command": command,
                    "parameters": parameters
                })
                
                response = await self.coap_manager.send_post_request(uri, payload)
                success = response is not None
            
            elif device.protocol == Protocol.LORAWAN:
                data = json.dumps({
                    "command": command,
                    "parameters": parameters
                })
                
                success = self.lorawan_manager.send_data(device_id, data)
            
            else:
                logger.warning(f"Unsupported protocol for command: {device.protocol}")
                success = False
            
            if success:
                command_obj.status = "sent"
                self.stats['commands_executed'] += 1
                self.stats['messages_sent'] += 1
            else:
                command_obj.status = "failed"
            
            return success
            
        except Exception as e:
            logger.error(f"Error sending command to device {device_id}: {str(e)}")
            command_obj.status = "failed"
            return False
    
    async def collect_sensor_data(self, device_id: str, sensor_type: str) -> Optional[SensorReading]:
        """Collect sensor data from device."""
        device = self.device_registry.get_device(device_id)
        if not device:
            return None
        
        try:
            if device.protocol == Protocol.MQTT:
                # For MQTT, we rely on devices pushing data
                # Return the latest reading
                readings = self.device_registry.get_sensor_readings(device_id, limit=1)
                return readings[0] if readings else None
            
            elif device.protocol == Protocol.COAP:
                uri = f"coap://{device.ip_address}/sensors/{sensor_type}"
                response = await self.coap_manager.send_get_request(uri)
                
                if response:
                    data = json.loads(response)
                    reading = SensorReading(
                        device_id=device_id,
                        sensor_type=sensor_type,
                        value=float(data.get('value', 0)),
                        unit=data.get('unit', ''),
                        timestamp=datetime.now(),
                        quality=data.get('quality', 1.0)
                    )
                    
                    self.device_registry.add_sensor_reading(reading)
                    self.stats['sensor_readings_collected'] += 1
                    
                    return reading
            
            elif device.protocol == Protocol.LORAWAN:
                # Mock LoRaWAN sensor reading
                reading = SensorReading(
                    device_id=device_id,
                    sensor_type=sensor_type,
                    value=random.uniform(20, 30),  # Mock temperature
                    unit="°C",
                    timestamp=datetime.now(),
                    quality=0.95
                )
                
                self.device_registry.add_sensor_reading(reading)
                self.stats['sensor_readings_collected'] += 1
                
                return reading
            
        except Exception as e:
            logger.error(f"Error collecting sensor data from {device_id}: {str(e)}")
        
        return None
    
    def _on_device_message(self, topic: str, payload: str):
        """Handle device messages."""
        self.stats['messages_received'] += 1
        logger.info(f"Received device message on {topic}")
    
    def get_device_status(self, device_id: str) -> Dict[str, Any]:
        """Get comprehensive device status."""
        device = self.device_registry.get_device(device_id)
        if not device:
            return {}
        
        recent_readings = self.device_registry.get_sensor_readings(device_id, limit=10)
        pending_commands = self.device_registry.get_device_commands(device_id, status="pending")
        
        return {
            "device_info": {
                "id": device.device_id,
                "name": device.name,
                "type": device.device_type,
                "protocol": device.protocol.value,
                "status": device.status.value,
                "ip_address": device.ip_address,
                "firmware_version": device.firmware_version,
                "last_seen": device.last_seen.isoformat() if device.last_seen else None
            },
            "recent_readings": len(recent_readings),
            "pending_commands": len(pending_commands),
            "metadata": device.metadata
        }
    
    def get_network_topology(self) -> Dict[str, Any]:
        """Get network topology information."""
        devices_by_protocol = {}
        
        for device in self.device_registry.list_devices():
            protocol = device.protocol.value
            if protocol not in devices_by_protocol:
                devices_by_protocol[protocol] = []
            
            devices_by_protocol[protocol].append({
                "id": device.device_id,
                "name": device.name,
                "status": device.status.value,
                "ip_address": device.ip_address
            })
        
        return {
            "total_devices": len(self.device_registry.devices),
            "devices_by_protocol": devices_by_protocol,
            "protocol_status": {
                "mqtt": self.mqtt_manager.connected,
                "coap": self.coap_manager.context is not None,
                "lorawan": self.lorawan_manager.connected
            }
        }
    
    def get_processing_stats(self) -> Dict[str, Any]:
        """Get processing statistics."""
        return self.stats.copy()

# Example usage and testing
async def main():
    """Example usage of the IoT Device Manager."""
    print("=== IoT Device Manager Demo ===")
    
    # Initialize IoT device manager
    iot_manager = IoTDeviceManager()
    await iot_manager.initialize()
    
    # Test 1: Device discovery
    print("\n1. Testing device discovery...")
    
    for protocol in [Protocol.MQTT, Protocol.COAP, Protocol.LORAWAN]:
        devices = await iot_manager.discover_devices(protocol)
        print(f"Discovered {len(devices)} {protocol.value} devices:")
        for device in devices:
            print(f"  - {device.name} ({device.device_id})")
            
            # Register discovered device
            await iot_manager.register_device(device)
    
    # Test 2: Device status
    print("\n2. Device status:")
    all_devices = iot_manager.device_registry.list_devices()
    for device in all_devices[:3]:  # Show first 3 devices
        status = iot_manager.get_device_status(device.device_id)
        print(f"  {device.name}: {device.status.value}")
    
    # Test 3: Send commands
    print("\n3. Testing device commands...")
    if all_devices:
        device = all_devices[0]
        success = await iot_manager.send_command(
            device.device_id,
            "set_temperature",
            {"temperature": 22.5}
        )
        print(f"Command sent to {device.name}: {'Success' if success else 'Failed'}")
    
    # Test 4: Collect sensor data
    print("\n4. Testing sensor data collection...")
    if all_devices:
        device = all_devices[0]
        reading = await iot_manager.collect_sensor_data(device.device_id, "temperature")
        if reading:
            print(f"Sensor reading from {device.name}: {reading.value} {reading.unit}")
    
    # Test 5: Network topology
    print("\n5. Network topology:")
    topology = iot_manager.get_network_topology()
    print(f"Total devices: {topology['total_devices']}")
    for protocol, devices in topology['devices_by_protocol'].items():
        print(f"  {protocol}: {len(devices)} devices")
    
    # Test 6: Processing statistics
    print("\n6. Processing statistics:")
    stats = iot_manager.get_processing_stats()
    for key, value in stats.items():
        print(f"  {key}: {value}")
    
    # Test 7: Available protocols
    print("\n7. Available protocol support:")
    print(f"  MQTT: {MQTT_AVAILABLE}")
    print(f"  CoAP: {COAP_AVAILABLE}")
    print(f"  Serial (LoRaWAN): {SERIAL_AVAILABLE}")
    
    # Cleanup
    await iot_manager.shutdown()
    
    print("\n=== IoT Device Manager Demo Complete ===")

if __name__ == "__main__":
    asyncio.run(main()) 