"""
Browser Bindings for TuskLang
Provides browser-specific functionality for TuskLang
"""

from typing import Dict, Any, Optional, Callable
import json

class BrowserBindings:
    """Browser-specific bindings for TuskLang"""
    
    def __init__(self):
        self.is_browser = False
        self.dom_ready = False
        
        # Check if running in browser
        try:
            import js
            self.is_browser = True
        except ImportError:
            pass
    
    def get_element(self, selector: str) -> Optional[Any]:
        """Get DOM element by selector"""
        if not self.is_browser:
            return None
        
        try:
            import js
            return js.document.querySelector(selector)
        except Exception:
            return None
    
    def set_inner_html(self, selector: str, html: str) -> bool:
        """Set inner HTML of element"""
        if not self.is_browser:
            return False
        
        try:
            import js
            element = js.document.querySelector(selector)
            if element:
                element.innerHTML = html
                return True
        except Exception:
            pass
        
        return False
    
    def add_event_listener(self, selector: str, event: str, handler: Callable) -> bool:
        """Add event listener to element"""
        if not self.is_browser:
            return False
        
        try:
            import js
            element = js.document.querySelector(selector)
            if element:
                element.addEventListener(event, handler)
                return True
        except Exception:
            pass
        
        return False
    
    def console_log(self, message: str) -> None:
        """Log message to browser console"""
        if not self.is_browser:
            print(message)
            return
        
        try:
            import js
            js.console.log(message)
        except Exception:
            print(message) 