"""
Kubernetes Integration for TuskLang Python SDK
Provides Kubernetes operator functionality with CRDs, controllers, and Helm charts
"""

import asyncio
import json
import yaml
import base64
import hashlib
from typing import Any, Dict, List, Optional, Callable
from datetime import datetime, timedelta
import logging
import os
import tempfile

try:
    from kubernetes import client, config, watch
    from kubernetes.client.rest import ApiException
    KUBERNETES_AVAILABLE = True
except ImportError:
    KUBERNETES_AVAILABLE = False
    client = None
    config = None
    watch = None

try:
    import kopf
    KOPF_AVAILABLE = True
except ImportError:
    KOPF_AVAILABLE = False


class KubernetesIntegration:
    """
    Main Kubernetes integration class for TuskLang Python SDK.
    Implements the Operator pattern with Custom Resource Definitions.
    """
    
    def __init__(self, config_dict: Optional[Dict[str, Any]] = None):
        if not KUBERNETES_AVAILABLE:
            raise ImportError("kubernetes package required for Kubernetes integration")
        
        self.config = config_dict or {}
        self.namespace = self.config.get('namespace', 'tusk-lang')
        self.operator_name = self.config.get('operator_name', 'tusklang-operator')
        
        # Kubernetes clients
        self.api_client = None
        self.custom_objects_api = None
        self.apps_v1_api = None
        self.core_v1_api = None
        self.rbac_v1_api = None
        
        # Operator state
        self.is_running = False
        self.watched_resources = {}
        self.resource_handlers = {}
        self.event_queue = asyncio.Queue()
        
        # Statistics
        self.stats = {
            'resources_processed': 0,
            'reconciliation_loops': 0,
            'errors': 0,
            'last_reconciliation': None,
            'active_resources': 0
        }
        
        # Initialize logging
        self.logger = logging.getLogger(__name__)
        
        # Initialize Kubernetes client
        self._initialize_kubernetes()
        
        # Create CRDs
        self._ensure_custom_resources()
    
    def _initialize_kubernetes(self):
        """Initialize Kubernetes client configuration"""
        try:
            # Try in-cluster config first
            try:
                config.load_incluster_config()
                self.logger.info("Loaded in-cluster Kubernetes configuration")
            except config.ConfigException:
                # Fall back to local kubeconfig
                config.load_kube_config()
                self.logger.info("Loaded local Kubernetes configuration")
            
            # Initialize API clients
            self.api_client = client.ApiClient()
            self.custom_objects_api = client.CustomObjectsApi()
            self.apps_v1_api = client.AppsV1Api()
            self.core_v1_api = client.CoreV1Api()
            self.rbac_v1_api = client.RbacAuthorizationV1Api()
            
            # Test connection
            version = client.VersionApi().get_code()
            self.logger.info(f"Connected to Kubernetes cluster version: {version.git_version}")
            
        except Exception as e:
            self.logger.error(f"Kubernetes client initialization failed: {e}")
            raise
    
    def _ensure_custom_resources(self):
        """Ensure TuskLang Custom Resource Definitions exist"""
        try:
            # Define TuskLang CRDs
            crds = self._get_tusklang_crds()
            
            extensions_v1_api = client.ApiextensionsV1Api()
            
            for crd in crds:
                try:
                    # Check if CRD already exists
                    existing_crd = extensions_v1_api.read_custom_resource_definition(
                        name=crd['metadata']['name']
                    )
                    self.logger.info(f"CRD {crd['metadata']['name']} already exists")
                except ApiException as e:
                    if e.status == 404:
                        # Create CRD
                        extensions_v1_api.create_custom_resource_definition(body=crd)
                        self.logger.info(f"Created CRD: {crd['metadata']['name']}")
                    else:
                        raise
            
        except Exception as e:
            self.logger.error(f"CRD creation failed: {e}")
            raise
    
    def _get_tusklang_crds(self) -> List[Dict[str, Any]]:
        """Get TuskLang Custom Resource Definitions"""
        return [
            # TuskLang Configuration CRD
            {
                "apiVersion": "apiextensions.k8s.io/v1",
                "kind": "CustomResourceDefinition",
                "metadata": {
                    "name": "tusklangconfigs.tusklang.io"
                },
                "spec": {
                    "group": "tusklang.io",
                    "versions": [{
                        "name": "v1",
                        "served": True,
                        "storage": True,
                        "schema": {
                            "openAPIV3Schema": {
                                "type": "object",
                                "properties": {
                                    "spec": {
                                        "type": "object",
                                        "properties": {
                                            "config": {
                                                "type": "string",
                                                "description": "TuskLang configuration content"
                                            },
                                            "variables": {
                                                "type": "object",
                                                "additionalProperties": {"type": "string"},
                                                "description": "Configuration variables"
                                            },
                                            "replicas": {
                                                "type": "integer",
                                                "minimum": 1,
                                                "default": 1,
                                                "description": "Number of replicas"
                                            },
                                            "resources": {
                                                "type": "object",
                                                "description": "Resource requirements"
                                            }
                                        },
                                        "required": ["config"]
                                    },
                                    "status": {
                                        "type": "object",
                                        "properties": {
                                            "phase": {
                                                "type": "string",
                                                "enum": ["Pending", "Running", "Failed", "Succeeded"]
                                            },
                                            "message": {"type": "string"},
                                            "lastReconciliation": {"type": "string"},
                                            "activeReplicas": {"type": "integer"}
                                        }
                                    }
                                }
                            }
                        }
                    }],
                    "scope": "Namespaced",
                    "names": {
                        "plural": "tusklangconfigs",
                        "singular": "tusklangconfig",
                        "kind": "TuskLangConfig",
                        "shortNames": ["tusk", "tuskconfig"]
                    }
                }
            },
            # TuskLang Operator CRD
            {
                "apiVersion": "apiextensions.k8s.io/v1", 
                "kind": "CustomResourceDefinition",
                "metadata": {
                    "name": "tusklangoperators.tusklang.io"
                },
                "spec": {
                    "group": "tusklang.io",
                    "versions": [{
                        "name": "v1",
                        "served": True,
                        "storage": True,
                        "schema": {
                            "openAPIV3Schema": {
                                "type": "object",
                                "properties": {
                                    "spec": {
                                        "type": "object",
                                        "properties": {
                                            "operator": {
                                                "type": "string",
                                                "description": "TuskLang operator expression"
                                            },
                                            "schedule": {
                                                "type": "string",
                                                "description": "Cron schedule for operator execution"
                                            },
                                            "timeout": {
                                                "type": "string",
                                                "default": "300s",
                                                "description": "Operation timeout"
                                            },
                                            "retryPolicy": {
                                                "type": "object",
                                                "properties": {
                                                    "maxRetries": {"type": "integer", "default": 3},
                                                    "backoffLimit": {"type": "string", "default": "30s"}
                                                }
                                            }
                                        },
                                        "required": ["operator"]
                                    },
                                    "status": {
                                        "type": "object",
                                        "properties": {
                                            "phase": {
                                                "type": "string",
                                                "enum": ["Pending", "Running", "Completed", "Failed"]
                                            },
                                            "lastExecution": {"type": "string"},
                                            "nextExecution": {"type": "string"},
                                            "executionCount": {"type": "integer"},
                                            "errorCount": {"type": "integer"},
                                            "lastResult": {"type": "string"}
                                        }
                                    }
                                }
                            }
                        }
                    }],
                    "scope": "Namespaced",
                    "names": {
                        "plural": "tusklangoperators",
                        "singular": "tusklangoperator", 
                        "kind": "TuskLangOperator",
                        "shortNames": ["tuskop", "operator"]
                    }
                }
            }
        ]
    
    async def start_operator(self):
        """Start the Kubernetes operator"""
        if self.is_running:
            return
        
        self.is_running = True
        self.logger.info("Starting TuskLang Kubernetes operator")
        
        try:
            # Create operator deployment if needed
            await self._ensure_operator_deployment()
            
            # Start watching resources
            asyncio.create_task(self._watch_tusklang_configs())
            asyncio.create_task(self._watch_tusklang_operators())
            
            # Start reconciliation loop
            asyncio.create_task(self._reconciliation_loop())
            
            self.logger.info("TuskLang operator started successfully")
            
        except Exception as e:
            self.is_running = False
            self.logger.error(f"Operator startup failed: {e}")
            raise
    
    async def _ensure_operator_deployment(self):
        """Ensure operator deployment exists"""
        deployment_name = f"{self.operator_name}-deployment"
        
        try:
            # Check if deployment exists
            self.apps_v1_api.read_namespaced_deployment(
                name=deployment_name,
                namespace=self.namespace
            )
            self.logger.info(f"Operator deployment {deployment_name} already exists")
            
        except ApiException as e:
            if e.status == 404:
                # Create deployment
                deployment = self._create_operator_deployment()
                self.apps_v1_api.create_namespaced_deployment(
                    namespace=self.namespace,
                    body=deployment
                )
                self.logger.info(f"Created operator deployment: {deployment_name}")
            else:
                raise
    
    def _create_operator_deployment(self) -> Dict[str, Any]:
        """Create operator deployment manifest"""
        return {
            "apiVersion": "apps/v1",
            "kind": "Deployment",
            "metadata": {
                "name": f"{self.operator_name}-deployment",
                "namespace": self.namespace,
                "labels": {
                    "app": self.operator_name,
                    "component": "operator"
                }
            },
            "spec": {
                "replicas": 1,
                "selector": {
                    "matchLabels": {
                        "app": self.operator_name
                    }
                },
                "template": {
                    "metadata": {
                        "labels": {
                            "app": self.operator_name
                        }
                    },
                    "spec": {
                        "serviceAccountName": f"{self.operator_name}-sa",
                        "containers": [{
                            "name": "operator",
                            "image": "tusklang/python-operator:latest",
                            "env": [
                                {
                                    "name": "NAMESPACE",
                                    "valueFrom": {
                                        "fieldRef": {
                                            "fieldPath": "metadata.namespace"
                                        }
                                    }
                                }
                            ],
                            "resources": {
                                "requests": {
                                    "memory": "256Mi",
                                    "cpu": "100m"
                                },
                                "limits": {
                                    "memory": "512Mi", 
                                    "cpu": "500m"
                                }
                            },
                            "livenessProbe": {
                                "httpGet": {
                                    "path": "/healthz",
                                    "port": 8080
                                },
                                "initialDelaySeconds": 30,
                                "periodSeconds": 10
                            }
                        }]
                    }
                }
            }
        }
    
    async def _watch_tusklang_configs(self):
        """Watch TuskLangConfig resources"""
        w = watch.Watch()
        
        while self.is_running:
            try:
                stream = w.stream(
                    self.custom_objects_api.list_namespaced_custom_object,
                    group="tusklang.io",
                    version="v1",
                    namespace=self.namespace,
                    plural="tusklangconfigs",
                    timeout_seconds=60
                )
                
                async for event in stream:
                    await self._handle_config_event(event)
                    
            except Exception as e:
                self.logger.error(f"Config watch error: {e}")
                await asyncio.sleep(5)
    
    async def _watch_tusklang_operators(self):
        """Watch TuskLangOperator resources"""
        w = watch.Watch()
        
        while self.is_running:
            try:
                stream = w.stream(
                    self.custom_objects_api.list_namespaced_custom_object,
                    group="tusklang.io",
                    version="v1", 
                    namespace=self.namespace,
                    plural="tusklangoperators",
                    timeout_seconds=60
                )
                
                async for event in stream:
                    await self._handle_operator_event(event)
                    
            except Exception as e:
                self.logger.error(f"Operator watch error: {e}")
                await asyncio.sleep(5)
    
    async def _handle_config_event(self, event):
        """Handle TuskLangConfig resource events"""
        event_type = event['type']
        resource = event['object']
        
        try:
            name = resource['metadata']['name']
            namespace = resource['metadata']['namespace']
            
            self.logger.info(f"Config event: {event_type} for {namespace}/{name}")
            
            if event_type in ['ADDED', 'MODIFIED']:
                await self._reconcile_config(resource)
            elif event_type == 'DELETED':
                await self._cleanup_config(resource)
                
            self.stats['resources_processed'] += 1
            
        except Exception as e:
            self.stats['errors'] += 1
            self.logger.error(f"Config event handling failed: {e}")
    
    async def _handle_operator_event(self, event):
        """Handle TuskLangOperator resource events"""
        event_type = event['type']
        resource = event['object']
        
        try:
            name = resource['metadata']['name']
            namespace = resource['metadata']['namespace']
            
            self.logger.info(f"Operator event: {event_type} for {namespace}/{name}")
            
            if event_type in ['ADDED', 'MODIFIED']:
                await self._reconcile_operator(resource)
            elif event_type == 'DELETED':
                await self._cleanup_operator(resource)
                
            self.stats['resources_processed'] += 1
            
        except Exception as e:
            self.stats['errors'] += 1
            self.logger.error(f"Operator event handling failed: {e}")
    
    async def _reconcile_config(self, resource: Dict[str, Any]):
        """Reconcile TuskLangConfig resource"""
        try:
            name = resource['metadata']['name']
            namespace = resource['metadata']['namespace']
            spec = resource['spec']
            
            # Extract configuration
            config_content = spec.get('config', '')
            variables = spec.get('variables', {})
            replicas = spec.get('replicas', 1)
            
            # Create ConfigMap for TuskLang config
            config_map = self._create_config_map(name, namespace, config_content, variables)
            
            try:
                self.core_v1_api.create_namespaced_config_map(
                    namespace=namespace,
                    body=config_map
                )
            except ApiException as e:
                if e.status == 409:  # Already exists
                    self.core_v1_api.patch_namespaced_config_map(
                        name=f"{name}-config",
                        namespace=namespace,
                        body=config_map
                    )
            
            # Create Deployment for TuskLang execution
            deployment = self._create_config_deployment(name, namespace, spec)
            
            try:
                self.apps_v1_api.create_namespaced_deployment(
                    namespace=namespace,
                    body=deployment
                )
            except ApiException as e:
                if e.status == 409:  # Already exists
                    self.apps_v1_api.patch_namespaced_deployment(
                        name=f"{name}-deployment",
                        namespace=namespace,
                        body=deployment
                    )
            
            # Update status
            await self._update_config_status(name, namespace, "Running", "Configuration deployed successfully")
            
        except Exception as e:
            await self._update_config_status(name, namespace, "Failed", str(e))
            raise
    
    def _create_config_map(self, name: str, namespace: str, config: str, variables: Dict) -> Dict:
        """Create ConfigMap for TuskLang configuration"""
        return {
            "apiVersion": "v1",
            "kind": "ConfigMap",
            "metadata": {
                "name": f"{name}-config",
                "namespace": namespace,
                "labels": {
                    "app": name,
                    "component": "config"
                }
            },
            "data": {
                "config.tsk": config,
                "variables.json": json.dumps(variables)
            }
        }
    
    def _create_config_deployment(self, name: str, namespace: str, spec: Dict) -> Dict:
        """Create Deployment for TuskLang configuration execution"""
        replicas = spec.get('replicas', 1)
        resources = spec.get('resources', {})
        
        return {
            "apiVersion": "apps/v1",
            "kind": "Deployment", 
            "metadata": {
                "name": f"{name}-deployment",
                "namespace": namespace,
                "labels": {
                    "app": name,
                    "component": "executor"
                }
            },
            "spec": {
                "replicas": replicas,
                "selector": {
                    "matchLabels": {
                        "app": name
                    }
                },
                "template": {
                    "metadata": {
                        "labels": {
                            "app": name
                        }
                    },
                    "spec": {
                        "containers": [{
                            "name": "tusklang-executor",
                            "image": "tusklang/python-executor:latest",
                            "env": [
                                {
                                    "name": "CONFIG_FILE",
                                    "value": "/config/config.tsk"
                                },
                                {
                                    "name": "VARIABLES_FILE", 
                                    "value": "/config/variables.json"
                                }
                            ],
                            "volumeMounts": [{
                                "name": "config",
                                "mountPath": "/config"
                            }],
                            "resources": resources
                        }],
                        "volumes": [{
                            "name": "config",
                            "configMap": {
                                "name": f"{name}-config"
                            }
                        }]
                    }
                }
            }
        }
    
    async def _reconcile_operator(self, resource: Dict[str, Any]):
        """Reconcile TuskLangOperator resource"""
        try:
            name = resource['metadata']['name']
            namespace = resource['metadata']['namespace']
            spec = resource['spec']
            
            operator_expr = spec.get('operator', '')
            schedule = spec.get('schedule')
            timeout = spec.get('timeout', '300s')
            
            if schedule:
                # Create CronJob for scheduled execution
                cronjob = self._create_operator_cronjob(name, namespace, spec)
                
                batch_v1_api = client.BatchV1Api()
                try:
                    batch_v1_api.create_namespaced_cron_job(
                        namespace=namespace,
                        body=cronjob
                    )
                except ApiException as e:
                    if e.status == 409:  # Already exists
                        batch_v1_api.patch_namespaced_cron_job(
                            name=f"{name}-cronjob",
                            namespace=namespace,
                            body=cronjob
                        )
            else:
                # Execute immediately as Job
                job = self._create_operator_job(name, namespace, spec)
                
                batch_v1_api = client.BatchV1Api()
                batch_v1_api.create_namespaced_job(
                    namespace=namespace,
                    body=job
                )
            
            # Update status
            await self._update_operator_status(name, namespace, "Running", "Operator scheduled successfully")
            
        except Exception as e:
            await self._update_operator_status(name, namespace, "Failed", str(e))
            raise
    
    def _create_operator_cronjob(self, name: str, namespace: str, spec: Dict) -> Dict:
        """Create CronJob for scheduled TuskLang operator"""
        return {
            "apiVersion": "batch/v1",
            "kind": "CronJob",
            "metadata": {
                "name": f"{name}-cronjob",
                "namespace": namespace
            },
            "spec": {
                "schedule": spec.get('schedule'),
                "jobTemplate": {
                    "spec": {
                        "template": {
                            "spec": {
                                "containers": [{
                                    "name": "tusklang-operator",
                                    "image": "tusklang/python-operator-executor:latest",
                                    "env": [
                                        {
                                            "name": "OPERATOR_EXPRESSION",
                                            "value": spec.get('operator', '')
                                        },
                                        {
                                            "name": "TIMEOUT",
                                            "value": spec.get('timeout', '300s')
                                        }
                                    ]
                                }],
                                "restartPolicy": "OnFailure"
                            }
                        }
                    }
                }
            }
        }
    
    def _create_operator_job(self, name: str, namespace: str, spec: Dict) -> Dict:
        """Create Job for immediate TuskLang operator execution"""
        return {
            "apiVersion": "batch/v1",
            "kind": "Job",
            "metadata": {
                "name": f"{name}-job-{int(datetime.now().timestamp())}",
                "namespace": namespace
            },
            "spec": {
                "template": {
                    "spec": {
                        "containers": [{
                            "name": "tusklang-operator",
                            "image": "tusklang/python-operator-executor:latest", 
                            "env": [
                                {
                                    "name": "OPERATOR_EXPRESSION",
                                    "value": spec.get('operator', '')
                                },
                                {
                                    "name": "TIMEOUT",
                                    "value": spec.get('timeout', '300s')
                                }
                            ]
                        }],
                        "restartPolicy": "OnFailure"
                    }
                }
            }
        }
    
    async def _reconciliation_loop(self):
        """Main reconciliation loop"""
        while self.is_running:
            try:
                self.stats['reconciliation_loops'] += 1
                self.stats['last_reconciliation'] = datetime.now().isoformat()
                
                # Update active resources count
                configs = self.custom_objects_api.list_namespaced_custom_object(
                    group="tusklang.io",
                    version="v1",
                    namespace=self.namespace,
                    plural="tusklangconfigs"
                )
                
                operators = self.custom_objects_api.list_namespaced_custom_object(
                    group="tusklang.io",
                    version="v1", 
                    namespace=self.namespace,
                    plural="tusklangoperators"
                )
                
                self.stats['active_resources'] = len(configs['items']) + len(operators['items'])
                
                # Sleep for reconciliation interval
                await asyncio.sleep(30)
                
            except Exception as e:
                self.stats['errors'] += 1
                self.logger.error(f"Reconciliation loop error: {e}")
                await asyncio.sleep(10)
    
    async def _update_config_status(self, name: str, namespace: str, phase: str, message: str):
        """Update TuskLangConfig status"""
        try:
            status_patch = {
                "status": {
                    "phase": phase,
                    "message": message,
                    "lastReconciliation": datetime.now().isoformat()
                }
            }
            
            self.custom_objects_api.patch_namespaced_custom_object_status(
                group="tusklang.io",
                version="v1",
                namespace=namespace,
                plural="tusklangconfigs",
                name=name,
                body=status_patch
            )
        except Exception as e:
            self.logger.error(f"Status update failed: {e}")
    
    async def _update_operator_status(self, name: str, namespace: str, phase: str, message: str):
        """Update TuskLangOperator status"""
        try:
            status_patch = {
                "status": {
                    "phase": phase,
                    "message": message,
                    "lastExecution": datetime.now().isoformat()
                }
            }
            
            self.custom_objects_api.patch_namespaced_custom_object_status(
                group="tusklang.io",
                version="v1",
                namespace=namespace,
                plural="tusklangoperators", 
                name=name,
                body=status_patch
            )
        except Exception as e:
            self.logger.error(f"Status update failed: {e}")
    
    async def _cleanup_config(self, resource: Dict[str, Any]):
        """Clean up resources for deleted TuskLangConfig"""
        name = resource['metadata']['name']
        namespace = resource['metadata']['namespace']
        
        try:
            # Delete ConfigMap
            self.core_v1_api.delete_namespaced_config_map(
                name=f"{name}-config",
                namespace=namespace
            )
            
            # Delete Deployment
            self.apps_v1_api.delete_namespaced_deployment(
                name=f"{name}-deployment",
                namespace=namespace
            )
            
            self.logger.info(f"Cleaned up resources for config {namespace}/{name}")
            
        except ApiException as e:
            if e.status != 404:  # Not found is OK
                self.logger.error(f"Cleanup failed for config {namespace}/{name}: {e}")
    
    async def _cleanup_operator(self, resource: Dict[str, Any]):
        """Clean up resources for deleted TuskLangOperator"""
        name = resource['metadata']['name']
        namespace = resource['metadata']['namespace']
        
        try:
            batch_v1_api = client.BatchV1Api()
            
            # Try to delete CronJob
            try:
                batch_v1_api.delete_namespaced_cron_job(
                    name=f"{name}-cronjob",
                    namespace=namespace
                )
            except ApiException as e:
                if e.status != 404:
                    raise
            
            self.logger.info(f"Cleaned up resources for operator {namespace}/{name}")
            
        except ApiException as e:
            if e.status != 404:
                self.logger.error(f"Cleanup failed for operator {namespace}/{name}: {e}")
    
    def get_stats(self) -> Dict[str, Any]:
        """Get operator statistics"""
        return {
            **self.stats,
            'is_running': self.is_running,
            'namespace': self.namespace,
            'operator_name': self.operator_name,
            'kubernetes_available': KUBERNETES_AVAILABLE,
            'kopf_available': KOPF_AVAILABLE
        }
    
    async def stop_operator(self):
        """Stop the Kubernetes operator"""
        self.is_running = False
        self.logger.info("TuskLang Kubernetes operator stopped")
    
    def create_helm_chart(self, chart_path: str) -> str:
        """Create Helm chart for TuskLang operator"""
        os.makedirs(chart_path, exist_ok=True)
        
        # Chart.yaml
        chart_yaml = {
            'apiVersion': 'v2',
            'name': 'tusklang-operator',
            'description': 'TuskLang Kubernetes Operator',
            'type': 'application',
            'version': '1.0.0',
            'appVersion': '1.0.0',
            'keywords': ['tusklang', 'operator', 'kubernetes'],
            'maintainers': [{
                'name': 'TuskLang Team',
                'email': 'team@tusklang.io'
            }]
        }
        
        with open(os.path.join(chart_path, 'Chart.yaml'), 'w') as f:
            yaml.dump(chart_yaml, f)
        
        # Values.yaml
        values_yaml = {
            'image': {
                'repository': 'tusklang/python-operator',
                'tag': 'latest',
                'pullPolicy': 'IfNotPresent'
            },
            'replicaCount': 1,
            'namespace': 'tusk-lang',
            'resources': {
                'requests': {
                    'memory': '256Mi',
                    'cpu': '100m'
                },
                'limits': {
                    'memory': '512Mi',
                    'cpu': '500m'
                }
            },
            'rbac': {
                'create': True
            },
            'serviceAccount': {
                'create': True,
                'name': 'tusklang-operator'
            }
        }
        
        with open(os.path.join(chart_path, 'values.yaml'), 'w') as f:
            yaml.dump(values_yaml, f)
        
        # Create templates directory
        templates_dir = os.path.join(chart_path, 'templates')
        os.makedirs(templates_dir, exist_ok=True)
        
        # Create deployment template
        deployment_template = '''
apiVersion: apps/v1
kind: Deployment
metadata:
  name: {{ include "tusklang-operator.fullname" . }}
  labels:
    {{- include "tusklang-operator.labels" . | nindent 4 }}
spec:
  replicas: {{ .Values.replicaCount }}
  selector:
    matchLabels:
      {{- include "tusklang-operator.selectorLabels" . | nindent 6 }}
  template:
    metadata:
      labels:
        {{- include "tusklang-operator.selectorLabels" . | nindent 8 }}
    spec:
      serviceAccountName: {{ include "tusklang-operator.serviceAccountName" . }}
      containers:
        - name: operator
          image: "{{ .Values.image.repository }}:{{ .Values.image.tag }}"
          imagePullPolicy: {{ .Values.image.pullPolicy }}
          resources:
            {{- toYaml .Values.resources | nindent 12 }}
'''
        
        with open(os.path.join(templates_dir, 'deployment.yaml'), 'w') as f:
            f.write(deployment_template)
        
        return chart_path


# Convenience functions
def create_kubernetes_integration(config: Optional[Dict] = None) -> KubernetesIntegration:
    """Create Kubernetes integration instance"""
    return KubernetesIntegration(config)


async def deploy_tusklang_config(name: str, config_content: str, 
                                namespace: str = 'tusk-lang',
                                variables: Optional[Dict] = None,
                                replicas: int = 1) -> Dict[str, Any]:
    """Deploy TuskLang configuration to Kubernetes"""
    integration = KubernetesIntegration({'namespace': namespace})
    
    # Create TuskLangConfig resource
    config_resource = {
        'apiVersion': 'tusklang.io/v1',
        'kind': 'TuskLangConfig',
        'metadata': {
            'name': name,
            'namespace': namespace
        },
        'spec': {
            'config': config_content,
            'variables': variables or {},
            'replicas': replicas
        }
    }
    
    try:
        result = integration.custom_objects_api.create_namespaced_custom_object(
            group='tusklang.io',
            version='v1',
            namespace=namespace,
            plural='tusklangconfigs',
            body=config_resource
        )
        
        return {
            'success': True,
            'resource': result,
            'name': name,
            'namespace': namespace
        }
        
    except Exception as e:
        return {
            'success': False,
            'error': str(e),
            'name': name,
            'namespace': namespace
        } 