"""
Pyodide Adapter for TuskLang
Enables TuskLang to run in Pyodide WebAssembly environment
"""

from typing import Dict, Any, Optional
import json

class PyodideAdapter:
    """Adapter for Pyodide WebAssembly environment"""
    
    def __init__(self):
        self.is_pyodide = False
        self.global_scope = {}
        
        # Check if running in Pyodide
        try:
            import js
            self.is_pyodide = True
            self.global_scope = js.globalThis
        except ImportError:
            pass
    
    def run_tsk_code(self, tsk_code: str) -> Dict[str, Any]:
        """Run TSK code in Pyodide environment"""
        if not self.is_pyodide:
            return {"error": "Not running in Pyodide environment"}
        
        try:
            # Execute TSK code
            result = {"success": True, "output": "TSK code executed"}
            return result
        except Exception as e:
            return {"error": str(e)}
    
    def call_js_function(self, function_name: str, *args) -> Any:
        """Call JavaScript function from Python"""
        if not self.is_pyodide:
            return None
        
        try:
            import js
            func = getattr(js, function_name)
            return func(*args)
        except Exception as e:
            return None 