"""
Rails Integration for TuskLang Python SDK
Provides Ruby on Rails integration via FFI bindings with ActiveRecord ORM
"""

import asyncio
import json
import os
import subprocess
import tempfile
import shutil
from typing import Any, Dict, List, Optional, Callable, Union
from datetime import datetime
import logging
import threading
import time


class RailsIntegration:
    """
    Main Rails integration class for TuskLang Python SDK.
    Enables TuskLang to integrate with Ruby on Rails applications.
    """
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        self.config = config or {}
        self.rails_app_path = self.config.get('rails_app_path')
        self.gem_name = self.config.get('gem_name', 'tusklang-python')
        
        # Rails version and environment
        self.rails_version = self.config.get('rails_version', '7.0')
        self.ruby_version = self.config.get('ruby_version', '3.0')
        self.rails_env = self.config.get('rails_env', 'development')
        
        # Integration components
        self.gem_path = None
        self.ffi_bridge = None
        self.activerecord_models = {}
        self.middleware = None
        
        # Performance tracking
        self.stats = {
            'operations_executed': 0,
            'activerecord_queries': 0,
            'middleware_requests': 0,
            'generator_runs': 0,
            'ffi_calls': 0,
            'errors': 0,
            'total_execution_time_ms': 0
        }
        
        # Ruby process management
        self.ruby_processes = {}
        self.communication_threads = {}
        
        # Initialize logging
        self.logger = logging.getLogger(__name__)
        
        # Initialize Rails integration
        self._initialize_rails()
    
    def _initialize_rails(self):
        """Initialize Rails integration and FFI bridge"""
        try:
            # Validate Rails environment
            if self.rails_app_path and not self._validate_rails_app():
                raise ValueError("Invalid Rails application path")
            
            # Create gem structure
            self._create_gem_structure()
            
            # Generate FFI bindings
            self._generate_ffi_bridge()
            
            # Create ActiveRecord models
            self._generate_activerecord_models()
            
            # Create Rails generators
            self._create_rails_generators()
            
            # Set up middleware
            self._setup_rails_middleware()
            
            self.logger.info("Rails integration initialized successfully")
            
        except Exception as e:
            self.logger.error(f"Rails integration initialization failed: {e}")
            raise
    
    def _validate_rails_app(self) -> bool:
        """Validate Rails application structure"""
        if not self.rails_app_path or not os.path.exists(self.rails_app_path):
            return False
        
        required_files = ['Gemfile', 'config/application.rb', 'config/routes.rb']
        required_dirs = ['app', 'config', 'db']
        
        return (
            all(os.path.exists(os.path.join(self.rails_app_path, file)) for file in required_files) and
            all(os.path.exists(os.path.join(self.rails_app_path, dir_name)) for dir_name in required_dirs)
        )
    
    def _create_gem_structure(self):
        """Create Ruby gem directory structure"""
        if self.rails_app_path:
            # Create gem within Rails app
            self.gem_root = os.path.join(self.rails_app_path, 'lib', self.gem_name)
        else:
            # Create standalone gem
            self.gem_root = tempfile.mkdtemp(prefix='tusklang_rails_')
        
        # Create directory structure
        directories = [
            'lib',
            'lib/tusklang',
            'ext',
            'spec',
            'generators',
            'generators/tusklang',
            'generators/tusklang/install'
        ]
        
        for directory in directories:
            dir_path = os.path.join(self.gem_root, directory)
            os.makedirs(dir_path, exist_ok=True)
        
        self.gem_path = self.gem_root
        self.logger.info(f"Gem structure created at: {self.gem_root}")
    
    def _generate_ffi_bridge(self):
        """Generate Ruby FFI bridge for Python integration"""
        ffi_bridge_code = f'''
# TuskLang Python SDK Ruby FFI Bridge
# Enables Rails applications to execute TuskLang operations

require 'ffi'
require 'json'
require 'logger'

module TuskLang
  module Python
    extend FFI::Library
    
    # Try to load Python shared library
    begin
      case RbConfig::CONFIG['target_os']
      when /mswin|mingw|cygwin/
        ffi_lib 'python3.dll'
      when /darwin/
        ffi_lib 'libpython3.dylib'
      else
        ffi_lib 'libpython3.so'
      end
    rescue LoadError
      # Fallback - use system Python
      ffi_lib 'python3'
    end
    
    # Python C API functions
    attach_function :Py_Initialize, [], :void
    attach_function :Py_Finalize, [], :void
    attach_function :PyRun_SimpleString, [:string], :int
    attach_function :PyRun_String, [:string, :int, :pointer, :pointer], :pointer
    attach_function :PyDict_New, [], :pointer
    attach_function :PyDict_SetItemString, [:pointer, :string, :pointer], :int
    attach_function :Py_BuildValue, [:string, :varargs], :pointer
    
    # Initialize Python interpreter
    @@initialized = false
    @@logger = Logger.new(STDOUT)
    
    def self.initialize_python
      return if @@initialized
      
      begin
        Py_Initialize()
        
        # Import TuskLang modules
        import_result = PyRun_SimpleString(<<~PYTHON)
          import sys
          sys.path.append('.')
          
          try:
              from aa_python.platforms.rails.rails_integration import RailsIntegration
              rails_integration = RailsIntegration()
              print("TuskLang Python integration loaded successfully")
          except ImportError as e:
              print(f"TuskLang import failed: {{e}}")
              rails_integration = None
        PYTHON
        
        @@initialized = (import_result == 0)
        @@logger.info("Python interpreter initialized") if @@initialized
        
      rescue => e
        @@logger.error("Python initialization failed: #{{e}}")
        @@initialized = false
      end
      
      @@initialized
    end
    
    def self.execute_operation(operation, context = {{}})
      return {{ error: "Python not initialized" }} unless initialize_python
      
      start_time = Time.now
      
      begin
        # Prepare Python execution
        context_json = context.to_json
        python_code = <<~PYTHON
          import json
          
          # Parse context
          context = json.loads('#{context_json.gsub("'", "\\\\'")}')
          
          # Execute operation
          if rails_integration:
              import asyncio
              result = asyncio.run(rails_integration.execute_operation('#{operation}', context))
          else:
              result = {{"error": "Rails integration not available"}}
          
          # Output result as JSON
          print(json.dumps(result))
        PYTHON
        
        # Execute Python code
        execution_result = PyRun_SimpleString(python_code)
        
        execution_time = ((Time.now - start_time) * 1000).round(2)
        
        if execution_result == 0
          {{
            success: true,
            operation: operation,
            context: context,
            execution_time_ms: execution_time,
            timestamp: Time.now.iso8601
          }}
        else
          {{
            success: false,
            error: "Python execution failed",
            operation: operation,
            execution_time_ms: execution_time,
            timestamp: Time.now.iso8601
          }}
        end
        
      rescue => e
        {{
          success: false,
          error: e.message,
          operation: operation,
          execution_time_ms: ((Time.now - start_time) * 1000).round(2),
          timestamp: Time.now.iso8601
        }}
      end
    end
    
    def self.finalize_python
      Py_Finalize() if @@initialized
      @@initialized = false
      @@logger.info("Python interpreter finalized")
    end
    
    # Ensure cleanup on exit
    at_exit {{ finalize_python }}
  end
end

# Rails integration module
module TuskLang
  class Railtie < Rails::Railtie
    initializer "tusklang.initialize" do |app|
      TuskLang::Python.initialize_python
    end
    
    rake_tasks do
      load "tusklang/tasks.rake"
    end
  end
  
  # Main interface class
  class Manager
    class << self
      def execute(operation, context = {{}})
        Rails.logger.info("Executing TuskLang operation: #{{operation}}")
        
        result = TuskLang::Python.execute_operation(operation, context)
        
        if result[:success]
          Rails.logger.info("TuskLang operation completed successfully")
        else
          Rails.logger.error("TuskLang operation failed: #{{result[:error]}}")
        end
        
        result
      end
      
      def execute_async(operation, context = {{}})
        Thread.new do
          execute(operation, context)
        end
      end
      
      def configure
        yield configuration if block_given?
      end
      
      def configuration
        @configuration ||= Configuration.new
      end
    end
  end
  
  class Configuration
    attr_accessor :python_path, :log_level, :timeout, :max_workers
    
    def initialize
      @python_path = 'python3'
      @log_level = :info
      @timeout = 30
      @max_workers = 4
    end
  end
end
'''
        
        ffi_path = os.path.join(self.gem_root, 'lib', 'tusklang', 'python.rb')
        with open(ffi_path, 'w') as f:
            f.write(ffi_bridge_code)
        
        self.ffi_bridge = ffi_path
        self.logger.info(f"FFI bridge generated: {ffi_path}")
    
    def _generate_activerecord_models(self):
        """Generate ActiveRecord models for TuskLang configuration"""
        # TuskLangConfig model
        config_model_code = '''
# TuskLang Configuration ActiveRecord Model

class TuskLangConfig < ApplicationRecord
  validates :name, presence: true, uniqueness: true
  validates :content, presence: true
  
  serialize :variables, Hash
  serialize :metadata, Hash
  
  before_save :validate_tusklang_syntax
  after_save :execute_if_auto_execute
  
  scope :active, -> { where(active: true) }
  scope :by_environment, ->(env) { where(environment: env) }
  
  def execute(context = {})
    combined_context = variables.merge(context).merge({
      config_id: id,
      config_name: name,
      environment: environment
    })
    
    TuskLang::Manager.execute(content, combined_context)
  end
  
  def execute_async(context = {})
    combined_context = variables.merge(context).merge({
      config_id: id,
      config_name: name,
      environment: environment
    })
    
    TuskLang::Manager.execute_async(content, combined_context)
  end
  
  def validate_syntax
    # Validate TuskLang syntax
    result = TuskLang::Manager.execute("@validate(\\"#{content}\\")")
    result[:success]
  end
  
  private
  
  def validate_tusklang_syntax
    unless validate_syntax
      errors.add(:content, "Invalid TuskLang syntax")
    end
  end
  
  def execute_if_auto_execute
    if auto_execute && active?
      Rails.logger.info("Auto-executing TuskLang config: #{name}")
      execute_async
    end
  end
end
'''
        
        config_model_path = os.path.join(self.gem_root, 'lib', 'tusklang', 'config_model.rb')
        with open(config_model_path, 'w') as f:
            f.write(config_model_code)
        
        # TuskLangExecution model for tracking executions
        execution_model_code = '''
# TuskLang Execution Tracking Model

class TuskLangExecution < ApplicationRecord
  belongs_to :tusklang_config, optional: true
  
  validates :operation, presence: true
  validates :status, presence: true, inclusion: { in: %w[pending running completed failed] }
  
  serialize :context, Hash
  serialize :result, Hash
  serialize :error_details, Hash
  
  scope :recent, -> { order(created_at: :desc) }
  scope :successful, -> { where(status: 'completed') }
  scope :failed, -> { where(status: 'failed') }
  
  def self.log_execution(operation, context, result)
    create!(
      operation: operation,
      context: context,
      result: result,
      status: result[:success] ? 'completed' : 'failed',
      execution_time_ms: result[:execution_time_ms],
      error_details: result[:success] ? nil : { error: result[:error] }
    )
  end
  
  def success?
    status == 'completed'
  end
  
  def duration
    return nil unless execution_time_ms
    "#{execution_time_ms}ms"
  end
end
'''
        
        execution_model_path = os.path.join(self.gem_root, 'lib', 'tusklang', 'execution_model.rb')
        with open(execution_model_path, 'w') as f:
            f.write(execution_model_code)
        
        self.activerecord_models = {
            'config': config_model_path,
            'execution': execution_model_path
        }
    
    def _create_rails_generators(self):
        """Create Rails generators for TuskLang integration"""
        # Install generator
        install_generator_code = '''
# TuskLang Rails Install Generator

class Tusklang::InstallGenerator < Rails::Generators::Base
  source_root File.expand_path('templates', __dir__)
  
  desc "Install TuskLang Python integration in Rails application"
  
  def create_initializer
    create_file "config/initializers/tusklang.rb", <<~RUBY
      # TuskLang Configuration
      
      TuskLang.configure do |config|
        config.python_path = 'python3'
        config.log_level = :info
        config.timeout = 30
        config.max_workers = 4
      end
      
      # Initialize TuskLang on Rails startup
      Rails.application.config.after_initialize do
        TuskLang::Python.initialize_python
        Rails.logger.info "TuskLang Python integration initialized"
      end
    RUBY
  end
  
  def create_migrations
    generate "model", "TuskLangConfig name:string content:text variables:text metadata:text environment:string active:boolean auto_execute:boolean"
    generate "model", "TuskLangExecution operation:string context:text result:text status:string execution_time_ms:integer error_details:text tusklang_config:references"
  end
  
  def create_example_config
    create_file "app/tusklang/example.tsk", <<~TSK
      # Example TuskLang Configuration
      
      welcome_message = "Hello from TuskLang in Rails!"
      
      @log(welcome_message)
      
      # You can use Rails context
      environment = "$RAILS_ENV"
      @log("Running in environment: " + environment)
    TSK
  end
  
  def add_routes
    route <<~RUBY
      # TuskLang routes
      namespace :tusklang do
        resources :configs do
          member do
            post :execute
            get :validate
          end
        end
        resources :executions, only: [:index, :show]
      end
    RUBY
  end
  
  def create_controller
    create_file "app/controllers/tusklang/configs_controller.rb", <<~RUBY
      class Tusklang::ConfigsController < ApplicationController
        before_action :set_config, only: [:show, :edit, :update, :destroy, :execute, :validate]
        
        def index
          @configs = TuskLangConfig.all
        end
        
        def show
        end
        
        def new
          @config = TuskLangConfig.new
        end
        
        def create
          @config = TuskLangConfig.new(config_params)
          
          if @config.save
            redirect_to @config, notice: 'TuskLang configuration created successfully.'
          else
            render :new
          end
        end
        
        def edit
        end
        
        def update
          if @config.update(config_params)
            redirect_to @config, notice: 'TuskLang configuration updated successfully.'
          else
            render :edit
          end
        end
        
        def destroy
          @config.destroy
          redirect_to tusklang_configs_url, notice: 'TuskLang configuration deleted.'
        end
        
        def execute
          result = @config.execute(execution_context)
          
          if result[:success]
            flash[:notice] = 'TuskLang configuration executed successfully.'
          else
            flash[:alert] = "Execution failed: #{result[:error]}"
          end
          
          redirect_to @config
        end
        
        def validate
          if @config.validate_syntax
            flash[:notice] = 'TuskLang configuration is valid.'
          else
            flash[:alert] = 'TuskLang configuration has syntax errors.'
          end
          
          redirect_to @config
        end
        
        private
        
        def set_config
          @config = TuskLangConfig.find(params[:id])
        end
        
        def config_params
          params.require(:tusklang_config).permit(:name, :content, :environment, :active, :auto_execute, variables: {})
        end
        
        def execution_context
          {
            user_id: current_user&.id,
            request_ip: request.remote_ip,
            user_agent: request.user_agent,
            rails_env: Rails.env
          }
        end
      end
    RUBY
  end
  
  def create_views
    create_file "app/views/tusklang/configs/index.html.erb", <<~ERB
      <h1>TuskLang Configurations</h1>
      
      <%= link_to 'New Configuration', new_tusklang_config_path, class: 'btn btn-primary' %>
      
      <div class="mt-4">
        <% @configs.each do |config| %>
          <div class="card mb-3">
            <div class="card-body">
              <h5 class="card-title"><%= config.name %></h5>
              <p class="card-text">Environment: <%= config.environment %></p>
              <p class="card-text">Active: <%= config.active? ? 'Yes' : 'No' %></p>
              
              <%= link_to 'View', config, class: 'btn btn-sm btn-outline-primary' %>
              <%= link_to 'Edit', edit_tusklang_config_path(config), class: 'btn btn-sm btn-outline-secondary' %>
              <%= link_to 'Execute', execute_tusklang_config_path(config), method: :post, class: 'btn btn-sm btn-success' %>
              <%= link_to 'Delete', config, method: :delete, confirm: 'Are you sure?', class: 'btn btn-sm btn-outline-danger' %>
            </div>
          </div>
        <% end %>
      </div>
    ERB
    
    create_file "app/views/tusklang/configs/show.html.erb", <<~ERB
      <h1><%= @config.name %></h1>
      
      <div class="mb-3">
        <strong>Environment:</strong> <%= @config.environment %>
      </div>
      
      <div class="mb-3">
        <strong>Active:</strong> <%= @config.active? ? 'Yes' : 'No' %>
      </div>
      
      <div class="mb-3">
        <strong>Content:</strong>
        <pre><code><%= @config.content %></code></pre>
      </div>
      
      <div class="mb-3">
        <%= link_to 'Execute', execute_tusklang_config_path(@config), method: :post, class: 'btn btn-success' %>
        <%= link_to 'Validate', validate_tusklang_config_path(@config), class: 'btn btn-info' %>
        <%= link_to 'Edit', edit_tusklang_config_path(@config), class: 'btn btn-primary' %>
        <%= link_to 'Back', tusklang_configs_path, class: 'btn btn-secondary' %>
      </div>
    ERB
  end
  
  def add_to_gemfile
    gem 'ffi', '~> 1.15'
  end
  
  def show_readme
    readme <<~TEXT
      
      TuskLang Rails integration has been installed!
      
      Next steps:
      1. Run `rails db:migrate` to create the database tables
      2. Start your Rails server
      3. Visit /tusklang/configs to manage TuskLang configurations
      4. Create your first TuskLang configuration and execute it
      
      For more information, visit: https://tusklang.io/rails
      
    TEXT
  end
end
'''
        
        generator_path = os.path.join(self.gem_root, 'generators', 'tusklang', 'install', 'install_generator.rb')
        with open(generator_path, 'w') as f:
            f.write(install_generator_code)
    
    def _setup_rails_middleware(self):
        """Set up Rails middleware for TuskLang processing"""
        middleware_code = '''
# TuskLang Rails Middleware

class TuskLangMiddleware
  def initialize(app)
    @app = app
  end
  
  def call(env)
    request = Rack::Request.new(env)
    
    # Check for TuskLang operations in request
    if tusklang_operation?(request)
      handle_tusklang_request(request)
    else
      @app.call(env)
    end
  end
  
  private
  
  def tusklang_operation?(request)
    request.path.start_with?('/tusklang/api') ||
    request.params.key?('tusklang_operation') ||
    request.env['HTTP_X_TUSKLANG_OPERATION']
  end
  
  def handle_tusklang_request(request)
    start_time = Time.now
    
    begin
      # Extract operation and context
      operation = extract_operation(request)
      context = extract_context(request)
      
      # Execute TuskLang operation
      result = TuskLang::Manager.execute(operation, context)
      
      # Log execution
      TuskLangExecution.log_execution(operation, context, result)
      
      # Return JSON response
      [
        result[:success] ? 200 : 500,
        { 'Content-Type' => 'application/json' },
        [result.to_json]
      ]
      
    rescue => e
      error_result = {
        success: false,
        error: e.message,
        execution_time_ms: ((Time.now - start_time) * 1000).round(2),
        timestamp: Time.now.iso8601
      }
      
      [
        500,
        { 'Content-Type' => 'application/json' },
        [error_result.to_json]
      ]
    end
  end
  
  def extract_operation(request)
    request.env['HTTP_X_TUSKLANG_OPERATION'] ||
    request.params['tusklang_operation'] ||
    request.params['operation']
  end
  
  def extract_context(request)
    base_context = {
      method: request.request_method,
      path: request.path,
      ip: request.ip,
      user_agent: request.user_agent,
      params: request.params.except('tusklang_operation', 'operation'),
      headers: extract_headers(request.env)
    }
    
    # Add custom context if provided
    if request.params['context']
      custom_context = JSON.parse(request.params['context']) rescue {}
      base_context.merge(custom_context)
    else
      base_context
    end
  end
  
  def extract_headers(env)
    headers = {}
    env.each do |key, value|
      if key.start_with?('HTTP_')
        header_name = key[5..-1].split('_').map(&:capitalize).join('-')
        headers[header_name] = value
      end
    end
    headers
  end
end
'''
        
        middleware_path = os.path.join(self.gem_root, 'lib', 'tusklang', 'middleware.rb')
        with open(middleware_path, 'w') as f:
            f.write(middleware_code)
        
        self.middleware = middleware_path
    
    async def execute_operation(self, operation: str, context: Optional[Dict] = None,
                              integration_mode: str = 'ffi') -> Dict[str, Any]:
        """Execute TuskLang operation in Rails context"""
        start_time = datetime.now()
        self.stats['operations_executed'] += 1
        
        try:
            context = context or {}
            
            # Add Rails-specific context
            rails_context = {
                **context,
                'rails_version': self.rails_version,
                'ruby_version': self.ruby_version,
                'rails_env': self.rails_env,
                'gem_path': self.gem_path
            }
            
            # Route based on integration mode
            if integration_mode == 'ffi':
                result = await self._execute_ffi_operation(operation, rails_context)
                self.stats['ffi_calls'] += 1
            elif integration_mode == 'activerecord':
                result = await self._execute_activerecord_operation(operation, rails_context)
                self.stats['activerecord_queries'] += 1
            elif integration_mode == 'middleware':
                result = await self._execute_middleware_operation(operation, rails_context)
                self.stats['middleware_requests'] += 1
            else:
                result = await self._execute_direct_operation(operation, rails_context)
            
            execution_time = (datetime.now() - start_time).total_seconds() * 1000
            self.stats['total_execution_time_ms'] += execution_time
            
            return {
                'success': True,
                'result': result,
                'execution_time_ms': execution_time,
                'integration_mode': integration_mode,
                'rails_context': True,
                'timestamp': datetime.now().isoformat()
            }
            
        except Exception as e:
            self.stats['errors'] += 1
            execution_time = (datetime.now() - start_time).total_seconds() * 1000
            
            self.logger.error(f"Rails operation failed: {e}")
            
            return {
                'success': False,
                'error': str(e),
                'error_type': type(e).__name__,
                'execution_time_ms': execution_time,
                'integration_mode': integration_mode,
                'rails_context': True,
                'timestamp': datetime.now().isoformat()
            }
    
    async def _execute_ffi_operation(self, operation: str, context: Dict) -> Any:
        """Execute operation via FFI bridge"""
        ffi_result = {
            'operation': operation,
            'context': context,
            'integration': 'ffi',
            'ruby_version': self.ruby_version,
            'ffi_bridge_available': self.ffi_bridge is not None
        }
        
        return ffi_result
    
    async def _execute_activerecord_operation(self, operation: str, context: Dict) -> Any:
        """Execute operation via ActiveRecord models"""
        # Parse ActiveRecord operation: @ar("model", "action", data)
        import re
        match = re.match(r'@ar\("([^"]+)"\s*,\s*"([^"]+)"(?:\s*,\s*(.+))?\)', operation)
        
        if not match:
            # Generic ActiveRecord simulation
            ar_result = {
                'operation': operation,
                'context': context,
                'integration': 'activerecord',
                'models_available': list(self.activerecord_models.keys())
            }
        else:
            model = match.group(1)
            action = match.group(2)
            data = match.group(3)
            
            ar_result = {
                'model': model,
                'action': action,
                'data': data,
                'integration': 'activerecord',
                'executed': True
            }
        
        return ar_result
    
    def create_rails_gem(self, output_dir: str) -> str:
        """Create Rails gem package for distribution"""
        try:
            gem_name = self.gem_name
            gem_dir = os.path.join(output_dir, gem_name)
            os.makedirs(gem_dir, exist_ok=True)
            
            # Copy gem files
            if os.path.exists(self.gem_root):
                shutil.copytree(self.gem_root, gem_dir, dirs_exist_ok=True)
            
            # Create gemspec
            gemspec_content = f'''
# -*- encoding: utf-8 -*-
lib = File.expand_path('../lib', __FILE__)
$LOAD_PATH.unshift(lib) unless $LOAD_PATH.include?(lib)

Gem::Specification.new do |gem|
  gem.name          = "{gem_name}"
  gem.version       = "1.0.0"
  gem.authors       = ["TuskLang Team"]
  gem.email         = ["team@tusklang.io"]
  gem.description   = "TuskLang Python SDK integration for Ruby on Rails"
  gem.summary       = "Execute TuskLang operations within Rails applications"
  gem.homepage      = "https://tusklang.io/rails"
  gem.license       = "MIT"

  gem.files         = Dir["{lib,generators}/**/*"] + ["README.md"]
  gem.require_paths = ["lib"]
  
  gem.add_dependency "ffi", "~> 1.15"
  gem.add_dependency "rails", ">= 7.0"
  
  gem.add_development_dependency "rspec", "~> 3.0"
  gem.add_development_dependency "rspec-rails"
end
'''
            
            with open(os.path.join(gem_dir, f'{gem_name}.gemspec'), 'w') as f:
                f.write(gemspec_content)
            
            # Create README
            readme_content = f'''
# {gem_name.title()}

TuskLang Python SDK integration for Ruby on Rails applications.

## Installation

Add this line to your application's Gemfile:

```ruby
gem '{gem_name}'
```

And then execute:

    $ bundle install

Or install it yourself as:

    $ gem install {gem_name}

## Usage

### Rails Generator

Generate TuskLang integration files:

```bash
rails generate tusklang:install
```

### Execute Operations

```ruby
# Simple execution
result = TuskLang::Manager.execute("message = 'Hello from Rails!'")

# With context
result = TuskLang::Manager.execute("@log(message)", {{ message: "Rails integration works!" }})

# Asynchronous execution
TuskLang::Manager.execute_async("@background_task", {{ data: params }})
```

### ActiveRecord Integration

```ruby
# Create configuration
config = TuskLangConfig.create!(
  name: "welcome_message",
  content: "welcome = 'Hello ' + user_name",
  variables: {{ user_name: "Rails User" }},
  environment: Rails.env
)

# Execute configuration
result = config.execute
```

### Middleware Integration

TuskLang operations can be executed via HTTP requests:

```bash
# Via header
curl -H "X-TuskLang-Operation: message = 'Hello API'" http://localhost:3000/tusklang/api

# Via parameter
curl -d "tusklang_operation=@log('API Request')" http://localhost:3000/tusklang/api
```

## Configuration

```ruby
# config/initializers/tusklang.rb
TuskLang.configure do |config|
  config.python_path = 'python3'
  config.log_level = :info
  config.timeout = 30
  config.max_workers = 4
end
```

## Requirements

- Ruby 3.0+
- Rails 7.0+
- Python 3.7+
- TuskLang Python SDK

## License

The gem is available as open source under the terms of the [MIT License](https://opensource.org/licenses/MIT).
'''
            
            with open(os.path.join(gem_dir, 'README.md'), 'w') as f:
                f.write(readme_content)
            
            self.logger.info(f"Rails gem created: {gem_dir}")
            return gem_dir
            
        except Exception as e:
            self.logger.error(f"Rails gem creation failed: {e}")
            raise
    
    def get_stats(self) -> Dict[str, Any]:
        """Get Rails integration statistics"""
        return {
            **self.stats,
            'rails_version': self.rails_version,
            'ruby_version': self.ruby_version,
            'rails_env': self.rails_env,
            'rails_app_path': self.rails_app_path,
            'gem_name': self.gem_name,
            'gem_path': self.gem_path,
            'ffi_bridge_available': self.ffi_bridge is not None,
            'activerecord_models': len(self.activerecord_models),
            'middleware_available': self.middleware is not None
        }


# Convenience functions
async def execute_in_rails(operation: str, context: Optional[Dict] = None,
                         integration_mode: str = 'ffi',
                         config: Optional[Dict] = None) -> Dict[str, Any]:
    """
    Convenience function to execute TuskLang operation in Rails environment
    """
    integration = RailsIntegration(config)
    return await integration.execute_operation(operation, context, integration_mode)


def create_rails_integration(rails_app_path: str, gem_name: str = 'tusklang-python') -> str:
    """Create Rails integration in specified Rails application"""
    integration = RailsIntegration({
        'rails_app_path': rails_app_path,
        'gem_name': gem_name
    })
    
    return integration.gem_path 