"""
Unity Integration for TuskLang Python SDK
Provides Unity native plugin with C# wrapper and asset pipeline integration
"""

import asyncio
import json
import os
import subprocess
import tempfile
import shutil
from typing import Any, Dict, List, Optional, Callable, Union
from datetime import datetime
import logging
import threading
import queue
import platform


class UnityIntegration:
    """
    Main Unity integration class for TuskLang Python SDK.
    Enables TuskLang to integrate with Unity game engine projects.
    """
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        self.config = config or {}
        self.unity_project_path = self.config.get('unity_project_path')
        self.plugin_name = self.config.get('plugin_name', 'TuskLangPlugin')
        
        # Unity version and platform settings
        self.unity_version = self.config.get('unity_version', '2022.3')
        self.target_platforms = self.config.get('platforms', ['Windows', 'macOS', 'Linux', 'Android', 'iOS'])
        
        # Plugin architecture
        self.native_plugin_path = None
        self.csharp_wrapper_path = None
        self.asset_processor = None
        
        # Performance tracking
        self.stats = {
            'operations_executed': 0,
            'asset_operations': 0,
            'native_calls': 0,
            'script_executions': 0,
            'errors': 0,
            'memory_usage_mb': 0
        }
        
        # Unity communication
        self.unity_process = None
        self.communication_port = self.config.get('port', 7777)
        self.is_connected = False
        
        # Threading
        self.executor = threading.ThreadPoolExecutor(max_workers=2)
        self.message_queue = queue.Queue()
        
        # Initialize logging
        self.logger = logging.getLogger(__name__)
        
        # Initialize Unity integration
        self._initialize_unity()
    
    def _initialize_unity(self):
        """Initialize Unity integration and plugin system"""
        try:
            # Validate Unity project
            if self.unity_project_path and not self._validate_unity_project():
                raise ValueError("Invalid Unity project path")
            
            # Create plugin structure
            self._create_plugin_structure()
            
            # Generate C# wrapper
            self._generate_csharp_wrapper()
            
            # Build native plugin
            self._build_native_plugin()
            
            # Set up asset pipeline
            self._setup_asset_pipeline()
            
            self.logger.info("Unity integration initialized successfully")
            
        except Exception as e:
            self.logger.error(f"Unity integration initialization failed: {e}")
            raise
    
    def _validate_unity_project(self) -> bool:
        """Validate Unity project structure"""
        if not self.unity_project_path or not os.path.exists(self.unity_project_path):
            return False
        
        required_dirs = ['Assets', 'ProjectSettings', 'Packages']
        return all(
            os.path.exists(os.path.join(self.unity_project_path, dir_name))
            for dir_name in required_dirs
        )
    
    def _create_plugin_structure(self):
        """Create Unity plugin directory structure"""
        if not self.unity_project_path:
            # Create temporary plugin structure for standalone use
            self.plugin_root = tempfile.mkdtemp(prefix='tusklang_unity_')
        else:
            self.plugin_root = os.path.join(
                self.unity_project_path,
                'Assets',
                'Plugins',
                'TuskLang'
            )
        
        # Create directory structure
        directories = [
            'Native',
            'Scripts',
            'Editor',
            'Resources',
            'Examples',
            'Documentation'
        ]
        
        for directory in directories:
            dir_path = os.path.join(self.plugin_root, directory)
            os.makedirs(dir_path, exist_ok=True)
        
        self.logger.info(f"Plugin structure created at: {self.plugin_root}")
    
    def _generate_csharp_wrapper(self):
        """Generate C# wrapper for Unity integration"""
        csharp_code = f'''
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using UnityEngine;
using System.Threading.Tasks;

namespace TuskLang
{{
    /// <summary>
    /// TuskLang Python SDK Unity Integration
    /// Provides C# wrapper for Python TuskLang operations
    /// </summary>
    public class TuskLangManager : MonoBehaviour
    {{
        [Header("TuskLang Configuration")]
        public string pythonPath = "python";
        public string tuskLangModule = "aa_python";
        public bool initializeOnStart = true;
        public bool enableLogging = true;
        
        [Header("Performance Settings")]
        public int maxConcurrentOperations = 4;
        public float operationTimeoutSeconds = 30f;
        
        private static TuskLangManager _instance;
        private Dictionary<string, object> _context;
        private Queue<TuskLangOperation> _operationQueue;
        private bool _isInitialized = false;
        
        public static TuskLangManager Instance
        {{
            get
            {{
                if (_instance == null)
                {{
                    _instance = FindObjectOfType<TuskLangManager>();
                    if (_instance == null)
                    {{
                        GameObject go = new GameObject("TuskLangManager");
                        _instance = go.AddComponent<TuskLangManager>();
                        DontDestroyOnLoad(go);
                    }}
                }}
                return _instance;
            }}
        }}
        
        #region Native Plugin Imports
        
        [DllImport("{self.plugin_name}")]
        private static extern int InitializeTuskLang(string configPath);
        
        [DllImport("{self.plugin_name}")]
        private static extern IntPtr ExecuteOperation(string operation, string context);
        
        [DllImport("{self.plugin_name}")]
        private static extern int GetStats(IntPtr statsBuffer, int bufferSize);
        
        [DllImport("{self.plugin_name}")]
        private static extern void FreeTuskLangString(IntPtr ptr);
        
        [DllImport("{self.plugin_name}")]
        private static extern void ShutdownTuskLang();
        
        #endregion
        
        void Start()
        {{
            if (initializeOnStart)
            {{
                Initialize();
            }}
        }}
        
        void OnDestroy()
        {{
            Shutdown();
        }}
        
        /// <summary>
        /// Initialize TuskLang integration
        /// </summary>
        public bool Initialize()
        {{
            try
            {{
                _context = new Dictionary<string, object>();
                _operationQueue = new Queue<TuskLangOperation>();
                
                // Initialize native plugin
                string configPath = Application.persistentDataPath + "/tusklang_config.json";
                int result = InitializeTuskLang(configPath);
                
                _isInitialized = (result == 0);
                
                if (_isInitialized && enableLogging)
                {{
                    Debug.Log("TuskLang Unity integration initialized successfully");
                }}
                
                return _isInitialized;
            }}
            catch (Exception e)
            {{
                Debug.LogError($"TuskLang initialization failed: {{e.Message}}");
                return false;
            }}
        }}
        
        /// <summary>
        /// Execute TuskLang operation asynchronously
        /// </summary>
        public async Task<TuskLangResult> ExecuteAsync(string operation, Dictionary<string, object> context = null)
        {{
            if (!_isInitialized)
            {{
                return new TuskLangResult
                {{
                    Success = false,
                    Error = "TuskLang not initialized"
                }};
            }}
            
            return await Task.Run(() =>
            {{
                try
                {{
                    string contextJson = context != null ? JsonUtility.ToJson(context) : "{{}}";
                    
                    IntPtr resultPtr = ExecuteOperation(operation, contextJson);
                    if (resultPtr == IntPtr.Zero)
                    {{
                        return new TuskLangResult
                        {{
                            Success = false,
                            Error = "Native operation failed"
                        }};
                    }}
                    
                    string resultJson = Marshal.PtrToStringAnsi(resultPtr);
                    FreeTuskLangString(resultPtr);
                    
                    var result = JsonUtility.FromJson<TuskLangResult>(resultJson);
                    return result;
                }}
                catch (Exception e)
                {{
                    return new TuskLangResult
                    {{
                        Success = false,
                        Error = e.Message
                    }};
                }}
            }});
        }}
        
        /// <summary>
        /// Execute TuskLang operation synchronously
        /// </summary>
        public TuskLangResult Execute(string operation, Dictionary<string, object> context = null)
        {{
            return ExecuteAsync(operation, context).Result;
        }}
        
        /// <summary>
        /// Get execution statistics
        /// </summary>
        public TuskLangStats GetStatistics()
        {{
            try
            {{
                const int bufferSize = 1024;
                IntPtr buffer = Marshal.AllocHGlobal(bufferSize);
                
                int result = GetStats(buffer, bufferSize);
                if (result == 0)
                {{
                    string statsJson = Marshal.PtrToStringAnsi(buffer);
                    Marshal.FreeHGlobal(buffer);
                    
                    return JsonUtility.FromJson<TuskLangStats>(statsJson);
                }}
                
                Marshal.FreeHGlobal(buffer);
                return new TuskLangStats();
            }}
            catch (Exception e)
            {{
                Debug.LogError($"Failed to get statistics: {{e.Message}}");
                return new TuskLangStats();
            }}
        }}
        
        /// <summary>
        /// Shutdown TuskLang integration
        /// </summary>
        public void Shutdown()
        {{
            if (_isInitialized)
            {{
                ShutdownTuskLang();
                _isInitialized = false;
                
                if (enableLogging)
                {{
                    Debug.Log("TuskLang Unity integration shut down");
                }}
            }}
        }}
        
        /// <summary>
        /// Execute asset pipeline operation
        /// </summary>
        public async Task<TuskLangResult> ProcessAsset(string assetPath, string operation)
        {{
            var context = new Dictionary<string, object>
            {{
                {{ "asset_path", assetPath }},
                {{ "unity_version", Application.unityVersion }},
                {{ "platform", Application.platform.ToString() }}
            }};
            
            return await ExecuteAsync($"@asset(\\"{operation}\\", \\"{assetPath}\\")", context);
        }}
        
        /// <summary>
        /// Execute game object operation
        /// </summary>
        public async Task<TuskLangResult> ProcessGameObject(GameObject gameObject, string operation)
        {{
            var context = new Dictionary<string, object>
            {{
                {{ "object_name", gameObject.name }},
                {{ "object_id", gameObject.GetInstanceID() }},
                {{ "scene_name", gameObject.scene.name }}
            }};
            
            return await ExecuteAsync($"@gameobject(\\"{operation}\\", \\"{gameObject.name}\\")", context);
        }}
    }}
    
    /// <summary>
    /// TuskLang operation result
    /// </summary>
    [System.Serializable]
    public class TuskLangResult
    {{
        public bool Success;
        public string Error;
        public string Result;
        public float ExecutionTimeMs;
        public string Timestamp;
    }}
    
    /// <summary>
    /// TuskLang execution statistics
    /// </summary>
    [System.Serializable] 
    public class TuskLangStats
    {{
        public int OperationsExecuted;
        public int AssetOperations;
        public int NativeCalls;
        public int ScriptExecutions;
        public int Errors;
        public float MemoryUsageMB;
    }}
    
    /// <summary>
    /// Internal operation structure
    /// </summary>
    internal class TuskLangOperation
    {{
        public string Operation;
        public Dictionary<string, object> Context;
        public TaskCompletionSource<TuskLangResult> CompletionSource;
        public DateTime StartTime;
    }}
}}

/// <summary>
/// TuskLang Unity Editor utilities
/// </summary>
#if UNITY_EDITOR
namespace TuskLang.Editor
{{
    using UnityEditor;
    using UnityEditor.Build;
    using UnityEditor.Build.Reporting;
    
    public class TuskLangAssetProcessor : AssetPostprocessor
    {{
        static void OnPostprocessAllAssets(string[] importedAssets, string[] deletedAssets,
                                         string[] movedAssets, string[] movedFromAssetPaths)
        {{
            var manager = TuskLangManager.Instance;
            if (manager == null) return;
            
            foreach (string asset in importedAssets)
            {{
                if (asset.EndsWith(".tsk") || asset.EndsWith(".tusklang"))
                {{
                    ProcessTuskLangAsset(asset);
                }}
            }}
        }}
        
        static async void ProcessTuskLangAsset(string assetPath)
        {{
            try
            {{
                var result = await TuskLangManager.Instance.ProcessAsset(assetPath, "validate");
                
                if (!result.Success)
                {{
                    Debug.LogError($"TuskLang asset validation failed: {{result.Error}} - {{assetPath}}");
                }}
                else
                {{
                    Debug.Log($"TuskLang asset processed successfully: {{assetPath}}");
                }}
            }}
            catch (Exception e)
            {{
                Debug.LogError($"TuskLang asset processing error: {{e.Message}} - {{assetPath}}");
            }}
        }}
    }}
    
    [CustomEditor(typeof(TuskLangManager))]
    public class TuskLangManagerEditor : Editor
    {{
        public override void OnInspectorGUI()
        {{
            DrawDefaultInspector();
            
            GUILayout.Space(10);
            
            TuskLangManager manager = (TuskLangManager)target;
            
            if (GUILayout.Button("Initialize TuskLang"))
            {{
                manager.Initialize();
            }}
            
            if (GUILayout.Button("Get Statistics"))
            {{
                var stats = manager.GetStatistics();
                Debug.Log($"TuskLang Stats: {{JsonUtility.ToJson(stats, true)}}");
            }}
            
            if (GUILayout.Button("Test Operation"))
            {{
                TestOperation(manager);
            }}
            
            GUILayout.Space(10);
            EditorGUILayout.HelpBox("TuskLang Unity Integration v1.0.0", MessageType.Info);
        }}
        
        async void TestOperation(TuskLangManager manager)
        {{
            var result = await manager.ExecuteAsync("test = \\"Hello from Unity\\"");
            Debug.Log($"Test result: {{JsonUtility.ToJson(result, true)}}");
        }}
    }}
}}
#endif
'''
        
        csharp_path = os.path.join(self.plugin_root, 'Scripts', 'TuskLangManager.cs')
        with open(csharp_path, 'w') as f:
            f.write(csharp_code)
        
        self.csharp_wrapper_path = csharp_path
        self.logger.info(f"C# wrapper generated: {csharp_path}")
    
    def _build_native_plugin(self):
        """Build native plugin for Unity platforms"""
        try:
            # Create C++ native plugin source
            cpp_source = self._generate_native_plugin_source()
            
            # Determine build configuration for current platform
            current_platform = platform.system()
            
            if current_platform == "Windows":
                self._build_windows_plugin(cpp_source)
            elif current_platform == "Darwin":  # macOS
                self._build_macos_plugin(cpp_source)
            elif current_platform == "Linux":
                self._build_linux_plugin(cpp_source)
            else:
                self.logger.warning(f"Native plugin build not supported on {current_platform}")
                
        except Exception as e:
            self.logger.error(f"Native plugin build failed: {e}")
            # Continue without native plugin for development
    
    def _generate_native_plugin_source(self) -> str:
        """Generate C++ source for native plugin"""
        cpp_source = f'''
// TuskLang Unity Native Plugin
// Generated for Unity integration

#include <iostream>
#include <string>
#include <memory>
#include <map>
#include <mutex>
#include <thread>
#include <chrono>

#ifdef _WIN32
    #define EXPORT __declspec(dllexport)
#else
    #define EXPORT
#endif

// Global state
static std::mutex g_mutex;
static std::map<std::string, std::string> g_context;
static bool g_initialized = false;

struct TuskLangStats {{
    int operations_executed = 0;
    int asset_operations = 0;  
    int native_calls = 0;
    int script_executions = 0;
    int errors = 0;
    float memory_usage_mb = 0.0f;
}};

static TuskLangStats g_stats;

extern "C" {{
    
    EXPORT int InitializeTuskLang(const char* configPath) {{
        std::lock_guard<std::mutex> lock(g_mutex);
        
        try {{
            // Initialize TuskLang integration
            // This would integrate with the Python SDK
            
            g_initialized = true;
            
            std::cout << "TuskLang native plugin initialized: " << configPath << std::endl;
            return 0; // Success
        }}
        catch (const std::exception& e) {{
            std::cerr << "TuskLang initialization error: " << e.what() << std::endl;
            g_stats.errors++;
            return -1; // Error
        }}
    }}
    
    EXPORT char* ExecuteOperation(const char* operation, const char* context) {{
        std::lock_guard<std::mutex> lock(g_mutex);
        
        if (!g_initialized) {{
            g_stats.errors++;
            return strdup("{\\"success\\": false, \\"error\\": \\"TuskLang not initialized\\"}");
        }}
        
        try {{
            g_stats.operations_executed++;
            g_stats.native_calls++;
            
            // Simulate operation execution
            auto start = std::chrono::high_resolution_clock::now();
            
            std::string op_str(operation);
            std::string ctx_str(context);
            
            // Basic operation handling
            std::string result_json;
            
            if (op_str.find("@asset") != std::string::npos) {{
                g_stats.asset_operations++;
                result_json = "{\\"success\\": true, \\"result\\": \\"Asset operation completed\\", \\"type\\": \\"asset\\"}";
            }}
            else if (op_str.find("@gameobject") != std::string::npos) {{
                result_json = "{\\"success\\": true, \\"result\\": \\"GameObject operation completed\\", \\"type\\": \\"gameobject\\"}";
            }}
            else if (op_str.find("=") != std::string::npos) {{
                // Variable assignment
                size_t eq_pos = op_str.find("=");
                std::string var_name = op_str.substr(0, eq_pos);
                std::string var_value = op_str.substr(eq_pos + 1);
                
                g_context[var_name] = var_value;
                result_json = "{\\"success\\": true, \\"result\\": \\"Variable assigned\\", \\"variable\\": \\"" + var_name + "\\"}";
            }}
            else {{
                result_json = "{\\"success\\": true, \\"result\\": \\"Generic operation completed\\", \\"operation\\": \\"" + op_str + "\\"}";
            }}
            
            auto end = std::chrono::high_resolution_clock::now();
            auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(end - start);
            
            // Add execution time to result
            size_t last_brace = result_json.find_last_of("}}");
            result_json.insert(last_brace, ", \\"execution_time_ms\\": " + std::to_string(duration.count()));
            
            return strdup(result_json.c_str());
        }}
        catch (const std::exception& e) {{
            g_stats.errors++;
            std::string error_json = "{\\"success\\": false, \\"error\\": \\"" + std::string(e.what()) + "\\"}";
            return strdup(error_json.c_str());
        }}
    }}
    
    EXPORT int GetStats(char* buffer, int bufferSize) {{
        std::lock_guard<std::mutex> lock(g_mutex);
        
        try {{
            std::string stats_json = "{{" 
                "\\"operations_executed\\": " + std::to_string(g_stats.operations_executed) + ","
                "\\"asset_operations\\": " + std::to_string(g_stats.asset_operations) + ","
                "\\"native_calls\\": " + std::to_string(g_stats.native_calls) + ","
                "\\"script_executions\\": " + std::to_string(g_stats.script_executions) + ","
                "\\"errors\\": " + std::to_string(g_stats.errors) + ","
                "\\"memory_usage_mb\\": " + std::to_string(g_stats.memory_usage_mb) +
                "}}";
            
            if (stats_json.length() >= static_cast<size_t>(bufferSize)) {{
                return -1; // Buffer too small
            }}
            
            strcpy(buffer, stats_json.c_str());
            return 0; // Success
        }}
        catch (const std::exception& e) {{
            g_stats.errors++;
            return -1; // Error
        }}
    }}
    
    EXPORT void FreeTuskLangString(char* ptr) {{
        if (ptr) {{
            free(ptr);
        }}
    }}
    
    EXPORT void ShutdownTuskLang() {{
        std::lock_guard<std::mutex> lock(g_mutex);
        
        g_context.clear();
        g_initialized = false;
        
        std::cout << "TuskLang native plugin shut down" << std::endl;
    }}
}}
'''
        
        # Write C++ source file
        native_dir = os.path.join(self.plugin_root, 'Native')
        cpp_path = os.path.join(native_dir, f'{self.plugin_name}.cpp')
        
        with open(cpp_path, 'w') as f:
            f.write(cpp_source)
        
        return cpp_path
    
    def _build_windows_plugin(self, cpp_source: str):
        """Build Windows DLL plugin"""
        try:
            native_dir = os.path.dirname(cpp_source)
            output_path = os.path.join(native_dir, f'{self.plugin_name}.dll')
            
            # Use cl.exe if available, otherwise skip native build
            compile_cmd = [
                'cl.exe',
                '/LD',  # Build as DLL
                '/EHsc',  # Exception handling
                cpp_source,
                f'/Fe:{output_path}'
            ]
            
            result = subprocess.run(compile_cmd, capture_output=True, text=True)
            if result.returncode == 0:
                self.native_plugin_path = output_path
                self.logger.info(f"Windows plugin built: {output_path}")
            else:
                self.logger.warning(f"Windows plugin build failed: {result.stderr}")
                
        except FileNotFoundError:
            self.logger.warning("Visual Studio compiler not found - skipping native build")
    
    def _build_macos_plugin(self, cpp_source: str):
        """Build macOS bundle plugin"""
        try:
            native_dir = os.path.dirname(cpp_source)
            output_path = os.path.join(native_dir, f'{self.plugin_name}.bundle')
            
            compile_cmd = [
                'clang++',
                '-shared',
                '-fPIC',
                cpp_source,
                '-o', output_path
            ]
            
            result = subprocess.run(compile_cmd, capture_output=True, text=True)
            if result.returncode == 0:
                self.native_plugin_path = output_path
                self.logger.info(f"macOS plugin built: {output_path}")
            else:
                self.logger.warning(f"macOS plugin build failed: {result.stderr}")
                
        except FileNotFoundError:
            self.logger.warning("clang++ not found - skipping native build")
    
    def _build_linux_plugin(self, cpp_source: str):
        """Build Linux shared library plugin"""
        try:
            native_dir = os.path.dirname(cpp_source)
            output_path = os.path.join(native_dir, f'lib{self.plugin_name}.so')
            
            compile_cmd = [
                'g++',
                '-shared',
                '-fPIC',
                cpp_source,
                '-o', output_path
            ]
            
            result = subprocess.run(compile_cmd, capture_output=True, text=True)
            if result.returncode == 0:
                self.native_plugin_path = output_path
                self.logger.info(f"Linux plugin built: {output_path}")
            else:
                self.logger.warning(f"Linux plugin build failed: {result.stderr}")
                
        except FileNotFoundError:
            self.logger.warning("g++ not found - skipping native build")
    
    def _setup_asset_pipeline(self):
        """Set up Unity asset pipeline integration"""
        # Create asset importer for .tsk files
        importer_code = '''
using UnityEngine;
using UnityEditor;

[CustomEditor(typeof(TextAsset))]
public class TuskLangAssetImporter : Editor
{
    public override void OnInspectorGUI()
    {
        TextAsset asset = (TextAsset)target;
        
        if (AssetDatabase.GetAssetPath(asset).EndsWith(".tsk"))
        {
            EditorGUILayout.HelpBox("TuskLang Configuration File", MessageType.Info);
            
            if (GUILayout.Button("Validate TuskLang"))
            {
                ValidateTuskLangAsset(asset);
            }
            
            if (GUILayout.Button("Execute TuskLang"))
            {
                ExecuteTuskLangAsset(asset);
            }
        }
        
        base.OnInspectorGUI();
    }
    
    void ValidateTuskLangAsset(TextAsset asset)
    {
        // Validate TuskLang syntax
        Debug.Log($"Validating TuskLang asset: {AssetDatabase.GetAssetPath(asset)}");
    }
    
    void ExecuteTuskLangAsset(TextAsset asset)
    {
        // Execute TuskLang content
        var manager = TuskLangManager.Instance;
        if (manager != null)
        {
            manager.Execute(asset.text);
        }
    }
}
'''
        
        editor_path = os.path.join(self.plugin_root, 'Editor', 'TuskLangAssetImporter.cs')
        with open(editor_path, 'w') as f:
            f.write(importer_code)
    
    async def execute_operation(self, operation: str, context: Optional[Dict] = None,
                              target: str = 'runtime') -> Dict[str, Any]:
        """Execute TuskLang operation in Unity context"""
        start_time = datetime.now()
        self.stats['operations_executed'] += 1
        
        try:
            context = context or {}
            
            # Add Unity-specific context
            unity_context = {
                **context,
                'unity_version': self.unity_version,
                'target_platforms': self.target_platforms,
                'plugin_path': self.plugin_root
            }
            
            # Route to appropriate execution context
            if target == 'asset':
                result = await self._execute_asset_operation(operation, unity_context)
                self.stats['asset_operations'] += 1
            elif target == 'native':
                result = await self._execute_native_operation(operation, unity_context)
                self.stats['native_calls'] += 1
            elif target == 'script':
                result = await self._execute_script_operation(operation, unity_context)
                self.stats['script_executions'] += 1
            else:
                result = await self._execute_runtime_operation(operation, unity_context)
            
            execution_time = (datetime.now() - start_time).total_seconds() * 1000
            
            return {
                'success': True,
                'result': result,
                'execution_time_ms': execution_time,
                'target': target,
                'unity_context': True,
                'timestamp': datetime.now().isoformat()
            }
            
        except Exception as e:
            self.stats['errors'] += 1
            execution_time = (datetime.now() - start_time).total_seconds() * 1000
            
            self.logger.error(f"Unity operation failed: {e}")
            
            return {
                'success': False,
                'error': str(e),
                'error_type': type(e).__name__,
                'execution_time_ms': execution_time,
                'target': target,
                'unity_context': True,
                'timestamp': datetime.now().isoformat()
            }
    
    async def _execute_asset_operation(self, operation: str, context: Dict) -> Any:
        """Execute asset pipeline operations"""
        # Parse asset operation: @asset("action", "path")
        import re
        match = re.match(r'@asset\("([^"]+)"\s*,\s*"([^"]+)"\)', operation)
        
        if not match:
            raise ValueError("Invalid @asset operation syntax")
        
        action = match.group(1)
        asset_path = match.group(2)
        
        # Simulate asset operations
        asset_result = {
            'action': action,
            'asset_path': asset_path,
            'asset_type': self._get_asset_type(asset_path),
            'processed': True,
            'unity_version': self.unity_version
        }
        
        if action == 'validate':
            asset_result['valid'] = True
            asset_result['issues'] = []
        elif action == 'optimize':
            asset_result['optimized'] = True
            asset_result['size_reduction'] = '15%'
        elif action == 'convert':
            asset_result['converted'] = True
            asset_result['output_format'] = 'optimized'
        
        return asset_result
    
    def _get_asset_type(self, asset_path: str) -> str:
        """Determine Unity asset type from file extension"""
        extension = os.path.splitext(asset_path)[1].lower()
        
        asset_types = {
            '.png': 'Texture2D',
            '.jpg': 'Texture2D',
            '.fbx': 'Model',
            '.obj': 'Model',
            '.wav': 'AudioClip',
            '.mp3': 'AudioClip',
            '.tsk': 'TuskLangAsset',
            '.cs': 'MonoScript',
            '.prefab': 'Prefab'
        }
        
        return asset_types.get(extension, 'Unknown')
    
    async def _execute_native_operation(self, operation: str, context: Dict) -> Any:
        """Execute operations using native plugin"""
        if not self.native_plugin_path:
            raise RuntimeError("Native plugin not available")
        
        # Simulate native plugin call
        native_result = {
            'operation': operation,
            'context': context,
            'native_plugin': self.native_plugin_path,
            'executed_via': 'native_call',
            'performance': 'optimized'
        }
        
        return native_result
    
    async def _execute_script_operation(self, operation: str, context: Dict) -> Any:
        """Execute C# script operations"""
        # Parse script operation: @script("method", args)
        import re
        match = re.match(r'@script\("([^"]+)"(?:\s*,\s*(.+))?\)', operation)
        
        if not match:
            raise ValueError("Invalid @script operation syntax")
        
        method = match.group(1)
        args = match.group(2)
        
        script_result = {
            'method': method,
            'args': args,
            'executed_via': 'csharp_wrapper',
            'managed_code': True
        }
        
        return script_result
    
    async def _execute_runtime_operation(self, operation: str, context: Dict) -> Any:
        """Execute runtime operations"""
        try:
            # Try to use advanced operators if available
            from ...core.base.advanced_operators_integration import AdvancedOperatorIntegration
            
            operator_integration = AdvancedOperatorIntegration()
            
            if operation.startswith('@gameobject'):
                return await self._execute_gameobject_operation(operation, context)
            elif operation.startswith('@component'):
                return await self._execute_component_operation(operation, context)
            elif operation.startswith('@scene'):
                return await self._execute_scene_operation(operation, context)
            else:
                # Delegate to advanced operators
                return await self._delegate_to_advanced_operators(
                    operator_integration, operation, context
                )
                
        except ImportError:
            # Fallback to basic execution
            return await self._execute_basic_unity_operation(operation, context)
    
    async def _execute_gameobject_operation(self, operation: str, context: Dict) -> Any:
        """Execute GameObject operations"""
        # Parse: @gameobject("action", "name")
        import re
        match = re.match(r'@gameobject\("([^"]+)"\s*,\s*"([^"]+)"\)', operation)
        
        if not match:
            raise ValueError("Invalid @gameobject operation syntax")
        
        action = match.group(1)
        obj_name = match.group(2)
        
        gameobject_result = {
            'action': action,
            'object_name': obj_name,
            'object_type': 'GameObject',
            'scene': context.get('scene', 'current')
        }
        
        if action == 'find':
            gameobject_result['found'] = True
            gameobject_result['instance_id'] = 12345
        elif action == 'create':
            gameobject_result['created'] = True
            gameobject_result['position'] = [0, 0, 0]
        elif action == 'destroy':
            gameobject_result['destroyed'] = True
        
        return gameobject_result
    
    def create_unity_package(self, output_path: str) -> str:
        """Create Unity package for distribution"""
        try:
            package_json = {
                'name': 'com.tusklang.unity',
                'version': '1.0.0',
                'displayName': 'TuskLang Unity Integration',
                'description': 'TuskLang Python SDK integration for Unity',
                'unity': self.unity_version,
                'keywords': ['tusklang', 'python', 'integration'],
                'author': {
                    'name': 'TuskLang Team',
                    'email': 'team@tusklang.io'
                },
                'dependencies': {}
            }
            
            package_dir = os.path.join(output_path, 'com.tusklang.unity')
            os.makedirs(package_dir, exist_ok=True)
            
            # Copy plugin files
            if os.path.exists(self.plugin_root):
                shutil.copytree(self.plugin_root, 
                              os.path.join(package_dir, 'Runtime'),
                              dirs_exist_ok=True)
            
            # Write package.json
            with open(os.path.join(package_dir, 'package.json'), 'w') as f:
                json.dump(package_json, f, indent=2)
            
            # Create README
            readme_content = f'''
# TuskLang Unity Integration

This package provides Unity integration for the TuskLang Python SDK.

## Installation

1. Add this package to your Unity project
2. Configure the TuskLangManager in your scene
3. Start executing TuskLang operations!

## Usage

```csharp
// Get TuskLang manager
var tuskLang = TuskLangManager.Instance;

// Execute operation
var result = await tuskLang.ExecuteAsync("test = \\"Hello Unity\\"");

// Process assets
var assetResult = await tuskLang.ProcessAsset("path/to/asset.png", "optimize");
```

## Requirements

- Unity {self.unity_version} or later
- TuskLang Python SDK

For more information, visit: https://tusklang.io/unity
'''
            
            with open(os.path.join(package_dir, 'README.md'), 'w') as f:
                f.write(readme_content)
            
            self.logger.info(f"Unity package created: {package_dir}")
            return package_dir
            
        except Exception as e:
            self.logger.error(f"Unity package creation failed: {e}")
            raise
    
    def get_stats(self) -> Dict[str, Any]:
        """Get Unity integration statistics"""
        return {
            **self.stats,
            'unity_version': self.unity_version,
            'target_platforms': self.target_platforms,
            'plugin_root': self.plugin_root,
            'native_plugin_available': self.native_plugin_path is not None,
            'csharp_wrapper_available': self.csharp_wrapper_path is not None,
            'unity_project_path': self.unity_project_path
        }


# Convenience functions
async def execute_in_unity(operation: str, context: Optional[Dict] = None,
                          target: str = 'runtime',
                          config: Optional[Dict] = None) -> Dict[str, Any]:
    """
    Convenience function to execute TuskLang operation in Unity environment
    """
    integration = UnityIntegration(config)
    return await integration.execute_operation(operation, context, target)


def create_unity_plugin(unity_project_path: str, plugin_name: str = 'TuskLangPlugin') -> str:
    """Create Unity plugin in specified project"""
    integration = UnityIntegration({
        'unity_project_path': unity_project_path,
        'plugin_name': plugin_name
    })
    
    return integration.plugin_root 