#!/usr/bin/env python3
"""
Advanced Performance Optimization Engine for TuskLang Python SDK
Goal 7.1 Implementation - Performance Optimization System

Features:
- Intelligent caching with TTL and LRU eviction
- Memory management and garbage collection optimization
- Performance profiling and bottleneck detection
- Async operation optimization
- Resource pooling and connection management
"""

import time
import threading
import weakref
import gc
import asyncio
import functools
import logging
from typing import Any, Dict, List, Optional, Callable, Union
from collections import OrderedDict, defaultdict
from dataclasses import dataclass, field
from contextlib import contextmanager
import psutil
import tracemalloc

logger = logging.getLogger(__name__)

@dataclass
class CacheEntry:
    """Cache entry with metadata for intelligent eviction"""
    value: Any
    created_at: float
    last_accessed: float
    access_count: int = 0
    size_bytes: int = 0
    ttl: Optional[float] = None

class PerformanceEngine:
    """Advanced performance optimization engine for TSK SDK"""
    
    def __init__(self, max_cache_size: int = 1000, max_memory_mb: int = 512):
        self.max_cache_size = max_cache_size
        self.max_memory_mb = max_memory_mb
        self.cache: OrderedDict[str, CacheEntry] = OrderedDict()
        self.cache_lock = threading.RLock()
        self.profiling_data: Dict[str, List[float]] = defaultdict(list)
        self.memory_usage: List[float] = []
        self.performance_metrics: Dict[str, Any] = {}
        self.resource_pools: Dict[str, List[Any]] = defaultdict(list)
        self.pool_locks: Dict[str, threading.Lock] = defaultdict(threading.Lock)
        
        # Start memory monitoring
        self._start_memory_monitoring()
        
    def _start_memory_monitoring(self):
        """Start background memory monitoring"""
        def monitor_memory():
            while True:
                try:
                    process = psutil.Process()
                    memory_mb = process.memory_info().rss / 1024 / 1024
                    self.memory_usage.append(memory_mb)
                    
                    # Keep only last 1000 measurements
                    if len(self.memory_usage) > 1000:
                        self.memory_usage = self.memory_usage[-1000:]
                    
                    # Trigger cleanup if memory usage is high
                    if memory_mb > self.max_memory_mb * 0.8:
                        self._cleanup_memory()
                        
                    time.sleep(5)  # Check every 5 seconds
                except Exception as e:
                    logger.error(f"Memory monitoring error: {e}")
                    time.sleep(10)
        
        monitor_thread = threading.Thread(target=monitor_memory, daemon=True)
        monitor_thread.start()
    
    def cache_get(self, key: str, default: Any = None) -> Any:
        """Get value from cache with intelligent access tracking"""
        with self.cache_lock:
            if key in self.cache:
                entry = self.cache[key]
                
                # Check TTL
                if entry.ttl and time.time() - entry.created_at > entry.ttl:
                    del self.cache[key]
                    return default
                
                # Update access metadata
                entry.last_accessed = time.time()
                entry.access_count += 1
                
                # Move to end (LRU)
                self.cache.move_to_end(key)
                return entry.value
            
            return default
    
    def cache_set(self, key: str, value: Any, ttl: Optional[float] = None, 
                  size_bytes: Optional[int] = None) -> None:
        """Set value in cache with intelligent eviction"""
        with self.cache_lock:
            # Calculate size if not provided
            if size_bytes is None:
                try:
                    size_bytes = len(str(value).encode('utf-8'))
                except:
                    size_bytes = 1024  # Default size
            
            # Create cache entry
            entry = CacheEntry(
                value=value,
                created_at=time.time(),
                last_accessed=time.time(),
                ttl=ttl,
                size_bytes=size_bytes
            )
            
            # Evict if key already exists
            if key in self.cache:
                del self.cache[key]
            
            # Add new entry
            self.cache[key] = entry
            
            # Evict if cache is full
            if len(self.cache) > self.max_cache_size:
                self._evict_entries()
    
    def _evict_entries(self):
        """Intelligent cache eviction based on access patterns and TTL"""
        # Remove expired entries first
        current_time = time.time()
        expired_keys = [
            key for key, entry in self.cache.items()
            if entry.ttl and current_time - entry.created_at > entry.ttl
        ]
        
        for key in expired_keys:
            del self.cache[key]
        
        # If still over limit, remove least recently used
        while len(self.cache) > self.max_cache_size:
            # Remove oldest entry
            self.cache.popitem(last=False)
    
    def _cleanup_memory(self):
        """Aggressive memory cleanup when usage is high"""
        logger.info("Performing memory cleanup")
        
        # Clear cache
        with self.cache_lock:
            self.cache.clear()
        
        # Force garbage collection
        gc.collect()
        
        # Clear profiling data
        self.profiling_data.clear()
        
        # Clear old memory usage data
        if len(self.memory_usage) > 100:
            self.memory_usage = self.memory_usage[-50:]
    
    @contextmanager
    def profile_operation(self, operation_name: str):
        """Context manager for profiling operations"""
        start_time = time.time()
        start_memory = tracemalloc.get_traced_memory()[0] if tracemalloc.is_tracing() else 0
        
        try:
            yield
        finally:
            end_time = time.time()
            end_memory = tracemalloc.get_traced_memory()[0] if tracemalloc.is_tracing() else 0
            
            duration = end_time - start_time
            memory_delta = end_memory - start_memory
            
            self.profiling_data[operation_name].append(duration)
            
            # Keep only last 100 measurements per operation
            if len(self.profiling_data[operation_name]) > 100:
                self.profiling_data[operation_name] = self.profiling_data[operation_name][-100:]
            
            logger.debug(f"Operation '{operation_name}' took {duration:.4f}s, memory delta: {memory_delta} bytes")
    
    def get_performance_metrics(self) -> Dict[str, Any]:
        """Get comprehensive performance metrics"""
        metrics = {
            'cache': {
                'size': len(self.cache),
                'max_size': self.max_cache_size,
                'hit_rate': self._calculate_cache_hit_rate(),
                'memory_usage_mb': sum(entry.size_bytes for entry in self.cache.values()) / 1024 / 1024
            },
            'memory': {
                'current_mb': self.memory_usage[-1] if self.memory_usage else 0,
                'peak_mb': max(self.memory_usage) if self.memory_usage else 0,
                'average_mb': sum(self.memory_usage) / len(self.memory_usage) if self.memory_usage else 0
            },
            'operations': {}
        }
        
        # Calculate operation statistics
        for op_name, durations in self.profiling_data.items():
            if durations:
                metrics['operations'][op_name] = {
                    'count': len(durations),
                    'average_ms': sum(durations) / len(durations) * 1000,
                    'min_ms': min(durations) * 1000,
                    'max_ms': max(durations) * 1000,
                    'total_ms': sum(durations) * 1000
                }
        
        return metrics
    
    def _calculate_cache_hit_rate(self) -> float:
        """Calculate cache hit rate based on access patterns"""
        total_accesses = sum(entry.access_count for entry in self.cache.values())
        if total_accesses == 0:
            return 0.0
        
        # This is a simplified calculation - in a real system you'd track misses too
        return min(0.95, total_accesses / (total_accesses + len(self.cache)))
    
    def optimize_async_operations(self, coro_func: Callable) -> Callable:
        """Decorator to optimize async operations with caching and profiling"""
        @functools.wraps(coro_func)
        async def optimized_wrapper(*args, **kwargs):
            # Create cache key from function name and arguments
            cache_key = f"{coro_func.__name__}:{hash(str(args) + str(sorted(kwargs.items())))}"
            
            # Try to get from cache first
            cached_result = self.cache_get(cache_key)
            if cached_result is not None:
                return cached_result
            
            # Profile the operation
            with self.profile_operation(f"async_{coro_func.__name__}"):
                result = await coro_func(*args, **kwargs)
            
            # Cache the result
            self.cache_set(cache_key, result, ttl=300)  # 5 minute TTL
            
            return result
        
        return optimized_wrapper
    
    def get_resource_pool(self, pool_name: str, factory_func: Callable, max_size: int = 10):
        """Get or create resource from pool"""
        with self.pool_locks[pool_name]:
            if self.resource_pools[pool_name]:
                return self.resource_pools[pool_name].pop()
            else:
                return factory_func()
    
    def return_resource_to_pool(self, pool_name: str, resource: Any, max_size: int = 10):
        """Return resource to pool"""
        with self.pool_locks[pool_name]:
            if len(self.resource_pools[pool_name]) < max_size:
                self.resource_pools[pool_name].append(resource)
    
    def clear_all_caches(self):
        """Clear all caches and reset performance data"""
        with self.cache_lock:
            self.cache.clear()
        
        self.profiling_data.clear()
        self.memory_usage.clear()
        self.resource_pools.clear()
        
        logger.info("All caches and performance data cleared")

# Global performance engine instance
performance_engine = PerformanceEngine()

def optimize_operation(operation_name: str = None):
    """Decorator for profiling and optimizing operations"""
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            op_name = operation_name or func.__name__
            with performance_engine.profile_operation(op_name):
                return func(*args, **kwargs)
        return wrapper
    return decorator

def optimize_async_operation(operation_name: str = None):
    """Decorator for profiling and optimizing async operations"""
    def decorator(func):
        @functools.wraps(func)
        async def wrapper(*args, **kwargs):
            op_name = operation_name or func.__name__
            with performance_engine.profile_operation(f"async_{op_name}"):
                return await func(*args, **kwargs)
        return wrapper
    return decorator 