"""
Temporal Operator - Workflow Engine Integration
Production-ready Temporal integration with workflow orchestration, activity scheduling, and worker management.
"""

import asyncio
import json
import logging
import time
import uuid
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from typing import Any, Callable, Dict, List, Optional, Union
from enum import Enum

# Temporal Support
try:
    from temporalio import activity, workflow
    from temporalio.client import Client, WorkflowHandle
    from temporalio.worker import Worker
    from temporalio.common import RetryPolicy
    from temporalio.exceptions import (
        WorkflowAlreadyStartedError, WorkflowContinuedAsNewError,
        ActivityError, ChildWorkflowError, CancelledError, TerminatedError
    )
    TEMPORAL_AVAILABLE = True
except ImportError:
    TEMPORAL_AVAILABLE = False
    print("temporalio library not available. @temporal operator will be limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class WorkflowStatus(Enum):
    """Workflow status enumeration."""
    RUNNING = "RUNNING"
    COMPLETED = "COMPLETED"
    FAILED = "FAILED"
    CANCELED = "CANCELED"
    TERMINATED = "TERMINATED"
    CONTINUED_AS_NEW = "CONTINUED_AS_NEW"
    TIMED_OUT = "TIMED_OUT"

@dataclass
class TemporalConfig:
    """Temporal connection configuration."""
    target_host: str = "localhost:7233"
    namespace: str = "default"
    data_converter: Optional[Any] = None
    interceptors: List[Any] = field(default_factory=list)
    identity: Optional[str] = None
    tls: Optional[Any] = None
    rpc_metadata: Dict[str, str] = field(default_factory=dict)
    rpc_timeout: timedelta = timedelta(seconds=30)
    api_key: Optional[str] = None

@dataclass
class RetryPolicy:
    """Temporal retry policy configuration."""
    initial_interval: timedelta = timedelta(seconds=1)
    maximum_interval: timedelta = timedelta(seconds=100)
    maximum_attempts: int = 0
    backoff_coefficient: float = 2.0
    non_retryable_error_types: List[str] = field(default_factory=list)

@dataclass
class WorkflowDefinition:
    """Temporal workflow definition."""
    name: str
    workflow_class: Any
    task_queue: str
    workflow_id: Optional[str] = None
    workflow_type: Optional[str] = None
    execution_timeout: Optional[timedelta] = None
    run_timeout: Optional[timedelta] = None
    task_timeout: Optional[timedelta] = None
    retry_policy: Optional[RetryPolicy] = None
    cron_schedule: Optional[str] = None
    memo: Dict[str, Any] = field(default_factory=dict)
    search_attributes: Dict[str, Any] = field(default_factory=dict)

@dataclass
class ActivityDefinition:
    """Temporal activity definition."""
    name: str
    activity_function: Callable
    task_queue: str
    schedule_to_close_timeout: Optional[timedelta] = None
    schedule_to_start_timeout: Optional[timedelta] = None
    start_to_close_timeout: Optional[timedelta] = None
    heartbeat_timeout: Optional[timedelta] = None
    retry_policy: Optional[RetryPolicy] = None

@dataclass
class WorkflowExecution:
    """Workflow execution information."""
    workflow_id: str
    run_id: str
    workflow_type: str
    status: WorkflowStatus
    start_time: datetime
    close_time: Optional[datetime] = None
    execution_time: Optional[timedelta] = None
    result: Optional[Any] = None
    failure: Optional[str] = None
    history_length: int = 0

@dataclass
class ActivityExecution:
    """Activity execution information."""
    activity_id: str
    activity_type: str
    workflow_id: str
    run_id: str
    status: str
    start_time: datetime
    close_time: Optional[datetime] = None
    result: Optional[Any] = None
    failure: Optional[str] = None
    attempt: int = 1
    max_attempts: int = 1

@dataclass
class WorkflowSignal:
    """Workflow signal definition."""
    signal_name: str
    args: List[Any] = field(default_factory=list)
    kwargs: Dict[str, Any] = field(default_factory=dict)

@dataclass
class WorkflowQuery:
    """Workflow query definition."""
    query_name: str
    args: List[Any] = field(default_factory=list)
    kwargs: Dict[str, Any] = field(default_factory=dict)

class TemporalWorkflowManager:
    """Manages Temporal workflows and activities."""
    
    def __init__(self, client: 'Client'):
        self.client = client
        self.registered_workflows = {}
        self.registered_activities = {}
        self.running_workflows = {}
        self.workflow_handles = {}
    
    def register_workflow(self, definition: WorkflowDefinition):
        """Register workflow definition."""
        self.registered_workflows[definition.name] = definition
        logger.info(f"Registered workflow: {definition.name}")
    
    def register_activity(self, definition: ActivityDefinition):
        """Register activity definition."""
        self.registered_activities[definition.name] = definition
        logger.info(f"Registered activity: {definition.name}")
    
    async def start_workflow(self, workflow_name: str, *args, **kwargs) -> Any:
        """Start workflow execution."""
        if workflow_name not in self.registered_workflows:
            raise ValueError(f"Workflow {workflow_name} not registered")
        
        definition = self.registered_workflows[workflow_name]
        
        # Generate workflow ID if not provided
        workflow_id = definition.workflow_id or f"{workflow_name}-{uuid.uuid4()}"
        
        try:
            # Start workflow
            handle = await self.client.start_workflow(
                definition.workflow_class,
                *args,
                id=workflow_id,
                task_queue=definition.task_queue,
                execution_timeout=definition.execution_timeout,
                run_timeout=definition.run_timeout,
                task_timeout=definition.task_timeout,
                retry_policy=definition.retry_policy,
                cron_schedule=definition.cron_schedule,
                memo=definition.memo,
                search_attributes=definition.search_attributes,
                **kwargs
            )
            
            # Store handle
            self.workflow_handles[workflow_id] = handle
            self.running_workflows[workflow_id] = {
                'handle': handle,
                'start_time': datetime.now(),
                'definition': definition
            }
            
            logger.info(f"Started workflow: {workflow_id}")
            return handle
            
        except WorkflowAlreadyStartedError:
            # Get existing handle
            handle = self.client.get_workflow_handle(workflow_id)
            self.workflow_handles[workflow_id] = handle
            return handle
    
    async def get_workflow_result(self, workflow_id: str, timeout: Optional[timedelta] = None) -> Any:
        """Get workflow result."""
        if workflow_id not in self.workflow_handles:
            raise ValueError(f"Workflow {workflow_id} not found")
        
        handle = self.workflow_handles[workflow_id]
        
        try:
            if timeout:
                result = await asyncio.wait_for(handle.result(), timeout=timeout.total_seconds())
            else:
                result = await handle.result()
            
            # Update running workflows
            if workflow_id in self.running_workflows:
                self.running_workflows[workflow_id]['end_time'] = datetime.now()
                self.running_workflows[workflow_id]['result'] = result
            
            return result
            
        except Exception as e:
            logger.error(f"Error getting workflow result {workflow_id}: {str(e)}")
            raise
    
    async def signal_workflow(self, workflow_id: str, signal: WorkflowSignal):
        """Send signal to workflow."""
        if workflow_id not in self.workflow_handles:
            raise ValueError(f"Workflow {workflow_id} not found")
        
        handle = self.workflow_handles[workflow_id]
        
        try:
            await handle.signal(signal.signal_name, *signal.args, **signal.kwargs)
            logger.info(f"Sent signal {signal.signal_name} to workflow {workflow_id}")
            
        except Exception as e:
            logger.error(f"Error signaling workflow {workflow_id}: {str(e)}")
            raise
    
    async def query_workflow(self, workflow_id: str, query: WorkflowQuery) -> Any:
        """Query workflow."""
        if workflow_id not in self.workflow_handles:
            raise ValueError(f"Workflow {workflow_id} not found")
        
        handle = self.workflow_handles[workflow_id]
        
        try:
            result = await handle.query(query.query_name, *query.args, **query.kwargs)
            return result
            
        except Exception as e:
            logger.error(f"Error querying workflow {workflow_id}: {str(e)}")
            raise
    
    async def cancel_workflow(self, workflow_id: str) -> bool:
        """Cancel workflow."""
        if workflow_id not in self.workflow_handles:
            raise ValueError(f"Workflow {workflow_id} not found")
        
        handle = self.workflow_handles[workflow_id]
        
        try:
            await handle.cancel()
            
            # Update status
            if workflow_id in self.running_workflows:
                self.running_workflows[workflow_id]['end_time'] = datetime.now()
                self.running_workflows[workflow_id]['status'] = 'CANCELLED'
            
            logger.info(f"Cancelled workflow {workflow_id}")
            return True
            
        except Exception as e:
            logger.error(f"Error cancelling workflow {workflow_id}: {str(e)}")
            raise
    
    async def terminate_workflow(self, workflow_id: str, reason: str = "Terminated by operator") -> bool:
        """Terminate workflow."""
        if workflow_id not in self.workflow_handles:
            raise ValueError(f"Workflow {workflow_id} not found")
        
        handle = self.workflow_handles[workflow_id]
        
        try:
            await handle.terminate(reason)
            
            # Update status
            if workflow_id in self.running_workflows:
                self.running_workflows[workflow_id]['end_time'] = datetime.now()
                self.running_workflows[workflow_id]['status'] = 'TERMINATED'
                self.running_workflows[workflow_id]['reason'] = reason
            
            logger.info(f"Terminated workflow {workflow_id}")
            return True
            
        except Exception as e:
            logger.error(f"Error terminating workflow {workflow_id}: {str(e)}")
            raise
    
    async def get_workflow_history(self, workflow_id: str) -> List[Dict[str, Any]]:
        """Get workflow history."""
        if workflow_id not in self.workflow_handles:
            raise ValueError(f"Workflow {workflow_id} not found")
        
        handle = self.workflow_handles[workflow_id]
        
        try:
            history = []
            async for event in handle.fetch_history():
                history.append({
                    'event_id': event.event_id,
                    'event_time': event.event_time,
                    'event_type': event.event_type,
                    'attributes': event.attributes
                })
            
            return history
            
        except Exception as e:
            logger.error(f"Error getting workflow history {workflow_id}: {str(e)}")
            raise

class TemporalWorkerManager:
    """Manages Temporal workers."""
    
    def __init__(self, client: 'Client'):
        self.client = client
        self.workers = {}
        self.worker_stats = {}
    
    async def create_worker(self, task_queue: str, workflows: List[Any] = None, 
                           activities: List[Any] = None, 
                           max_concurrent_activities: int = 100,
                           max_concurrent_local_activities: int = 100,
                           max_concurrent_workflows: int = 100) -> Any:
        """Create and configure worker."""
        worker = Worker(
            self.client,
            task_queue=task_queue,
            workflows=workflows or [],
            activities=activities or [],
            max_concurrent_activities=max_concurrent_activities,
            max_concurrent_local_activities=max_concurrent_local_activities,
            max_concurrent_workflows=max_concurrent_workflows
        )
        
        self.workers[task_queue] = worker
        self.worker_stats[task_queue] = {
            'created_time': datetime.now(),
            'workflows_processed': 0,
            'activities_processed': 0,
            'status': 'CREATED'
        }
        
        logger.info(f"Created worker for task queue: {task_queue}")
        return worker
    
    async def start_worker(self, task_queue: str) -> bool:
        """Start worker."""
        if task_queue not in self.workers:
            raise ValueError(f"Worker for task queue {task_queue} not found")
        
        worker = self.workers[task_queue]
        
        try:
            # Start worker in background
            asyncio.create_task(worker.run())
            
            self.worker_stats[task_queue]['status'] = 'RUNNING'
            self.worker_stats[task_queue]['start_time'] = datetime.now()
            
            logger.info(f"Started worker for task queue: {task_queue}")
            return True
            
        except Exception as e:
            logger.error(f"Error starting worker {task_queue}: {str(e)}")
            self.worker_stats[task_queue]['status'] = 'ERROR'
            self.worker_stats[task_queue]['error'] = str(e)
            raise
    
    async def stop_worker(self, task_queue: str) -> bool:
        """Stop worker."""
        if task_queue not in self.workers:
            raise ValueError(f"Worker for task queue {task_queue} not found")
        
        worker = self.workers[task_queue]
        
        try:
            worker.shutdown()
            
            self.worker_stats[task_queue]['status'] = 'STOPPED'
            self.worker_stats[task_queue]['stop_time'] = datetime.now()
            
            logger.info(f"Stopped worker for task queue: {task_queue}")
            return True
            
        except Exception as e:
            logger.error(f"Error stopping worker {task_queue}: {str(e)}")
            raise

class TemporalOperator:
    """@temporal operator implementation with full production features."""
    
    def __init__(self):
        self.client: Optional['Client'] = None
        self.workflow_manager: Optional[TemporalWorkflowManager] = None
        self.worker_manager: Optional[TemporalWorkerManager] = None
        self.operation_stats = {
            'workflow_starts': 0,
            'workflow_completions': 0,
            'workflow_failures': 0,
            'signals_sent': 0,
            'queries_executed': 0,
            'activities_executed': 0
        }
        self._executor = ThreadPoolExecutor(max_workers=10)
    
    async def connect(self, config: Optional[TemporalConfig] = None) -> bool:
        """Connect to Temporal service."""
        if not TEMPORAL_AVAILABLE:
            logger.error("Temporal client library not available")
            return False
        
        if config is None:
            config = TemporalConfig()
        
        try:
            # Build client options
            client_options = {
                'target_host': config.target_host,
                'namespace': config.namespace
            }
            
            if config.data_converter:
                client_options['data_converter'] = config.data_converter
            if config.interceptors:
                client_options['interceptors'] = config.interceptors
            if config.identity:
                client_options['identity'] = config.identity
            if config.tls:
                client_options['tls'] = config.tls
            if config.rpc_metadata:
                client_options['rpc_metadata'] = config.rpc_metadata
            if config.rpc_timeout:
                client_options['rpc_timeout'] = config.rpc_timeout
            if config.api_key:
                client_options['api_key'] = config.api_key
            
            # Create client
            self.client = await Client.connect(**client_options)
            
            # Initialize managers
            self.workflow_manager = TemporalWorkflowManager(self.client)
            self.worker_manager = TemporalWorkerManager(self.client)
            
            logger.info(f"Connected to Temporal: {config.target_host}/{config.namespace}")
            return True
            
        except Exception as e:
            logger.error(f"Error connecting to Temporal: {str(e)}")
            return False
    
    # Workflow Operations
    async def register_workflow(self, name: str, workflow_class: Any, task_queue: str, **kwargs) -> bool:
        """Register workflow."""
        if not self.workflow_manager:
            raise RuntimeError("Not connected to Temporal")
        
        definition = WorkflowDefinition(
            name=name,
            workflow_class=workflow_class,
            task_queue=task_queue,
            **kwargs
        )
        
        self.workflow_manager.register_workflow(definition)
        return True
    
    async def register_activity(self, name: str, activity_function: Callable, task_queue: str, **kwargs) -> bool:
        """Register activity."""
        if not self.workflow_manager:
            raise RuntimeError("Not connected to Temporal")
        
        definition = ActivityDefinition(
            name=name,
            activity_function=activity_function,
            task_queue=task_queue,
            **kwargs
        )
        
        self.workflow_manager.register_activity(definition)
        return True
    
    async def start_workflow(self, workflow_name: str, *args, **kwargs) -> str:
        """Start workflow execution."""
        if not self.workflow_manager:
            raise RuntimeError("Not connected to Temporal")
        
        try:
            handle = await self.workflow_manager.start_workflow(workflow_name, *args, **kwargs)
            
            self.operation_stats['workflow_starts'] += 1
            return handle.id
            
        except Exception as e:
            self.operation_stats['workflow_failures'] += 1
            logger.error(f"Error starting workflow {workflow_name}: {str(e)}")
            raise
    
    async def get_workflow_result(self, workflow_id: str, timeout: Optional[int] = None) -> Any:
        """Get workflow result."""
        if not self.workflow_manager:
            raise RuntimeError("Not connected to Temporal")
        
        timeout_delta = timedelta(seconds=timeout) if timeout else None
        
        try:
            result = await self.workflow_manager.get_workflow_result(workflow_id, timeout_delta)
            self.operation_stats['workflow_completions'] += 1
            return result
            
        except Exception as e:
            self.operation_stats['workflow_failures'] += 1
            logger.error(f"Error getting workflow result {workflow_id}: {str(e)}")
            raise
    
    async def signal_workflow(self, workflow_id: str, signal_name: str, *args, **kwargs) -> bool:
        """Send signal to workflow."""
        if not self.workflow_manager:
            raise RuntimeError("Not connected to Temporal")
        
        signal = WorkflowSignal(signal_name=signal_name, args=list(args), kwargs=kwargs)
        
        try:
            await self.workflow_manager.signal_workflow(workflow_id, signal)
            self.operation_stats['signals_sent'] += 1
            return True
            
        except Exception as e:
            logger.error(f"Error signaling workflow {workflow_id}: {str(e)}")
            raise
    
    async def query_workflow(self, workflow_id: str, query_name: str, *args, **kwargs) -> Any:
        """Query workflow."""
        if not self.workflow_manager:
            raise RuntimeError("Not connected to Temporal")
        
        query = WorkflowQuery(query_name=query_name, args=list(args), kwargs=kwargs)
        
        try:
            result = await self.workflow_manager.query_workflow(workflow_id, query)
            self.operation_stats['queries_executed'] += 1
            return result
            
        except Exception as e:
            logger.error(f"Error querying workflow {workflow_id}: {str(e)}")
            raise
    
    async def cancel_workflow(self, workflow_id: str) -> bool:
        """Cancel workflow."""
        if not self.workflow_manager:
            raise RuntimeError("Not connected to Temporal")
        
        try:
            success = await self.workflow_manager.cancel_workflow(workflow_id)
            if success:
                self.operation_stats['workflow_completions'] += 1
            return success
            
        except Exception as e:
            logger.error(f"Error cancelling workflow {workflow_id}: {str(e)}")
            raise
    
    async def terminate_workflow(self, workflow_id: str, reason: str = "Terminated by operator") -> bool:
        """Terminate workflow."""
        if not self.workflow_manager:
            raise RuntimeError("Not connected to Temporal")
        
        try:
            success = await self.workflow_manager.terminate_workflow(workflow_id, reason)
            if success:
                self.operation_stats['workflow_completions'] += 1
            return success
            
        except Exception as e:
            logger.error(f"Error terminating workflow {workflow_id}: {str(e)}")
            raise
    
    # Worker Operations
    async def create_worker(self, task_queue: str, workflows: List[Any] = None,
                           activities: List[Any] = None, **kwargs) -> bool:
        """Create worker."""
        if not self.worker_manager:
            raise RuntimeError("Not connected to Temporal")
        
        try:
            await self.worker_manager.create_worker(task_queue, workflows, activities, **kwargs)
            return True
            
        except Exception as e:
            logger.error(f"Error creating worker {task_queue}: {str(e)}")
            raise
    
    async def start_worker(self, task_queue: str) -> bool:
        """Start worker."""
        if not self.worker_manager:
            raise RuntimeError("Not connected to Temporal")
        
        try:
            return await self.worker_manager.start_worker(task_queue)
            
        except Exception as e:
            logger.error(f"Error starting worker {task_queue}: {str(e)}")
            raise
    
    async def stop_worker(self, task_queue: str) -> bool:
        """Stop worker."""
        if not self.worker_manager:
            raise RuntimeError("Not connected to Temporal")
        
        try:
            return await self.worker_manager.stop_worker(task_queue)
            
        except Exception as e:
            logger.error(f"Error stopping worker {task_queue}: {str(e)}")
            raise
    
    # Monitoring and Statistics
    async def list_workflows(self, query: str = "") -> List[WorkflowExecution]:
        """List workflows."""
        if not self.client:
            raise RuntimeError("Not connected to Temporal")
        
        try:
            workflows = []
            async for workflow in self.client.list_workflows(query):
                execution = WorkflowExecution(
                    workflow_id=workflow.id,
                    run_id=workflow.run_id,
                    workflow_type=workflow.workflow_type,
                    status=WorkflowStatus(workflow.status.name),
                    start_time=workflow.start_time,
                    close_time=workflow.close_time,
                    execution_time=workflow.execution_time,
                    history_length=workflow.history_length
                )
                workflows.append(execution)
            
            return workflows
            
        except Exception as e:
            logger.error(f"Error listing workflows: {str(e)}")
            raise
    
    def get_statistics(self) -> Dict[str, Any]:
        """Get operation statistics."""
        stats = {
            'operations': self.operation_stats.copy(),
            'connected': self.client is not None
        }
        
        if self.workflow_manager:
            stats['registered_workflows'] = len(self.workflow_manager.registered_workflows)
            stats['registered_activities'] = len(self.workflow_manager.registered_activities)
            stats['running_workflows'] = len(self.workflow_manager.running_workflows)
        
        if self.worker_manager:
            stats['workers'] = len(self.worker_manager.workers)
            stats['worker_stats'] = self.worker_manager.worker_stats
        
        return stats
    
    async def close(self):
        """Close connections and cleanup."""
        # Stop all workers
        if self.worker_manager:
            for task_queue in list(self.worker_manager.workers.keys()):
                try:
                    await self.worker_manager.stop_worker(task_queue)
                except Exception as e:
                    logger.warning(f"Error stopping worker {task_queue}: {str(e)}")
        
        # Close client
        if self.client:
            await self.client.close()
        
        # Shutdown executor
        self._executor.shutdown(wait=True)
        
        logger.info("Temporal operator closed")

# Export the operator
__all__ = [
    'TemporalOperator', 'TemporalConfig', 'WorkflowDefinition', 'ActivityDefinition',
    'WorkflowExecution', 'ActivityExecution', 'WorkflowSignal', 'WorkflowQuery', 'WorkflowStatus'
] 