"""
TuskLang Python SDK - Role-Based Authorization Manager (g10.2)
Production RBAC system with permissions, policies, and access control
"""

import json
import logging
import uuid
from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from enum import Enum
from typing import Dict, List, Optional, Set, Any, Union, Callable
import fnmatch
import re


class PermissionEffect(Enum):
    ALLOW = "allow"
    DENY = "deny"


class ResourceType(Enum):
    API_ENDPOINT = "api_endpoint"
    DATABASE_TABLE = "database_table"
    FILE_SYSTEM = "file_system"
    SERVICE = "service"
    FEATURE = "feature"
    DATA_FIELD = "data_field"
    CUSTOM = "custom"


@dataclass
class Permission:
    """Individual permission"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    description: str = ""
    resource_type: ResourceType = ResourceType.API_ENDPOINT
    resource_pattern: str = "*"  # Glob pattern for resource matching
    actions: List[str] = field(default_factory=list)  # e.g., ["read", "write", "delete"]
    effect: PermissionEffect = PermissionEffect.ALLOW
    conditions: Dict[str, Any] = field(default_factory=dict)
    created_at: datetime = field(default_factory=datetime.now)
    created_by: str = ""
    is_active: bool = True


@dataclass
class Role:
    """Role containing permissions"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    description: str = ""
    permissions: List[str] = field(default_factory=list)  # Permission IDs
    parent_roles: List[str] = field(default_factory=list)  # Role inheritance
    metadata: Dict[str, Any] = field(default_factory=dict)
    created_at: datetime = field(default_factory=datetime.now)
    created_by: str = ""
    is_active: bool = True
    is_system_role: bool = False  # System roles cannot be deleted


@dataclass
class Policy:
    """Access control policy"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    description: str = ""
    version: str = "1.0"
    statements: List[Dict[str, Any]] = field(default_factory=list)
    conditions: Dict[str, Any] = field(default_factory=dict)
    effect: PermissionEffect = PermissionEffect.ALLOW
    priority: int = 0  # Higher priority = evaluated first
    created_at: datetime = field(default_factory=datetime.now)
    expires_at: Optional[datetime] = None
    is_active: bool = True


@dataclass
class Subject:
    """Authorization subject (user, service, etc.)"""
    id: str
    type: str  # "user", "service", "group"
    roles: List[str] = field(default_factory=list)  # Role IDs
    direct_permissions: List[str] = field(default_factory=list)  # Direct permission IDs
    attributes: Dict[str, Any] = field(default_factory=dict)
    groups: List[str] = field(default_factory=list)  # Group memberships
    temporary_roles: Dict[str, datetime] = field(default_factory=dict)  # Role -> expiry
    is_active: bool = True


@dataclass
class AuthorizationRequest:
    """Authorization request"""
    subject_id: str
    action: str
    resource: str
    resource_type: ResourceType = ResourceType.API_ENDPOINT
    context: Dict[str, Any] = field(default_factory=dict)
    timestamp: datetime = field(default_factory=datetime.now)


@dataclass
class AuthorizationResult:
    """Authorization decision result"""
    decision: PermissionEffect
    allowed: bool
    reason: str
    matched_permissions: List[str] = field(default_factory=list)
    matched_policies: List[str] = field(default_factory=list)
    evaluation_time_ms: float = 0.0
    warnings: List[str] = field(default_factory=list)


class ConditionEvaluator:
    """Evaluates policy conditions"""
    
    def __init__(self):
        self.operators = {
            'eq': lambda x, y: x == y,
            'ne': lambda x, y: x != y,
            'gt': lambda x, y: x > y,
            'gte': lambda x, y: x >= y,
            'lt': lambda x, y: x < y,
            'lte': lambda x, y: x <= y,
            'in': lambda x, y: x in y,
            'not_in': lambda x, y: x not in y,
            'contains': lambda x, y: y in x,
            'starts_with': lambda x, y: str(x).startswith(str(y)),
            'ends_with': lambda x, y: str(x).endswith(str(y)),
            'regex': lambda x, y: bool(re.match(y, str(x))),
            'glob': lambda x, y: fnmatch.fnmatch(str(x), str(y))
        }
    
    def evaluate(self, conditions: Dict[str, Any], context: Dict[str, Any]) -> bool:
        """Evaluate conditions against context"""
        if not conditions:
            return True
        
        try:
            return self._evaluate_expression(conditions, context)
        except Exception as e:
            logging.error(f"Condition evaluation error: {e}")
            return False
    
    def _evaluate_expression(self, expr: Dict[str, Any], context: Dict[str, Any]) -> bool:
        """Evaluate single expression"""
        if 'and' in expr:
            return all(self._evaluate_expression(sub_expr, context) for sub_expr in expr['and'])
        elif 'or' in expr:
            return any(self._evaluate_expression(sub_expr, context) for sub_expr in expr['or'])
        elif 'not' in expr:
            return not self._evaluate_expression(expr['not'], context)
        else:
            # Simple condition
            field = expr.get('field')
            operator = expr.get('operator', 'eq')
            value = expr.get('value')
            
            if field not in context:
                return False
            
            context_value = context[field]
            
            if operator in self.operators:
                return self.operators[operator](context_value, value)
            
            return False


class PolicyEngine:
    """Policy evaluation engine"""
    
    def __init__(self):
        self.condition_evaluator = ConditionEvaluator()
        self.logger = logging.getLogger(__name__)
    
    def evaluate_policies(self, policies: List[Policy], request: AuthorizationRequest) -> AuthorizationResult:
        """Evaluate policies against request"""
        start_time = datetime.now()
        matched_policies = []
        decision = PermissionEffect.DENY
        reasons = []
        warnings = []
        
        # Sort policies by priority (highest first)
        sorted_policies = sorted(
            [p for p in policies if p.is_active and self._policy_applies(p, request)],
            key=lambda p: p.priority,
            reverse=True
        )
        
        for policy in sorted_policies:
            if self._evaluate_policy(policy, request):
                matched_policies.append(policy.id)
                
                if policy.effect == PermissionEffect.ALLOW:
                    decision = PermissionEffect.ALLOW
                    reasons.append(f"Policy {policy.name} allows access")
                    break  # First allow wins
                elif policy.effect == PermissionEffect.DENY:
                    decision = PermissionEffect.DENY
                    reasons.append(f"Policy {policy.name} denies access")
                    # Continue to check for higher priority allows
        
        if not matched_policies:
            reasons.append("No matching policies found - default deny")
        
        evaluation_time = (datetime.now() - start_time).total_seconds() * 1000
        
        return AuthorizationResult(
            decision=decision,
            allowed=(decision == PermissionEffect.ALLOW),
            reason="; ".join(reasons),
            matched_policies=matched_policies,
            evaluation_time_ms=evaluation_time,
            warnings=warnings
        )
    
    def _policy_applies(self, policy: Policy, request: AuthorizationRequest) -> bool:
        """Check if policy applies to request"""
        if policy.expires_at and datetime.now() > policy.expires_at:
            return False
        
        # Check if any policy statement matches
        for statement in policy.statements:
            if self._statement_matches(statement, request):
                return True
        
        return False
    
    def _statement_matches(self, statement: Dict[str, Any], request: AuthorizationRequest) -> bool:
        """Check if policy statement matches request"""
        # Check actions
        actions = statement.get('actions', [])
        if actions and request.action not in actions:
            # Check for wildcard patterns
            if not any(fnmatch.fnmatch(request.action, pattern) for pattern in actions):
                return False
        
        # Check resources
        resources = statement.get('resources', [])
        if resources and request.resource not in resources:
            # Check for wildcard patterns
            if not any(fnmatch.fnmatch(request.resource, pattern) for pattern in resources):
                return False
        
        return True
    
    def _evaluate_policy(self, policy: Policy, request: AuthorizationRequest) -> bool:
        """Evaluate policy conditions"""
        # Build evaluation context
        context = {
            **request.context,
            'subject_id': request.subject_id,
            'action': request.action,
            'resource': request.resource,
            'resource_type': request.resource_type.value,
            'timestamp': request.timestamp.isoformat(),
            'time_of_day': request.timestamp.hour,
            'day_of_week': request.timestamp.weekday(),
        }
        
        return self.condition_evaluator.evaluate(policy.conditions, context)


class AuthorizationManager:
    """Main authorization manager implementing RBAC"""
    
    def __init__(self):
        self.permissions: Dict[str, Permission] = {}
        self.roles: Dict[str, Role] = {}
        self.policies: Dict[str, Policy] = {}
        self.subjects: Dict[str, Subject] = {}
        self.policy_engine = PolicyEngine()
        self.logger = logging.getLogger(__name__)
        
        # Initialize system roles and permissions
        self._initialize_system_roles()
        
        # Cache for performance
        self._role_permission_cache: Dict[str, Set[str]] = {}
        self._cache_expiry: Dict[str, datetime] = {}
        self.cache_ttl_minutes = 15
    
    def _initialize_system_roles(self):
        """Initialize system roles"""
        # Super Admin role
        super_admin_role = Role(
            name="super_admin",
            description="Super administrator with all permissions",
            is_system_role=True
        )
        self.roles[super_admin_role.id] = super_admin_role
        
        # Admin role
        admin_role = Role(
            name="admin",
            description="Administrator with most permissions",
            is_system_role=True
        )
        self.roles[admin_role.id] = admin_role
        
        # User role
        user_role = Role(
            name="user",
            description="Basic user role",
            is_system_role=True
        )
        self.roles[user_role.id] = user_role
        
        # Create basic permissions
        read_permission = Permission(
            name="read_all",
            description="Read access to all resources",
            resource_pattern="*",
            actions=["read", "list", "get"]
        )
        self.permissions[read_permission.id] = read_permission
        
        write_permission = Permission(
            name="write_all", 
            description="Write access to all resources",
            resource_pattern="*",
            actions=["write", "create", "update", "delete"]
        )
        self.permissions[write_permission.id] = write_permission
        
        # Assign permissions to roles
        super_admin_role.permissions = [read_permission.id, write_permission.id]
        admin_role.permissions = [read_permission.id]
        user_role.permissions = [read_permission.id]
    
    def create_permission(self, name: str, description: str, resource_type: ResourceType,
                         resource_pattern: str, actions: List[str], 
                         effect: PermissionEffect = PermissionEffect.ALLOW,
                         conditions: Optional[Dict[str, Any]] = None,
                         created_by: str = "system") -> Permission:
        """Create new permission"""
        permission = Permission(
            name=name,
            description=description,
            resource_type=resource_type,
            resource_pattern=resource_pattern,
            actions=actions,
            effect=effect,
            conditions=conditions or {},
            created_by=created_by
        )
        
        self.permissions[permission.id] = permission
        self._invalidate_cache()
        
        self.logger.info(f"Created permission: {name} ({permission.id})")
        return permission
    
    def create_role(self, name: str, description: str, permission_names: List[str] = None,
                   parent_role_names: List[str] = None, created_by: str = "system") -> Role:
        """Create new role"""
        # Resolve permission IDs
        permission_ids = []
        if permission_names:
            for perm_name in permission_names:
                perm = self.find_permission_by_name(perm_name)
                if perm:
                    permission_ids.append(perm.id)
        
        # Resolve parent role IDs
        parent_role_ids = []
        if parent_role_names:
            for role_name in parent_role_names:
                role = self.find_role_by_name(role_name)
                if role:
                    parent_role_ids.append(role.id)
        
        role = Role(
            name=name,
            description=description,
            permissions=permission_ids,
            parent_roles=parent_role_ids,
            created_by=created_by
        )
        
        self.roles[role.id] = role
        self._invalidate_cache()
        
        self.logger.info(f"Created role: {name} ({role.id})")
        return role
    
    def create_policy(self, name: str, description: str, statements: List[Dict[str, Any]],
                     effect: PermissionEffect = PermissionEffect.ALLOW,
                     conditions: Optional[Dict[str, Any]] = None,
                     priority: int = 0, expires_at: Optional[datetime] = None) -> Policy:
        """Create access control policy"""
        policy = Policy(
            name=name,
            description=description,
            statements=statements,
            effect=effect,
            conditions=conditions or {},
            priority=priority,
            expires_at=expires_at
        )
        
        self.policies[policy.id] = policy
        
        self.logger.info(f"Created policy: {name} ({policy.id})")
        return policy
    
    def create_subject(self, subject_id: str, subject_type: str,
                      role_names: List[str] = None, attributes: Optional[Dict[str, Any]] = None) -> Subject:
        """Create authorization subject"""
        # Resolve role IDs
        role_ids = []
        if role_names:
            for role_name in role_names:
                role = self.find_role_by_name(role_name)
                if role:
                    role_ids.append(role.id)
        
        subject = Subject(
            id=subject_id,
            type=subject_type,
            roles=role_ids,
            attributes=attributes or {}
        )
        
        self.subjects[subject_id] = subject
        self._invalidate_cache()
        
        self.logger.info(f"Created subject: {subject_id} ({subject_type})")
        return subject
    
    def assign_role_to_subject(self, subject_id: str, role_name: str, 
                              expires_at: Optional[datetime] = None) -> bool:
        """Assign role to subject"""
        subject = self.subjects.get(subject_id)
        role = self.find_role_by_name(role_name)
        
        if not subject or not role:
            return False
        
        if expires_at:
            subject.temporary_roles[role.id] = expires_at
        else:
            if role.id not in subject.roles:
                subject.roles.append(role.id)
        
        self._invalidate_cache_for_subject(subject_id)
        self.logger.info(f"Assigned role {role_name} to subject {subject_id}")
        return True
    
    def remove_role_from_subject(self, subject_id: str, role_name: str) -> bool:
        """Remove role from subject"""
        subject = self.subjects.get(subject_id)
        role = self.find_role_by_name(role_name)
        
        if not subject or not role:
            return False
        
        # Remove from permanent roles
        if role.id in subject.roles:
            subject.roles.remove(role.id)
        
        # Remove from temporary roles
        if role.id in subject.temporary_roles:
            del subject.temporary_roles[role.id]
        
        self._invalidate_cache_for_subject(subject_id)
        self.logger.info(f"Removed role {role_name} from subject {subject_id}")
        return True
    
    def check_permission(self, subject_id: str, action: str, resource: str,
                        resource_type: ResourceType = ResourceType.API_ENDPOINT,
                        context: Optional[Dict[str, Any]] = None) -> AuthorizationResult:
        """Check if subject has permission for action on resource"""
        start_time = datetime.now()
        
        subject = self.subjects.get(subject_id)
        if not subject or not subject.is_active:
            return AuthorizationResult(
                decision=PermissionEffect.DENY,
                allowed=False,
                reason=f"Subject {subject_id} not found or inactive"
            )
        
        # Build authorization request
        request = AuthorizationRequest(
            subject_id=subject_id,
            action=action,
            resource=resource,
            resource_type=resource_type,
            context=context or {}
        )
        
        # Add subject attributes to context
        request.context.update(subject.attributes)
        request.context['subject_type'] = subject.type
        request.context['subject_groups'] = subject.groups
        
        # Check policies first (higher precedence)
        relevant_policies = [p for p in self.policies.values() if p.is_active]
        if relevant_policies:
            policy_result = self.policy_engine.evaluate_policies(relevant_policies, request)
            if policy_result.matched_policies:  # If any policies matched
                return policy_result
        
        # Fall back to RBAC evaluation
        rbac_result = self._evaluate_rbac(subject, request)
        
        total_time = (datetime.now() - start_time).total_seconds() * 1000
        rbac_result.evaluation_time_ms = total_time
        
        return rbac_result
    
    def _evaluate_rbac(self, subject: Subject, request: AuthorizationRequest) -> AuthorizationResult:
        """Evaluate RBAC permissions"""
        matched_permissions = []
        decision = PermissionEffect.DENY
        reasons = []
        
        # Clean up expired temporary roles
        now = datetime.now()
        expired_temp_roles = [
            role_id for role_id, expiry in subject.temporary_roles.items()
            if now > expiry
        ]
        for role_id in expired_temp_roles:
            del subject.temporary_roles[role_id]
        
        # Get all effective permissions for subject
        all_permissions = self._get_subject_permissions(subject.id)
        
        # Check permissions
        for permission_id in all_permissions:
            permission = self.permissions.get(permission_id)
            if not permission or not permission.is_active:
                continue
            
            if self._permission_matches(permission, request):
                matched_permissions.append(permission_id)
                
                if permission.effect == PermissionEffect.ALLOW:
                    decision = PermissionEffect.ALLOW
                    reasons.append(f"Permission {permission.name} allows access")
                    break  # First allow wins
                elif permission.effect == PermissionEffect.DENY:
                    decision = PermissionEffect.DENY
                    reasons.append(f"Permission {permission.name} denies access")
        
        if not matched_permissions:
            reasons.append("No matching permissions found - default deny")
        
        return AuthorizationResult(
            decision=decision,
            allowed=(decision == PermissionEffect.ALLOW),
            reason="; ".join(reasons),
            matched_permissions=matched_permissions
        )
    
    def _permission_matches(self, permission: Permission, request: AuthorizationRequest) -> bool:
        """Check if permission matches request"""
        # Check resource type
        if permission.resource_type != request.resource_type:
            return False
        
        # Check resource pattern
        if not fnmatch.fnmatch(request.resource, permission.resource_pattern):
            return False
        
        # Check actions
        if permission.actions and request.action not in permission.actions:
            # Check for wildcard patterns
            if not any(fnmatch.fnmatch(request.action, pattern) for pattern in permission.actions):
                return False
        
        # Check conditions
        if permission.conditions:
            condition_evaluator = ConditionEvaluator()
            if not condition_evaluator.evaluate(permission.conditions, request.context):
                return False
        
        return True
    
    def _get_subject_permissions(self, subject_id: str) -> Set[str]:
        """Get all permissions for subject (with caching)"""
        # Check cache
        cache_key = f"permissions:{subject_id}"
        if (cache_key in self._role_permission_cache and 
            cache_key in self._cache_expiry and
            datetime.now() < self._cache_expiry[cache_key]):
            return self._role_permission_cache[cache_key]
        
        subject = self.subjects.get(subject_id)
        if not subject:
            return set()
        
        permissions = set()
        
        # Direct permissions
        permissions.update(subject.direct_permissions)
        
        # Role permissions (including temporary roles)
        all_role_ids = set(subject.roles)
        all_role_ids.update(subject.temporary_roles.keys())
        
        for role_id in all_role_ids:
            role_permissions = self._get_role_permissions(role_id)
            permissions.update(role_permissions)
        
        # Cache result
        self._role_permission_cache[cache_key] = permissions
        self._cache_expiry[cache_key] = datetime.now() + timedelta(minutes=self.cache_ttl_minutes)
        
        return permissions
    
    def _get_role_permissions(self, role_id: str) -> Set[str]:
        """Get all permissions for role (including inherited)"""
        visited = set()
        permissions = set()
        
        def collect_permissions(current_role_id: str):
            if current_role_id in visited:
                return  # Avoid cycles
            
            visited.add(current_role_id)
            role = self.roles.get(current_role_id)
            
            if role and role.is_active:
                permissions.update(role.permissions)
                
                # Recursive collect from parent roles
                for parent_role_id in role.parent_roles:
                    collect_permissions(parent_role_id)
        
        collect_permissions(role_id)
        return permissions
    
    def find_permission_by_name(self, name: str) -> Optional[Permission]:
        """Find permission by name"""
        for permission in self.permissions.values():
            if permission.name == name:
                return permission
        return None
    
    def find_role_by_name(self, name: str) -> Optional[Role]:
        """Find role by name"""
        for role in self.roles.values():
            if role.name == name:
                return role
        return None
    
    def _invalidate_cache(self):
        """Invalidate all caches"""
        self._role_permission_cache.clear()
        self._cache_expiry.clear()
    
    def _invalidate_cache_for_subject(self, subject_id: str):
        """Invalidate cache for specific subject"""
        cache_key = f"permissions:{subject_id}"
        if cache_key in self._role_permission_cache:
            del self._role_permission_cache[cache_key]
        if cache_key in self._cache_expiry:
            del self._cache_expiry[cache_key]
    
    def get_subject_roles(self, subject_id: str) -> List[Role]:
        """Get all roles for subject"""
        subject = self.subjects.get(subject_id)
        if not subject:
            return []
        
        roles = []
        all_role_ids = set(subject.roles)
        all_role_ids.update(subject.temporary_roles.keys())
        
        for role_id in all_role_ids:
            role = self.roles.get(role_id)
            if role:
                roles.append(role)
        
        return roles
    
    def get_subject_permissions_detailed(self, subject_id: str) -> Dict[str, Any]:
        """Get detailed permission information for subject"""
        subject = self.subjects.get(subject_id)
        if not subject:
            return {}
        
        result = {
            'subject_id': subject_id,
            'direct_permissions': [],
            'role_permissions': {},
            'total_permissions': 0
        }
        
        # Direct permissions
        for perm_id in subject.direct_permissions:
            permission = self.permissions.get(perm_id)
            if permission:
                result['direct_permissions'].append({
                    'id': permission.id,
                    'name': permission.name,
                    'resource_pattern': permission.resource_pattern,
                    'actions': permission.actions,
                    'effect': permission.effect.value
                })
        
        # Role permissions
        for role_id in subject.roles:
            role = self.roles.get(role_id)
            if role:
                role_perms = []
                role_permission_ids = self._get_role_permissions(role_id)
                
                for perm_id in role_permission_ids:
                    permission = self.permissions.get(perm_id)
                    if permission:
                        role_perms.append({
                            'id': permission.id,
                            'name': permission.name,
                            'resource_pattern': permission.resource_pattern,
                            'actions': permission.actions,
                            'effect': permission.effect.value
                        })
                
                result['role_permissions'][role.name] = {
                    'role_id': role.id,
                    'permissions': role_perms
                }
        
        # Total unique permissions
        all_permissions = self._get_subject_permissions(subject_id)
        result['total_permissions'] = len(all_permissions)
        
        return result


if __name__ == "__main__":
    # Example usage and testing
    logging.basicConfig(level=logging.INFO)
    
    # Create authorization manager
    auth_mgr = AuthorizationManager()
    
    # Create custom permissions
    api_read_perm = auth_mgr.create_permission(
        name="api_read",
        description="Read API endpoints",
        resource_type=ResourceType.API_ENDPOINT,
        resource_pattern="/api/v1/*",
        actions=["GET"]
    )
    
    user_write_perm = auth_mgr.create_permission(
        name="user_write", 
        description="Write user data",
        resource_type=ResourceType.DATABASE_TABLE,
        resource_pattern="users",
        actions=["INSERT", "UPDATE", "DELETE"]
    )
    
    # Create custom role
    api_user_role = auth_mgr.create_role(
        name="api_user",
        description="API user with limited access",
        permission_names=["api_read"],
        parent_role_names=["user"]
    )
    
    # Create subject
    user_subject = auth_mgr.create_subject(
        subject_id="user123",
        subject_type="user",
        role_names=["api_user"],
        attributes={"department": "engineering", "level": "senior"}
    )
    
    # Test authorization
    result = auth_mgr.check_permission(
        subject_id="user123",
        action="GET",
        resource="/api/v1/users",
        resource_type=ResourceType.API_ENDPOINT
    )
    
    print(f"Authorization result: {result.allowed} - {result.reason}")
    
    # Test with policy
    time_policy = auth_mgr.create_policy(
        name="business_hours_only",
        description="Only allow access during business hours",
        statements=[{
            "actions": ["*"],
            "resources": ["/api/v1/*"]
        }],
        conditions={
            "and": [
                {"field": "time_of_day", "operator": "gte", "value": 9},
                {"field": "time_of_day", "operator": "lte", "value": 17}
            ]
        },
        priority=100
    )
    
    # Test with current time
    result_with_policy = auth_mgr.check_permission(
        subject_id="user123",
        action="GET", 
        resource="/api/v1/data",
        context={"custom_field": "test_value"}
    )
    
    print(f"Policy result: {result_with_policy.allowed} - {result_with_policy.reason}")
    
    # Get detailed permissions
    perms_detail = auth_mgr.get_subject_permissions_detailed("user123")
    print(f"User has {perms_detail['total_permissions']} total permissions")
    
    print("\ng10.2: Role-Based Authorization Manager - COMPLETED ✅") 