#!/usr/bin/env python3
"""
Dependency Management Commands for TuskLang CLI
==============================================
Install optional dependencies for AI, data processing, and analytics features
"""

import subprocess
import sys
from typing import Dict, List, Optional

class DependencyManager:
    """Manages optional dependencies for TuskLang SDK"""
    
    DEPENDENCY_GROUPS = {
        'ai': [
            'torch',
            'transformers', 
            'tensorflow',
            'scikit-learn',
            'nltk',
            'spacy',
            'openai',
            'anthropic',
            'langchain',
            'sentence-transformers',
            'xgboost',
            'lightgbm',
            'nvidia-nccl-cu12'
        ],
        'data': [
            'pandas',
            'numpy',
            'scipy',
            'matplotlib',
            'seaborn',
            'plotly',
            'dask',
            'xarray',
            'polars',
            'openpyxl',
            'pyarrow',
            'vaex'
        ],
        'analytics': [
            'scikit-learn',
            'statsmodels',
            'prophet',
            'pmdarima',
            'pycaret',
            'optuna',
            'mlflow',
            'wandb'
        ],
        'database': [
            'pymongo',
            'motor',
            'asyncpg',
            'psycopg2-binary',
            'aiomysql',
            'pymysql',
            'redis',
            'aioredis'
        ],
        'communication': [
            'aiohttp',
            'websockets',
            'requests',
            'grpcio',
            'grpcio-tools',
            'grpcio-status'
        ],
        'cloud': [
            'boto3',
            'botocore',
            'jmespath',
            's3transfer',
            'google-cloud-container',
            'google-api-core',
            'proto-plus',
            'googleapis-common-protos',
            'azure-mgmt-containerservice',
            'azure-common',
            'azure-mgmt-core',
            'azure-core',
            'isodate'
        ],
        'security': [
            'cryptography',
            'bcrypt',
            'PyJWT',
            'argon2-cffi'
        ],
        'monitoring': [
            'psutil',
            'prometheus-client'
        ],
        'platform': [
            'docker',
            'kubernetes',
            'kafka-python',
            'avro-python3'
        ],
        'full': [
            # AI/ML
            'torch', 'transformers', 'tensorflow', 'scikit-learn', 'nltk', 'spacy',
            'openai', 'anthropic', 'langchain', 'sentence-transformers', 'xgboost', 'lightgbm',
            # Data Processing
            'pandas', 'numpy', 'scipy', 'matplotlib', 'seaborn', 'plotly',
            'dask', 'xarray', 'polars', 'openpyxl', 'pyarrow',
            # Analytics
            'statsmodels', 'prophet', 'pmdarima', 'pycaret', 'optuna', 'mlflow', 'wandb',
            # Database
            'pymongo', 'motor', 'asyncpg', 'psycopg2-binary', 'aiomysql', 'pymysql', 'redis',
            # Communication
            'aiohttp', 'websockets', 'requests', 'grpcio', 'grpcio-tools',
            # Cloud
            'boto3', 'google-cloud-container', 'azure-mgmt-containerservice',
            # Security
            'cryptography', 'bcrypt', 'PyJWT', 'argon2-cffi',
            # Monitoring
            'psutil', 'prometheus-client',
            # Platform
            'docker', 'kubernetes', 'kafka-python', 'avro-python3'
        ],
        'minimal': [
            # Core essentials for basic functionality
            'pandas',
            'numpy',
            'scikit-learn',
            'aiohttp',
            'redis',
            'cryptography',
            'psutil'
        ]
    }
    
    @classmethod
    def install_dependencies(cls, group: str, verbose: bool = False) -> bool:
        """Install dependencies for a specific group"""
        if group not in cls.DEPENDENCY_GROUPS:
            print(f"❌ Unknown dependency group: {group}")
            print(f"Available groups: {', '.join(cls.DEPENDENCY_GROUPS.keys())}")
            return False
        
        packages = cls.DEPENDENCY_GROUPS[group]
        print(f"📦 Installing {len(packages)} packages for '{group}' group...")
        
        if verbose:
            print(f"Packages: {', '.join(packages)}")
        
        try:
            # Install packages
            cmd = [sys.executable, '-m', 'pip', 'install'] + packages
            if verbose:
                print(f"Running: {' '.join(cmd)}")
            
            result = subprocess.run(cmd, capture_output=not verbose, text=True)
            
            if result.returncode == 0:
                print(f"✅ Successfully installed {len(packages)} packages for '{group}' group")
                return True
            else:
                print(f"❌ Failed to install packages: {result.stderr}")
                return False
                
        except Exception as e:
            print(f"❌ Error installing dependencies: {e}")
            return False
    
    @classmethod
    def list_groups(cls) -> None:
        """List all available dependency groups"""
        print("📋 Available dependency groups:")
        for group, packages in cls.DEPENDENCY_GROUPS.items():
            print(f"  {group}: {len(packages)} packages")
            print(f"    {', '.join(packages[:5])}{'...' if len(packages) > 5 else ''}")
    
    @classmethod
    def check_installed(cls, group: str) -> Dict[str, bool]:
        """Check which packages from a group are installed"""
        if group not in cls.DEPENDENCY_GROUPS:
            return {}
        
        packages = cls.DEPENDENCY_GROUPS[group]
        installed = {}
        
        for package in packages:
            try:
                __import__(package.replace('-', '_'))
                installed[package] = True
            except ImportError:
                installed[package] = False
        
        return installed

def install_deps(group: str, verbose: bool = False) -> int:
    """CLI command to install dependencies"""
    if group == 'list':
        DependencyManager.list_groups()
        return 0
    
    if group == 'check':
        print("🔍 Checking installed packages...")
        for group_name in DependencyManager.DEPENDENCY_GROUPS.keys():
            installed = DependencyManager.check_installed(group_name)
            installed_count = sum(installed.values())
            total_count = len(installed)
            print(f"  {group_name}: {installed_count}/{total_count} packages installed")
        return 0
    
    success = DependencyManager.install_dependencies(group, verbose)
    return 0 if success else 1

def handle_dependency_command(args, cli):
    """Handle dependency management commands"""
    if args.deps_command == 'install':
        return install_deps(args.group, args.verbose)
    elif args.deps_command == 'list':
        return install_deps('list')
    elif args.deps_command == 'check':
        return install_deps('check')
    else:
        print("❌ Unknown dependency command. Use 'tsk deps --help' for options.")
        return 1

def main():
    """Main entry point for dependency commands"""
    import argparse
    
    parser = argparse.ArgumentParser(description="TuskLang Dependency Manager")
    parser.add_argument('group', help='Dependency group to install (ai, data, analytics, database, communication, cloud, security, monitoring, platform, full, minimal, list, check)')
    parser.add_argument('--verbose', '-v', action='store_true', help='Verbose output')
    
    args = parser.parse_args()
    
    return install_deps(args.group, args.verbose)

if __name__ == '__main__':
    sys.exit(main()) 