"""
TuskLang Python SDK - Advanced Database Connector (g8.1)
Multi-engine database support with connection pooling and advanced features
"""

import asyncio
import json
import logging
from abc import ABC, abstractmethod
from contextlib import contextmanager
from dataclasses import dataclass
from typing import Dict, List, Optional, Any, Union, Tuple
from urllib.parse import urlparse
import threading
import time
from queue import Queue, Empty

try:
    import psycopg2
    from psycopg2 import pool as pg_pool
    import psycopg2.extras
    POSTGRES_AVAILABLE = True
except ImportError:
    POSTGRES_AVAILABLE = False

try:
    import pymongo
    from pymongo import MongoClient
    MONGO_AVAILABLE = True
except ImportError:
    MONGO_AVAILABLE = False

try:
    import mysql.connector
    from mysql.connector import pooling
    MYSQL_AVAILABLE = True
except ImportError:
    MYSQL_AVAILABLE = False

import sqlite3

@dataclass
class ConnectionConfig:
    """Database connection configuration"""
    engine: str
    host: str = "localhost"
    port: int = 5432
    database: str = ""
    username: str = ""
    password: str = ""
    pool_size: int = 10
    max_overflow: int = 20
    pool_timeout: int = 30
    ssl_mode: str = "disable"
    charset: str = "utf8mb4"
    
    def to_dict(self) -> Dict[str, Any]:
        return {
            'engine': self.engine,
            'host': self.host,
            'port': self.port,
            'database': self.database,
            'username': self.username,
            'password': self.password,
            'pool_size': self.pool_size,
            'max_overflow': self.max_overflow,
            'pool_timeout': self.pool_timeout,
            'ssl_mode': self.ssl_mode,
            'charset': self.charset
        }

class DatabaseAdapter(ABC):
    """Abstract database adapter"""
    
    @abstractmethod
    def connect(self) -> Any:
        pass
    
    @abstractmethod
    def execute_query(self, query: str, params: Optional[tuple] = None) -> List[Dict]:
        pass
    
    @abstractmethod
    def execute_command(self, command: str, params: Optional[tuple] = None) -> int:
        pass
    
    @abstractmethod
    def close(self):
        pass

class PostgreSQLAdapter(DatabaseAdapter):
    """PostgreSQL database adapter with connection pooling"""
    
    def __init__(self, config: ConnectionConfig):
        self.config = config
        self.pool = None
        if POSTGRES_AVAILABLE:
            self.pool = pg_pool.ThreadedConnectionPool(
                1, config.pool_size,
                host=config.host,
                port=config.port,
                database=config.database,
                user=config.username,
                password=config.password,
                sslmode=config.ssl_mode
            )
        
    @contextmanager
    def get_connection(self):
        """Get connection from pool"""
        if not self.pool:
            raise RuntimeError("PostgreSQL not available")
            
        conn = self.pool.getconn()
        try:
            yield conn
        finally:
            self.pool.putconn(conn)
    
    def connect(self) -> Any:
        return self.get_connection()
    
    def execute_query(self, query: str, params: Optional[tuple] = None) -> List[Dict]:
        """Execute SELECT query"""
        with self.get_connection() as conn:
            with conn.cursor(cursor_factory=psycopg2.extras.DictCursor) as cursor:
                cursor.execute(query, params)
                return [dict(row) for row in cursor.fetchall()]
    
    def execute_command(self, command: str, params: Optional[tuple] = None) -> int:
        """Execute INSERT/UPDATE/DELETE command"""
        with self.get_connection() as conn:
            with conn.cursor() as cursor:
                cursor.execute(command, params)
                conn.commit()
                return cursor.rowcount
                
    def close(self):
        if self.pool:
            self.pool.closeall()

class MySQLAdapter(DatabaseAdapter):
    """MySQL database adapter with connection pooling"""
    
    def __init__(self, config: ConnectionConfig):
        self.config = config
        self.pool = None
        if MYSQL_AVAILABLE:
            pool_config = {
                'pool_name': 'mysql_pool',
                'pool_size': config.pool_size,
                'pool_reset_session': True,
                'host': config.host,
                'port': config.port,
                'database': config.database,
                'user': config.username,
                'password': config.password,
                'charset': config.charset
            }
            self.pool = mysql.connector.pooling.MySQLConnectionPool(**pool_config)
    
    def connect(self) -> Any:
        if not self.pool:
            raise RuntimeError("MySQL not available")
        return self.pool.get_connection()
    
    def execute_query(self, query: str, params: Optional[tuple] = None) -> List[Dict]:
        """Execute SELECT query"""
        conn = self.connect()
        try:
            cursor = conn.cursor(dictionary=True)
            cursor.execute(query, params)
            return cursor.fetchall()
        finally:
            conn.close()
    
    def execute_command(self, command: str, params: Optional[tuple] = None) -> int:
        """Execute INSERT/UPDATE/DELETE command"""
        conn = self.connect()
        try:
            cursor = conn.cursor()
            cursor.execute(command, params)
            conn.commit()
            return cursor.rowcount
        finally:
            conn.close()
            
    def close(self):
        pass  # Pool cleanup handled by MySQL connector

class SQLiteAdapter(DatabaseAdapter):
    """SQLite database adapter"""
    
    def __init__(self, config: ConnectionConfig):
        self.config = config
        self.connection_pool = Queue(maxsize=config.pool_size)
        self.lock = threading.Lock()
        
        # Pre-populate connection pool
        for _ in range(config.pool_size):
            conn = sqlite3.connect(config.database, check_same_thread=False)
            conn.row_factory = sqlite3.Row
            self.connection_pool.put(conn)
    
    @contextmanager
    def get_connection(self):
        """Get connection from pool"""
        try:
            conn = self.connection_pool.get(timeout=self.config.pool_timeout)
            yield conn
        finally:
            self.connection_pool.put(conn)
    
    def connect(self) -> Any:
        return self.get_connection()
    
    def execute_query(self, query: str, params: Optional[tuple] = None) -> List[Dict]:
        """Execute SELECT query"""
        with self.get_connection() as conn:
            cursor = conn.execute(query, params or ())
            return [dict(row) for row in cursor.fetchall()]
    
    def execute_command(self, command: str, params: Optional[tuple] = None) -> int:
        """Execute INSERT/UPDATE/DELETE command"""
        with self.get_connection() as conn:
            cursor = conn.execute(command, params or ())
            conn.commit()
            return cursor.rowcount
            
    def close(self):
        while not self.connection_pool.empty():
            try:
                conn = self.connection_pool.get_nowait()
                conn.close()
            except Empty:
                break

class MongoDBAdapter(DatabaseAdapter):
    """MongoDB database adapter"""
    
    def __init__(self, config: ConnectionConfig):
        self.config = config
        if MONGO_AVAILABLE:
            connection_string = f"mongodb://{config.username}:{config.password}@{config.host}:{config.port}/"
            self.client = MongoClient(
                connection_string,
                maxPoolSize=config.pool_size,
                serverSelectionTimeoutMS=config.pool_timeout * 1000
            )
            self.db = self.client[config.database]
        else:
            self.client = None
            self.db = None
    
    def connect(self) -> Any:
        return self.db
    
    def execute_query(self, query: str, params: Optional[tuple] = None) -> List[Dict]:
        """Execute find query (MongoDB specific)"""
        if not self.db:
            raise RuntimeError("MongoDB not available")
            
        # Parse MongoDB query format
        query_dict = json.loads(query) if isinstance(query, str) else query
        collection_name = query_dict.get('collection', 'default')
        find_query = query_dict.get('query', {})
        projection = query_dict.get('projection', None)
        
        collection = self.db[collection_name]
        results = collection.find(find_query, projection)
        return [doc for doc in results]
    
    def execute_command(self, command: str, params: Optional[tuple] = None) -> int:
        """Execute insert/update/delete command"""
        if not self.db:
            raise RuntimeError("MongoDB not available")
            
        command_dict = json.loads(command) if isinstance(command, str) else command
        collection_name = command_dict.get('collection', 'default')
        operation = command_dict.get('operation', 'insert')
        data = command_dict.get('data', {})
        
        collection = self.db[collection_name]
        
        if operation == 'insert':
            result = collection.insert_one(data)
            return 1 if result.inserted_id else 0
        elif operation == 'update':
            filter_query = command_dict.get('filter', {})
            result = collection.update_many(filter_query, {'$set': data})
            return result.modified_count
        elif operation == 'delete':
            filter_query = command_dict.get('filter', {})
            result = collection.delete_many(filter_query)
            return result.deleted_count
        
        return 0
            
    def close(self):
        if self.client:
            self.client.close()

class DatabaseManager:
    """Unified database manager supporting multiple engines"""
    
    def __init__(self):
        self.connections: Dict[str, DatabaseAdapter] = {}
        self.logger = logging.getLogger(__name__)
        
    def add_connection(self, name: str, config: ConnectionConfig) -> DatabaseAdapter:
        """Add a new database connection"""
        adapter = self._create_adapter(config)
        self.connections[name] = adapter
        self.logger.info(f"Added {config.engine} connection: {name}")
        return adapter
    
    def _create_adapter(self, config: ConnectionConfig) -> DatabaseAdapter:
        """Create appropriate database adapter"""
        if config.engine.lower() == 'postgresql':
            return PostgreSQLAdapter(config)
        elif config.engine.lower() == 'mysql':
            return MySQLAdapter(config)
        elif config.engine.lower() == 'sqlite':
            return SQLiteAdapter(config)
        elif config.engine.lower() == 'mongodb':
            return MongoDBAdapter(config)
        else:
            raise ValueError(f"Unsupported database engine: {config.engine}")
    
    def get_connection(self, name: str) -> Optional[DatabaseAdapter]:
        """Get database connection by name"""
        return self.connections.get(name)
    
    def execute_query(self, connection_name: str, query: str, params: Optional[tuple] = None) -> List[Dict]:
        """Execute query on specified connection"""
        adapter = self.get_connection(connection_name)
        if not adapter:
            raise ValueError(f"Connection not found: {connection_name}")
        return adapter.execute_query(query, params)
    
    def execute_command(self, connection_name: str, command: str, params: Optional[tuple] = None) -> int:
        """Execute command on specified connection"""
        adapter = self.get_connection(connection_name)
        if not adapter:
            raise ValueError(f"Connection not found: {connection_name}")
        return adapter.execute_command(command, params)
    
    def close_all(self):
        """Close all database connections"""
        for name, adapter in self.connections.items():
            try:
                adapter.close()
                self.logger.info(f"Closed connection: {name}")
            except Exception as e:
                self.logger.error(f"Error closing connection {name}: {e}")
        self.connections.clear()
    
    def health_check(self) -> Dict[str, bool]:
        """Check health of all connections"""
        results = {}
        for name, adapter in self.connections.items():
            try:
                if isinstance(adapter, (PostgreSQLAdapter, MySQLAdapter)):
                    adapter.execute_query("SELECT 1")
                elif isinstance(adapter, SQLiteAdapter):
                    adapter.execute_query("SELECT 1")
                elif isinstance(adapter, MongoDBAdapter):
                    adapter.connect().list_collection_names()
                results[name] = True
            except Exception as e:
                self.logger.error(f"Health check failed for {name}: {e}")
                results[name] = False
        return results

# Global database manager instance
db_manager = DatabaseManager()

def create_connection(name: str, engine: str, **kwargs) -> DatabaseAdapter:
    """Convenience function to create database connection"""
    config = ConnectionConfig(engine=engine, **kwargs)
    return db_manager.add_connection(name, config)

def query(connection_name: str, sql: str, params: Optional[tuple] = None) -> List[Dict]:
    """Convenience function to execute query"""
    return db_manager.execute_query(connection_name, sql, params)

def execute(connection_name: str, sql: str, params: Optional[tuple] = None) -> int:
    """Convenience function to execute command"""
    return db_manager.execute_command(connection_name, sql, params)

# Advanced database features
class DatabaseMetrics:
    """Database performance metrics"""
    
    def __init__(self):
        self.query_count = 0
        self.execution_times = []
        self.errors = 0
        self.lock = threading.Lock()
    
    def record_query(self, execution_time: float, success: bool = True):
        """Record query execution metrics"""
        with self.lock:
            self.query_count += 1
            self.execution_times.append(execution_time)
            if not success:
                self.errors += 1
    
    def get_stats(self) -> Dict[str, Any]:
        """Get performance statistics"""
        with self.lock:
            if not self.execution_times:
                return {
                    'query_count': 0,
                    'avg_execution_time': 0,
                    'max_execution_time': 0,
                    'min_execution_time': 0,
                    'error_rate': 0
                }
            
            return {
                'query_count': self.query_count,
                'avg_execution_time': sum(self.execution_times) / len(self.execution_times),
                'max_execution_time': max(self.execution_times),
                'min_execution_time': min(self.execution_times),
                'error_rate': self.errors / self.query_count if self.query_count > 0 else 0
            }

# Global metrics instance
metrics = DatabaseMetrics()

if __name__ == "__main__":
    # Example usage
    logging.basicConfig(level=logging.INFO)
    
    # SQLite example
    sqlite_adapter = create_connection(
        "local_db", 
        "sqlite", 
        database="test.db",
        pool_size=5
    )
    
    # Create test table
    execute("local_db", """
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY,
            name TEXT,
            email TEXT
        )
    """)
    
    # Insert data
    execute("local_db", "INSERT INTO users (name, email) VALUES (?, ?)", ("John Doe", "john@example.com"))
    
    # Query data
    results = query("local_db", "SELECT * FROM users")
    print(f"Query results: {results}")
    
    # Health check
    health = db_manager.health_check()
    print(f"Health check: {health}")
    
    # Get metrics
    stats = metrics.get_stats()
    print(f"Performance metrics: {stats}")
    
    # Cleanup
    db_manager.close_all()
    
    print("g8.1: Advanced Database Connector - COMPLETED ✅") 